import json
import time

import requests

from utils.common import log as logger
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.ssh_util2 import Ssh
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2

from plugins.eBackup.scripts.common.ebackup_util import CommonTools
from plugins.eBackup.scripts.common.ebackup_util import CustomParametersUtil
from plugins.eBackup.common import constant
from plugins.eBackup.common import model
from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.common.ebackup_rest import EbackupRest

ADMIN_NODE = "0"
PRIMARY_NODE = "2"
STANDBY_NODE = "3"


class EBackupUtil(object):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = constant.EBACKUP_SERVICE_NAME
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ssh_client = Ssh()
        self.rest_api = EbackupRest()
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.manageone = ManageOneUtil2()
        self.custom_pararms = CustomParametersUtil(project_id, pod_id)

    @staticmethod
    def netmask_to_bit_length(netmask):
        if "." in netmask:
            return str(sum([bin(int(i)).count('1') for i in netmask.split('.')]))
        else:
            return netmask

    def get_internal_float_ip(self):
        float_ip = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, 'datamover_internal_float_ip')
        if not float_ip:
            raise Exception("Failed to obtain the datamover_internal_float_ip.")
        return float_ip

    def get_management_float_ip(self):
        if self.param_tool.is_expand_ebackup_server():
            datamover_ip_list = self._get_param_by_param_key('datamover_externalom_iplist').split(',')
            if len(datamover_ip_list) == 1:
                return datamover_ip_list[0]
            return self._get_param_by_param_key('datamover_management_float_ip')
        elif self.is_ex_az_reuse_ebackup():
            datamover_ip_list = self._get_param_by_param_key('existed_datamover_externalom_iplist').split(',')
            if len(datamover_ip_list) == 1:
                return datamover_ip_list[0]
            return self._get_param_by_param_key('existed_datamover_mg_float_ip')
        else:
            datamover_server_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "datamover_server_ip")
            if not datamover_server_ip:
                datamover_server_ip = self._get_param_by_param_key('datamover_externalom_iplist').split(',')[0]
            return self.get_management_float_ip_by_server_ip(datamover_server_ip)

    def get_externalom_iplist(self):
        return self._get_param_by_param_key('datamover_externalom_iplist').split(',')

    def get_node_ip_netmask(self, ips, user, password):
        logger.info(f"Start to get netmask on node {ips}")
        # 只查询第一个的netmask
        for node_ip in ips:
            cmd = '''netmask=$(ip a | grep -w %s | grep inet | awk '{print $2}' | awk -F'/' '{print $2}');
            netcard_name=$(ip a | grep -w %s | grep inet | awk '{print $NF}' );
            gateway=$(route -n | awk '{if ($1=="0.0.0.0"&&$8=="'${netcard_name}'")print $2}');
            echo "ebackup_netmask: ${gateway}/${netmask}"''' % (node_ip, node_ip)
            ssh_client = self.ssh_client.ssh_create_client(node_ip, user, password)
            result = self.ssh_client.ssh_exec_command_return_list(ssh_client, cmd)
            for lines in result:
                if "ebackup_netmask: " in lines and "echo" not in lines:
                    logger.info(f"Get netmask {lines} on node {node_ip}")
                    return lines.split("ebackup_netmask:")[1].strip()
            return ""

    def get_active_stanby_node(self, datamover_ip):
        base_url = self.get_server_url(datamover_ip)
        portal_account = self.get_portal_account(datamover_ip)
        _, headers = self.login_portal(base_url, portal_account)
        rsp = requests.get(url=f"{base_url}/vbackup_server", headers=headers, verify=False)
        rsp_dict = rsp.json()
        error_code = rsp_dict.get("error", dict()).get("code")
        if error_code is None:
            logger.error("errcode is empty.")
            raise Exception("Get active standby node failed. errcode is empty.")
        if error_code != 0:
            error_des = rsp_dict.get("error").get("description")
            raise Exception(f"Get active standby node failed. error description is {error_des}.")

        node_list = set()
        server_list = rsp_dict.get("data", [])
        for server_info in server_list:
            role = server_info.get("ROLE")
            if str(role).strip() in (ADMIN_NODE, PRIMARY_NODE, STANDBY_NODE):
                ip_addr = server_info.get("IPADDR", "")
                node_list.add(ip_addr)
        logger.info(f"Successful get node list:{list(node_list)}")
        return list(node_list)

    def get_management_float_ip_by_server_ip(self, server_ip):
        cmd_base = "grep -w listen /opt/huawei-data-protection/ebackup/" \
                   "microservice/ebk_lb/bin/nginx/conf/nginx.conf|grep -w 8088"
        cmd_awk = "|awk '{print $2}'|awk -F '[' '{print $2}'|awk -F ']' '{print $1}'" \
            if self.param_tool.is_ipv6_scene() else "|awk '{print $2}'|awk -F ':' '{print $1}'"
        cmd = cmd_base + cmd_awk
        node_user = self.get_node_user(server_ip)
        try:
            ssh_client = self.ssh_client.ssh_create_client(server_ip, node_user.user_name, node_user.user_pwd)
            self.ssh_client.ssh_send_command(ssh_client, 'su', 'Password:', 100)
            self.ssh_client.ssh_send_command(ssh_client, node_user.root_pwd, '#', 100)
            self.ssh_client.ssh_send_command(ssh_client, "TMOUT=0", '#', 100)
        except Exception:
            ssh_client = self.ssh_client.ssh_create_client(server_ip, node_user.root_name, node_user.root_pwd)
            self.ssh_client.ssh_send_command(ssh_client, "TMOUT=0", '#', 100)
        result = self.ssh_client.ssh_exec_command_return_list(ssh_client, cmd)
        self.ssh_client.ssh_close(ssh_client)
        return result[0]

    def _get_param_by_param_key(self, param_key):
        param_value = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, param_key)
        if not param_value:
            logger.error(f"Failed to obtain the {param_key}.")
        return param_value

    def is_ex_az_reuse_ebackup(self):
        return self.param_tool.is_expand_az() and \
            self.param_tool.is_installed_csbs() and \
            not self.param_tool.is_csha_storage()

    def get_server_url(self, management_float_ip):
        if self.param_tool.is_ipv6_scene():
            return f"https://[{management_float_ip}]:8088/rest/dev"
        return f"https://{management_float_ip}:8088/rest/dev"

    def get_default_node_user(self):
        os_user_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "hcp_ssh_user")
        if not os_user_name:
            os_user_name = "hcp"
        root_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "eBackup_os_root_username")
        if not root_name:
            root_name = "root"
        os_user_pwd = self._get_param_by_param_key("hcp_ssh_password")
        root_pwd = self._get_param_by_param_key("eBackup_os_root_password")
        return model.NodeUser(os_user_name, os_user_pwd, root_name, root_pwd)

    def get_node_user(self, vm_ip):
        node_user = self.get_default_node_user()
        region_id = self.param_tool.get_region_id()
        try:
            account_info_list = self.param_tool.get_account_pwd_from_unify_pwd(
                region_id, [(node_user.user_name, 1), (node_user.root_name, 1)], vm_ip=vm_ip)
            for account_info in account_info_list:
                if account_info.get("ip") != vm_ip:
                    continue
                if account_info.get('accountName') == node_user.user_name:
                    node_user.user_pwd = account_info.get('newPasswd')
                if account_info.get('accountName') == node_user.root_name:
                    node_user.root_pwd = account_info.get('newPasswd')
        except Exception as err:
            logger.warning(f"Failed to obtain account info from unify pwd platform, err_mag:{str(err)}.")
        return node_user

    def get_portal_account(self, management_float_ip=None, is_upgrade=False):
        account_name = "admin"
        account_pwd_key = "eBackup_admin_pwd" if is_upgrade else "eBackup_weblogin_password"
        default_pwd = None
        if self.param_tool.is_expand_ebackup_server():
            account_pwd = self._get_param_by_param_key(account_pwd_key)
            if not is_upgrade:
                default_pwd = self._get_param_by_param_key("eBackup_weblogin_default_password")
        else:
            if not management_float_ip:
                management_float_ip = self.get_management_float_ip()
            region_id = self.param_tool.get_region_id()
            account_info_list = self.param_tool.get_account_pwd_from_unify_pwd(
                region_id, [(account_name, 6)], vm_ip=management_float_ip)
            account_pwd = None
            for account_info in account_info_list:
                if account_info.get("ip") == management_float_ip:
                    account_pwd = account_info.get('newPasswd')
                    break
        if not account_pwd:
            raise Exception(f"Failed to obtain the {account_name}'s passwd.")
        account = model.Account(account_name, account_pwd)
        if default_pwd:
            account.default_pwd = default_pwd
        return account

    def login_portal(self, server_url, portal_account):
        logger.info(f"Start to login ebackup portal, server url:{server_url}.")
        login_url = f"{server_url}/login"
        headers = {'Accept': 'application/json'}
        data = {"scope": 0,
                "username": portal_account.account_name,
                "password": portal_account.account_pwd}
        rsp = self.rest_api.post(login_url, headers, json.dumps(data))
        if rsp.status_code != 200:
            logger.error("Response status_code is not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("Errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            i_base_token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            device_id = rsp.json()["data"].get("deviceid", 'dev')
        else:
            description = rsp.json()["error"]["description"]
            logger.error(f"Login failed, description:{description}.")
            raise HCCIException(653022, description)
        logger.info("Succeed to login ebackup portal.")
        headers = {
            'Accept': 'application/json;version=2.2;charset=UTF-8',
            'iBaseToken': i_base_token,
            'Cookie': f'language=en;{session_content}; DEVICE_ID=dev; sessionIdleTime=60000; '
                      f'MACHINEROLE=0; CSRF_IBASE_TOKEN={i_base_token}'}
        return device_id, headers

    def logout_portal(self, server_url, headers):
        url = f"{server_url}/sessions"
        try:
            self.rest_api.delete(url, headers)
        except Exception as err:
            logger.warning(str(err))
        logger.info("Succeed to logout portal.")

    def add_new_user(self, management_float_ip, user_name, is_upgrade=False):
        base_url = self.get_server_url(management_float_ip)
        portal_account = self.get_portal_account(management_float_ip, is_upgrade)
        device_id, headers = self.login_portal(base_url, portal_account)

        add_user_url = f"https://{management_float_ip}:8088/rest/{device_id}/user"
        if user_name == "EBKUser":
            default_passwd = self._get_param_by_param_key('EBKUser_default_password')
        elif user_name == "NBIUser":
            default_passwd = self._get_param_by_param_key('eBackup_azconfig_apipwd')
        else:
            raise Exception("Unknown ebackup user name.")
        body = {
            "SCOPE": 4,
            "NAME": user_name,
            "PASSWORD": default_passwd,
            "LEVEL": 2,
            "SESSIONMODE": 0,
            "SESSIONNUM": 5
        }
        rsp = self.rest_api.post(add_user_url, headers, json.dumps(body))
        if rsp.status_code != 200:
            logger.error("Response status_code is not 200.")
            self.logout_portal(base_url, headers)
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("Errcode is empty.")
            self.logout_portal(base_url, headers)
            raise HCCIException(653020)
        if errcode == 0:
            logger.info(f"Succeed to add {user_name}.")
            self.logout_portal(base_url, headers)
        elif rsp.json()["error"].get('code') == 1610664452:
            logger.info(f"{user_name} is already exist.")
            self.logout_portal(base_url, headers)
        else:
            description = rsp.json()["error"]["description"]
            logger.error(f"Add EBKUser failed, description:{description}.")
            self.logout_portal(base_url, headers)
            raise HCCIException(650069, description)

    def set_cloud_service_info_of_ebackup(self, param_dict, az_list_str, host_ips=None, is_upgrade=True):
        index_name = 'eBackup'
        if is_upgrade:
            input_version = param_dict.get('eBackup_Version')
        else:
            input_version = self.get_old_version(host_ips, param_dict)
        cloud_info_list = self.cmdb_util.get_cloud_service_info_v3(param_dict.get('current_region_id'), index_name)
        for cloud_service in cloud_info_list:
            deploy_node = self.cmdb_util.get_deploy_node_info(param_dict.get('current_region_id'), index_name)
            if cloud_service.get("name", "").startswith("eBackup_service1"):
                extend_infos = cloud_service.get("extendInfos", [])
                extend_infos_list = [extend_info for extend_info in extend_infos if extend_info.get("key") != "az_list"]
                extend_infos_list.append({"key": "az_list", "value": az_list_str})
                cloud_service.update({"extendInfos": extend_infos_list})

            cloud_service.update({"deploy_node_infos": deploy_node, "version": input_version})
            service_name = cloud_service.get("name", "default")
            if is_upgrade and service_name.startswith("eBackup_service"):
                self.set_service_name(deploy_node, service_name, param_dict)
            logger.info(f"start set cloud info, info {cloud_service}.")
            self.cmdb_util.set_cloud_service_info_v3(param_dict.get('current_region_id'), cloud_service)

    def set_service_name(self, deploy_nodes, service_name, param_dict):
        upgrade_ip_list = param_dict["eBackup_Datamover_nodes"].replace(" ", "").split("|")
        cmdb_ip_list = []
        for deploy_node in deploy_nodes:
            ip_addresses = deploy_node.get('ipAddresses')
            cmdb_ip_list += [ip_address.get("ip", "") for ip_address in ip_addresses]
        for group_ips in upgrade_ip_list:
            node_ips = group_ips.split(";")
            for node_ip in node_ips:
                logger.info(f"Start to check current node:{node_ip} and cmdb_ip:{cmdb_ip_list}")
                if node_ip in cmdb_ip_list:  # cmdb上存在存储平面ip 因此需要判断是否在升级的管理ip列表中
                    CommonTools(self.project_id, self.pod_id).set_service_name_to_node(node_ips, service_name)
                    return  # 找到该IP后代表该ip所在的集群属于service_name所代表的集群，该集群只需调用一次设置方法

    def get_old_version(self, host_ips, param_dict):
        host_ip = host_ips[0]
        hcp_password = param_dict["eBackup_hcp_pwd"]
        root_password = param_dict["eBackup_root_pwd"]
        cmd = "cat /tmp/upgrade/upgrade_pkg/upgrade/versions.conf.old"
        ssh_client = self.ssh_client.ssh_create_client(host_ip, "hcp", hcp_password)
        self.ssh_client.ssh_send_command(ssh_client, 'su', 'Password:', 100)
        self.ssh_client.ssh_send_command(ssh_client, root_password, '#', 100)
        # The response of the function would be a list and "\n" in the string in the list, so take [0][:-1]
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)[0][:-1]
        self.ssh_client.ssh_close(ssh_client)
        return result

    def config_alarm(self):
        management_float_ip = self.get_management_float_ip()
        base_url = self.get_server_url(management_float_ip)
        portal_account = self.get_portal_account(management_float_ip)
        _, headers = self.login_portal(base_url, portal_account)
        try:
            self.config_alarm_report(base_url, headers)
        finally:
            if base_url != "" or headers != "":
                self.logout_portal(base_url, headers)
        return

    def config_alarm_report(self, base_url, headers):
        logger.info("Begin to config alarm report.")

        url = base_url + '/alarm_settings/default'
        alarm_user, alarm_pwd = self.manageone.getThirdpartyInfor(self.pod_id)[:2]
        if not alarm_user:
            logger.error("Alarm user is empty.")
            raise HCCIException(653039, "alarm user is empty")
        if not alarm_pwd:
            logger.error("Alarm pwd is empty.")
            raise HCCIException(653039, "alarm pwd is empty")

        oc_addrss = self.custom_pararms.get_oc_domain_name()
        logger.info("Alarm url:" + oc_addrss)
        data = '{"ISALARMON":1,"ALARMSERVICETYPE":"1","ALARMSERVERURL":"' + oc_addrss + '","ALARMUSERNAME":"' + \
               alarm_user + '","ALARMPASSWORD":"' + alarm_pwd + '"}'

        description = ''
        num = 0
        while num < 3:
            rsp = self.rest_api.put(url, headers, data)
            if rsp.status_code != 200:
                logger.error("Rest req return not 200.")
                raise HCCIException(653020)
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("Errcode or description is empty.")
                raise HCCIException(653019)
            if errcode != 0:
                logger.error(f"Config alarm report failed,description:{description}, retry config.")
                num += 1
                time.sleep(30)
            else:
                logger.info("Config alarm report succeeded.")
                return
        logger.error("config alarm report failed, description:" + description)
        raise HCCIException(653039, description)
