# -*- coding:utf-8 -*-
import ipaddress
import re

from IPy import IP
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_util import ProjectApi
from utils.common import log as logger
from utils.common import param_check as paramCheckUnit
from utils.common.error.hcci_error_code import get_code_msg

ODK_PARAMS_DICT_ALL = {
    'equipment_model': {'type': 'string', 'suggest': 653122},
    'bmc_ip': {'type': 'ip', 'suggest': 653121},
    'eBackup_install_os_params': {'type': 'string', 'suggest': 653125},
    'bmc_name': {'type': 'ip', 'suggest': 653123},
    'bmc_passwd': {'type': 'external_password', 'suggest': 653124},
    'management_plane': {'type': 'string', 'suggest': 653119},
    'production_storage_plane': {'type': 'string', 'suggest': 653120},
    'backup_storage_plane': {'type': 'string', 'suggest': 653126},
    'eBackup_independent_network': {'type': 'same', 'suggest': 653127},
    'plane_ports': {'type': 'string', 'suggest': 653134},
    'datamover_producestorage_iplist': {'type': 'string', 'suggest': 653135},
    'datamover_producestorage_iplist_storage_data': {'type': 'string', 'suggest': 653141}

}

SERVERLIST_PARAMS_DICT_ALL = {
    'cascading_region': {'type': 'string', 'suggest': 653111},
    'cascading_domain': {'type': 'string', 'suggest': 653111},
    'cascaded_domain': {'type': 'string', 'suggest': 653111},
    'cascaded_region': {'type': 'string', 'suggest': 653111},
    'openstack_region': {'type': 'string', 'suggest': 653111},
    'openstack_domain': {'type': 'domain_name', 'suggest': 653111},
    'cascaded_root_password': {'type': 'external_password', 'suggest': 653111},
    'cascaded_fsp_password': {'type': 'external_password', 'suggest': 653111},
    'cascaded_env_os_password': {'type': 'external_password',
                                 'suggest': 653111},
    'cascaded_externalom_netmask': {'type': 'netmask2', 'suggest': 653110},
    'cascaded_externalom_gateway': {'type': 'ip', 'suggest': 653110},
    'cascaded_externalom_tag': {'type': 'integer', 'suggest': 653112},
    'openstack_root_password': {'type': 'external_password',
                                'suggest': 653111},
    'openstack_fsp_password': {'type': 'external_password', 'suggest': 653111},
    'openstack_env_os_password':
        {'type': 'external_password', 'suggest': 653111},
    'external_om_netmask': {'type': 'netmask2', 'suggest': 653110},
    'external_om_gateway': {'type': 'ip', 'suggest': 653110},
    'external_om_vlan_id': {'type': 'integer', 'suggest': 653112},
    'datamover_externalom_netmask': {'type': 'netmask2', 'suggest': 653110},
    'datamover_externalom_gateway': {'type': 'ip', 'suggest': 653110},
    'datamover_externalom_vlan_id': {'type': 'integer', 'suggest': 653112},
    'FSMOMFloatingIP': {'type': 'ip', 'suggest': 653110},
    'FSMdswarePassword': {'type': 'external_password', 'suggest': 653111},
    'FSMrootPassword': {'type': 'external_password', 'suggest': 653111},
    'CascadedFsmCmdPasswd': {'type': 'external_password', 'suggest': 653111},
    'datamover_externalom_iplist': {'type': 'string', 'suggest': 653115},
    'existed_datamover_externalom_iplist':
        {'type': 'string', 'suggest': 653115},
    'existed_datamover_mg_float_ip': {'type': 'ip', 'suggest': 653110},
    'datamover_producestorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_producestorage_netmask': {'type': 'ip', 'suggest': 653110},
    'datamover_producestorage_gateway': {'type': 'ip', 'suggest': 653110},
    'existed_datamover_producestorage_iplist':
        {'type': 'string', 'suggest': 653115},
    'existed_datamover_producestorage_netmask':
        {'type': 'ip', 'suggest': 653110},
    'existed_datamover_producestorage_gateway':
        {'type': 'ip', 'suggest': 653110},
    'datamover_backupstorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_backupstorage_netmask': {'type': 'ip', 'suggest': 653110},
    'datamover_backupstorage_gateway': {'type': 'ip', 'suggest': 653110},
    'datamover_management_float_ip': {'type': 'ip', 'suggest': 653116},
    'datamover_internal_float_ip': {'type': 'ip', 'suggest': 653116},
    'az_related_to_datamover': {'type': 'string', 'suggest': 653111},
    'business_storage_urllist': {'type': 'string', 'suggest': 653113},
    'business_storage_admin_username': {'type': 'string', 'suggest': 653111},
    'business_storage_admin_password':
        {'type': 'external_password', 'suggest': 653111},
    'business_storage_superadmin_username':
        {'type': 'string', 'suggest': 653111},
    'business_storage_superadmin_password':
        {'type': 'external_password', 'suggest': 653111},
    'mdb_path': {'type': 'string', 'suggest': 653114},
    'mdb_port': {'type': 'integer', 'suggest': 653112},
    'mdb_user_or_s3ak': {'type': 'string', 'suggest': 653111},
    'mdb_password_or_s3sk': {'type': 'external_password', 'suggest': 653111},
    'backup_storageunit_path': {'type': 'string', 'suggest': 653114},
    'backup_storageunit_dns_ip': {'type': 'ip', 'suggest': 653110},
    'cifs_storageunit_username': {'type': 'string', 'suggest': 653111},
    'cifs_storageunit_password':
        {'type': 'external_password', 'suggest': 653111},
    's3_storageunit_ak': {'type': 's3_ak_string', 'suggest': 653111},
    's3_storageunit_sk': {'type': 's3_sk_string', 'suggest': 653111},
    'system_backup_sftp_dir': {'type': 'string', 'suggest': 653117},
    'system_backup_sftp_ip': {'type': 'ip', 'suggest': 653110},
    'system_backup_sftp_username': {'type': 'string', 'suggest': 653111},
    'system_backup_sftp_password':
        {'type': 'external_password', 'suggest': 653111},
    'system_backup_sftp_port': {'type': 'integer', 'suggest': 653112},
    'region_id': {'type': 'string', 'suggest': 653111},
    'timezone': {'type': 'string', 'suggest': 653111}
}

SERVERLIST_PARAMS_DICT_DATAMOVER = {
    'cascaded_externalom_netmask': {'type': 'netmask2', 'suggest': 653110},
    'cascaded_externalom_gateway': {'type': 'ip', 'suggest': 653110},
    'cascaded_externalom_tag': {'type': 'integer', 'suggest': 653112},
    'datamover_server_ip': {'type': 'ip', 'suggest': 653110},
    'external_om_netmask': {'type': 'netmask2', 'suggest': 653110},
    'external_om_gateway': {'type': 'ip', 'suggest': 653111},
    'external_om_vlan_id': {'type': 'integer', 'suggest': 653112},
    'FSMOMFloatingIP': {'type': 'ip', 'suggest': 653110},
    'FSMdswarePassword': {'type': 'external_password', 'suggest': 653111},
    'FSMrootPassword': {'type': 'external_password', 'suggest': 653111},
    'CascadedFsmCmdPasswd': {'type': 'external_password', 'suggest': 653111},
    'datamover_externalom_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_producestorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_producestorage_netmask': {'type': 'ip', 'suggest': 653110},
    'datamover_producestorage_gateway': {'type': 'ip', 'suggest': 653110},
    'datamover_backupstorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_backupstorage_netmask': {'type': 'ip', 'suggest': 653110},
    'datamover_backupstorage_gateway': {'type': 'ip', 'suggest': 653110},
    'datamover_management_float_ip': {'type': 'ip', 'suggest': 653116},
    'datamover_internal_float_ip': {'type': 'ip', 'suggest': 653110},
    'az_related_to_datamover': {'type': 'string', 'suggest': 653111},
    'business_storage_urllist': {'type': 'string', 'suggest': 653113},
    'business_storage_admin_username': {'type': 'string', 'suggest': 653111},
    'business_storage_admin_password':
        {'type': 'external_password', 'suggest': 653111},
    'business_storage_superadmin_username':
        {'type': 'string', 'suggest': 653111},
    'business_storage_superadmin_password':
        {'type': 'external_password', 'suggest': 653111},
    'mdb_path': {'type': 'string', 'suggest': 653114},
    'mdb_port': {'type': 'integer', 'suggest': 653112},
    'mdb_user_or_s3ak': {'type': 'string', 'suggest': 653111},
    'mdb_password_or_s3sk': {'type': 'external_password', 'suggest': 653111},
    'backup_storageunit_path': {'type': 'string', 'suggest': 653114},
    'backup_storageunit_dns_ip': {'type': 'ip', 'suggest': 653110},
    'cifs_storageunit_username': {'type': 'string', 'suggest': 653111},
    'cifs_storageunit_password':
        {'type': 'external_password', 'suggest': 653111},
    's3_storageunit_ak': {'type': 's3_ak_string', 'suggest': 653111},
    's3_storageunit_sk': {'type': 's3_sk_string', 'suggest': 653111},
    'timezone': {'type': 'string', 'suggest': 653111}
}

SERVERLIST_PARAMS_DICT_DATAMOVER_AND_DRIVER = {
    'cascaded_externalom_netmask': {'type': 'netmask2', 'suggest': 653110},
    'cascaded_externalom_gateway': {'type': 'ip', 'suggest': 653110},
    'cascaded_externalom_tag': {'type': 'integer', 'suggest': 653112},
    'datamover_server_ip': {'type': 'ip', 'suggest': 653110},
    'external_om_netmask': {'type': 'netmask2', 'suggest': 653110},
    'external_om_gateway': {'type': 'ip', 'suggest': 653111},
    'external_om_vlan_id': {'type': 'integer', 'suggest': 653112},
    'datamover_producestorage_netmask':
        {'type': 'netmask2', 'suggest': 653110},
    'FSMOMFloatingIP': {'type': 'ip', 'suggest': 653110},
    'FSMdswarePassword': {'type': 'external_password', 'suggest': 653111},
    'FSMrootPassword': {'type': 'external_password', 'suggest': 653111},
    'CascadedFsmCmdPasswd': {'type': 'external_password', 'suggest': 653111},
    'timezone': {'type': 'string', 'suggest': 653111},
    'cascaded_domain': {'type': 'string', 'suggest': 653111},
    'cascaded_region': {'type': 'string', 'suggest': 653111},
    'openstack_region': {'type': 'string', 'suggest': 653111},
    'openstack_domain': {'type': 'domain_name', 'suggest': 653111},
    'cascaded_root_password': {'type': 'external_password', 'suggest': 653111},
    'cascaded_fsp_password': {'type': 'external_password', 'suggest': 653111},
    'cascaded_env_os_password':
        {'type': 'external_password', 'suggest': 653111},
    'openstack_root_password': {'type': 'external_password',
                                'suggest': 653111},
    'openstack_fsp_password': {'type': 'external_password', 'suggest': 653111},
    'openstack_env_os_password':
        {'type': 'external_password', 'suggest': 653111},
    'datamover_externalom_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_producestorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_backupstorage_iplist': {'type': 'string', 'suggest': 653115},
    'datamover_backupstorage_netmask': {'type': 'ip', 'suggest': 653110},
    'datamover_backupstorage_gateway': {'type': 'ip', 'suggest': 653110},
    'datamover_management_float_ip': {'type': 'ip', 'suggest': 653116},
    'datamover_internal_float_ip': {'type': 'ip', 'suggest': 653110},
    'az_related_to_datamover': {'type': 'string', 'suggest': 653111},
    'business_storage_urllist': {'type': 'string', 'suggest': 653113},
    'business_storage_admin_username': {'type': 'string', 'suggest': 653111},
    'business_storage_admin_password':
        {'type': 'external_password', 'suggest': 653111},
    'business_storage_superadmin_username':
        {'type': 'string', 'suggest': 653111},
    'business_storage_superadmin_password':
        {'type': 'external_password', 'suggest': 653111},
    'mdb_path': {'type': 'string', 'suggest': 653114},
    'mdb_port': {'type': 'integer', 'suggest': 653112},
    'mdb_user_or_s3ak': {'type': 'string', 'suggest': 653111},
    'mdb_password_or_s3sk': {'type': 'external_password', 'suggest': 653111},
    'backup_storageunit_path': {'type': 'string', 'suggest': 653114},
    'backup_storageunit_dns_ip': {'type': 'ip', 'suggest': 653110},
    'cifs_storageunit_username': {'type': 'string', 'suggest': 653111},
    'cifs_storageunit_password':
        {'type': 'external_password', 'suggest': 653111},
    's3_storageunit_ak': {'type': 's3_ak_string', 'suggest': 653111},
    's3_storageunit_sk': {'type': 's3_sk_string', 'suggest': 653111}
}


class UserParams:
    external_om = [
        'datamover_externalom_iplist',
        'datamover_internal_float_ip',
        'datamover_management_float_ip'
    ]

    def __init__(self, params, ip_version):
        self.param_list = params.get("param_list")
        self.ip_version = ip_version
        self.external_om_range_key = 'external_om_range'
        self.external_om_range = ''
        self.error_log = {}

    def get_ebackup_param_ips(self):
        ips = []
        for item in self.param_list:
            if item.get('key') == self.external_om_range_key:
                self.external_om_range = item.get('value')
            if not item.get('value'):
                continue
            for i in item.get('value', "").split(','):
                if self.is_valid_ip(i):
                    ips.append(i)
        return ips

    def check_datamover_ip(self, datamover):
        ip_keys = [
            'datamover_externalom_iplist',
            'datamover_producestorage_iplist',
            'datamover_backupstorage_iplist',
            'datamover_internal_float_ip',
            'datamover_management_float_ip'
        ]

        ebackup_param_ips = self.get_ebackup_param_ips()
        for ip_key in ip_keys:
            if not datamover.get(ip_key):
                continue
            for datamover_ip in datamover.get(ip_key).split(','):
                if ebackup_param_ips.count(datamover_ip) > 1:
                    self.error_log.update({ip_key: get_code_msg(653071) % datamover_ip})

        for ip_key in self.external_om:
            if datamover.get(ip_key):
                for datamover_ip in datamover.get(ip_key).split(','):
                    self.judge_in_segment(ip_key, datamover_ip)

    def judge_in_segment(self, ip_key, host_ip):
        if not self.external_om_range:
            return
        for ip_range in self.external_om_range.split(','):
            net_start_ip, net_end_ip = ip_range.split('-')
            host_ip_num = IP(host_ip).int()
            start_ip_num = IP(net_start_ip).int()
            end_ip_num = IP(net_end_ip).int()
            if start_ip_num <= host_ip_num <= end_ip_num:
                self.error_log.update(
                    {ip_key: get_code_msg(653098) % host_ip})

    def is_valid_ip(self, datamover_ip):
        return paramCheckUnit.check_param_ip(datamover_ip, self.ip_version)


def get_sence_type(project_id):
    ebackup_deploy_type = '0'
    checked_params_dict = SERVERLIST_PARAMS_DICT_ALL
    if get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver'):
        ebackup_deploy_type = '1'
        checked_params_dict = SERVERLIST_PARAMS_DICT_DATAMOVER_AND_DRIVER
    if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
        ebackup_deploy_type = '2'
        checked_params_dict = SERVERLIST_PARAMS_DICT_DATAMOVER
    return ebackup_deploy_type, checked_params_dict


def get_mdb_backup_type(project_id):
    mdb_backup_type = ''
    mdb_backup_type_dict = {'eBackup_MSTU_FTPS': 'FTPS', 'eBackup_MSTU_NFS': 'NFS', 'eBackup_MSTU_S3': 'S3',
                            'eBackup_MSTU_FTP': 'FTP', 'eBackup_MSTU_SFTP': 'SFTP'}
    if not get_project_condition_boolean(project_id, 'eBackup_MSTU_OFF'):
        for key, value in mdb_backup_type_dict.items():
            if get_project_condition_boolean(project_id, key):
                mdb_backup_type = value
                break
    return mdb_backup_type


def get_backup_storageunit_type(project_id, datamover):
    host_num = 0
    ebackup_independent_network = False
    advanced_options_dict = ProjectApi().get_project_info(project_id).get('advanced_options')

    backup_storageunit_type = advanced_options_dict.get('eBackup_BSTU', 'not_BSTU')
    logger.info("backup_storageunit_type is %s" % backup_storageunit_type)

    if 'datamover_externalom_iplist' in datamover:
        host_num = len(datamover.get('datamover_externalom_iplist').split(','))

    if 'eBackupIndependentNetwork' in advanced_options_dict:
        ebackup_independent_network = advanced_options_dict.get('eBackupIndependentNetwork')
    return backup_storageunit_type, host_num, ebackup_independent_network


def update_error_msg(msg_key, checked_params_dict, dict_key=None):
    if not dict_key:
        dict_key = msg_key
    msg = "param key %s check failed" % msg_key
    if 'suggest' in checked_params_dict[dict_key]:
        msg = get_code_msg(checked_params_dict[dict_key].get('suggest'))
    return {dict_key: msg}


def deal_datamover_externalom_iplist(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') != 'datamover_externalom_iplist':
        return
    external_om_gateway = params.get('external_om_info').get('external_om_gateway')
    external_om_mask = params.get('external_om_info').get('external_om_netmask')
    if external_om_gateway != "" and external_om_mask != "":
        check_om_mask = paramCheckUnit.is_netmask2(external_om_mask)
        if not check_om_mask:
            logger.error(f"input param's type is wrong, external_om_netmask: {check_om_mask}")
            params.get('error_msg_dict_om_info').update(
                update_error_msg("check_om_mask", params.get('checked_params_dictlue'), 'external_om_netmask'))

        datamover_ip = params.get('value').split(',')
        check_data = [external_om_gateway, external_om_mask, datamover_ip]
        try:
            ip_segment_result = check_ip_segment(check_data)
        except Exception as err:
            logger.error(f"Failed to check if ip is in the correct segment, and the reason is {err}.")
            return
        if params.get('ebackup_independent_network'):
            if ip_segment_result:
                logger.error(
                    "input Planning param's is wrong ,info: key is %s,value is %s" % \
                    (params.get('key'), params.get('value')))
                params.get('error_msg_dict_om_info', dict()).update(update_error_msg(
                        params.get('key'), params.get('checked_params_dictlue'), "eBackup_independent_network"))
        else:
            if not ip_segment_result:
                logger.error(f"input param's type is wrong ,{params.get('key')}: {params.get('value')}")
                params.get('error_msg_dict_om_info').update(update_error_msg(
                        params.get('key'), params.get('checked_params_dictlue'), "eBackup_independent_network"))
    else:
        logger.error(f"input param's type is wrong, external_om_netmask: {external_om_mask}")
        params.get("error_msg_dict_om_info", dict()).update(
            update_error_msg("external_om_netmask", params.get('checked_params_dictlue')))


def check_basic_parameters0(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'datamover_externalom_iplist' or params.get('key') == 'datamover_producestorage_iplist':
        if params.get('value') == '':
            logger.error("input param's type is wrong,info:key is %s,value is %s" % \
                         (params.get('key'), params.get('value')))
            params.get('error_msg_dict', dict()).update(
                update_error_msg(params.get('key'), params.get('checked_params_dict')))
            return True
        if len(params.get('value').split(',')) != params.get('host_num'):
            logger.error(f"number of ip(key:{params.get('key')}) is not equal to datamover_externalom_iplist,"
                         f"value is {params.get('value')}")
            params.get('error_msg_dict', dict()).update(
                update_error_msg(params.get('key'), params.get('checked_params_dict')))
            return True
        for datamover_ip in params.get('value').split(','):
            if not paramCheckUnit.check_param_ip(datamover_ip, params.get('ip_version')):
                logger.error(f"input param's type is wrong, info:key is \
                    {params.get('key')}, value is {params.get('value')}")
                params.get('error_msg_dict', dict()).update(update_error_msg(params.get('key'), \
                                                                             params.get('checked_params_dict')))
        deal_datamover_externalom_iplist(**params)
        return True
    return False


def check_basic_parameters1(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'datamover_management_float_ip' \
            and 'datamover_externalom_iplist' in params.get('datamover') \
            and len(params.get('datamover').get('datamover_externalom_iplist').split(',')) == 1:
        return True

    if params.get('key') == 'existed_datamover_mg_float_ip' \
            and 'existed_datamover_externalom_iplist' in params.get('datamover') \
            and len(params.get('datamover', dict()).get('existed_datamover_externalom_iplist').split(',')) == 1:
        return True

    if params.get('key') == 'datamover_server_ip' and params.get('ebackup_deploy_type') == '1':
        return True
    if params.get('key') == 'cascaded_externalom_gateway' or params.get('key') == 'external_om_gateway':
        if 'datamover_externalom_iplist' in params.get('datamover') \
                and len(params.get('datamover', dict()).get('datamover_externalom_iplist').split(',')) == 1:
            return True

    if params.get('key') == 'datamover_internal_float_ip' \
            and len(params.get('datamover').get('datamover_externalom_iplist').split(',')) > 1 \
            and params.get('value') == params.get('datamover').get('datamover_management_float_ip'):
        logger.error("datamover_internal_float_ip can not be equal to datamover_management_float_ip(nodes >= 2)")
        params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True

    if params.get('key') == 'datamover_backupstorage_netmask' \
            and params.get('datamover').get('datamover_backupstorage_iplist') == '':
        return True

    if params.get('key') == 'datamover_backupstorage_gateway' and params.get('value') == '':
        return True
    return False


def check_basic_parameters2(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'business_storage_urllist' and params.get('value') != '':
        if not check_url_list(params.get('value'), params.get('ip_version')):
            logger.error("business_storage_urllist must be xx.xx.xx.xx:xxxx.")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True

    if params.get('key') == 'business_storage_urllist' and params.get('value') == '':
        return True

    if 'business_storage_urllist' in params.get('datamover') and \
            params.get('datamover').get('business_storage_urllist') == '' \
            and params.get('key') in ['business_storage_superadmin_username', 'business_storage_superadmin_password',
                                      'business_storage_admin_username', 'business_storage_admin_password']:
        return True

    if params.get('key') in ['mdb_port', 'mdb_path', 'mdb_user_or_s3ak',
                             'mdb_password_or_s3sk'] and get_project_condition_boolean(params.get('project_id'),
                                                                                       'eBackup_MSTU_OFF'):
        return True

    if params.get('key') in ['mdb_port', 'mdb_path', 'mdb_user_or_s3ak', 'mdb_password_or_s3sk'] \
            and params.get('datamover').get('mdb_backup_type') == 'S3' and not params.get('is_add_cloud_service'):
        return True

    if params.get('key') == 'system_backup_sftp_dir':
        if not check_system_backup_path(params.get('value')):
            logger.error("path must be /path")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True
    return False


def check_basic_parameters3(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'mdb_path':
        if params.get('datamover').get('mdb_backup_type') != '':
            if not check_backup_path(
                    params.get('datamover').get('mdb_backup_type'), params.get('value'), params.get('ip_version')):
                logger.error("path must be ip:/path")
                params.get('error_msg_dict').update(
                    update_error_msg(params.get('key'), params.get('checked_params_dict')))
        if params.get('value') != '' \
                and params.get('datamover').get('backup_storageunit_path', '') == params.get('value'):
            logger.error("the values of backup_storageunit_path  mdb_path are the same.")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True

    if params.get('key') in ['mdb_user_or_s3ak', 'mdb_password_or_s3sk', 'mdb_port'] \
            and params.get('datamover').get('mdb_backup_type') == 'NFS':
        return True

    if params.get('key') == 'mdb_port' and params.get('datamover').get('mdb_backup_type') == 'S3':
        return True
    return False


def check_basic_parameters4(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'backup_storageunit_path' and params.get('value') == '':
        return True

    if params.get('key') == 'backup_storageunit_dns_ip' and not params.get('value'):
        return True

    if params.get('key') == 'datamover_management_float_ip':
        if (params.get('ebackup_deploy_type') != '1' or params.get('host_num') < 2) and not params.get('value'):
            return True

    if params.get('key') in ['cifs_storageunit_username', 'cifs_storageunit_password'] \
            and params.get('datamover').get('backup_storageunit_path', '') == '':
        return True

    if (params.get('key') == 's3_storageunit_ak' or params.get('key') == 's3_storageunit_sk') \
            and params.get('datamover').get('backup_storageunit_path', '') == '':
        return True

    if params.get('key') in ['cifs_storageunit_username', 'cifs_storageunit_password'] \
            and params.get('datamover').get('cifs_storageunit_username') == '' \
            and params.get('datamover').get('cifs_storageunit_password') == '':
        return True

    if params.get('key') == 'backup_storageunit_path' and params.get('value') != '':
        if not check_backup_path(params.get('backup_storageunit_type'), params.get('value'), params.get('ip_version')):
            logger.error("path must be ip:/path")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True
    return False


def check_basic_parameters5(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    if params.get('key') == 'existed_datamover_externalom_iplist' \
            or params.get('key') == 'existed_datamover_producestorage_iplist':
        if params.get('value') == '':
            logger.error(f"input param's type is wrong,info:key is {params.get('key')}, value is{params.get('value')}")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
            return True
        for datamover_ip in params.get('value').split(','):
            if not paramCheckUnit.check_param_ip(datamover_ip, params.get('ip_version')):
                logger.error(f"input param's type is wrong,info:key is \
                    {params.get('key')}, value is{params.get('value')}")
                params.get('error_msg_dict').update(
                    update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True

    if params.get('key') == "openstack_region":
        if not check_openstack_region(params.get('value')):
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True
    if params.get('key') == "az_related_to_datamover":
        tmplist = params.get('value').split(',')
        for sub_value in tmplist:
            if not check_openstack_region_sub(sub_value):
                params.get('error_msg_dict').update(
                    update_error_msg(params.get('key'), params.get('checked_params_dict')))
                break
        return True
    return False


def check_basic_parameters(**params):
    """
    :param params:    backup_storageunit_type,        checked_params_dict,    datamover,          ebackup_deploy_type,
                    ebackup_independent_network,    error_msg_dict,         external_om_info,   host_num,
                    ip_version,                     is_add_cloud_service,   key,                project_id,     value
    """
    logger.info("key is:" + params.get('key'))
    if check_basic_parameters0(**params):
        return True
    logger.info("check param(%s)'s progress:10" % params.get('key'))
    if params.get('key') == 'datamover_backupstorage_iplist':
        if len(params.get('value').split(',')) != params.get('host_num'):
            logger.error(f"number of ip(key:{params.get('key')}) is not equal to datamover_externalom_iplist,"
                         f"value is {params.get('value')}")
            params.get('error_msg_dict').update(update_error_msg(params.get('key'), params.get('checked_params_dict')))
            return True
        for node_ip in params.get('value').split(','):
            if not paramCheckUnit.check_param_ip(node_ip, params.get('ip_version')):
                logger.error(f"input param's type is wrong,info:key is \
                    {params.get('key')}, value is {params.get('value')}")
                params.get('error_msg_dict').update(
                    update_error_msg(params.get('key'), params.get('checked_params_dict')))
        return True

    if check_basic_parameters1(**params):
        return True

    logger.info("check param(%s)'s progress:40" % params.get('key'))
    if check_basic_parameters2(**params):
        return True

    if check_basic_parameters3(**params):
        return True
    logger.info("check param(%s)'s progress:70" % params.get('key'))

    if check_basic_parameters4(**params):
        return True
    if check_basic_parameters5(**params):
        return True
    logger.info("check param(%s)'s progress:100" % params.get('key'))
    return False


def checked_params_dict_keys(*args, **kwargs):
    """
    :param args:  error_msg_dict, ip_version, external_om_info, datamover, project_id
    """
    ebackup_deploy_type, checked_params_dict = get_sence_type(args[4])
    backup_storageunit_type, host_num, ebackup_independent_network = get_backup_storageunit_type(args[4], args[3])

    is_add_cloud_service = False
    if get_project_condition_boolean(args[4], 'ExpansionAdCloudService'):
        is_add_cloud_service = True

    value = ''
    for key in list(checked_params_dict.keys()):
        found = False
        for tmp_dict in kwargs.get("params").get("param_list"):
            if tmp_dict.get("key") == key:
                found = True
                value = tmp_dict.get("value")
                break
        if not found:
            continue
        basic_parameters_dict = {'backup_storageunit_type': backup_storageunit_type,
                                 'checked_params_dict': checked_params_dict, 'ebackup_deploy_type': ebackup_deploy_type,
                                 'ebackup_independent_network': ebackup_independent_network,
                                 'is_add_cloud_service': is_add_cloud_service, 'host_num': host_num,
                                 'key': key, 'value': value, 'error_msg_dict': args[0], 'ip_version': args[1],
                                 'external_om_info': args[2], 'datamover': args[3], 'project_id': args[4]}
        if check_basic_parameters(**basic_parameters_dict):
            continue
        param_type = checked_params_dict.get(key, dict()).get('type')
        if not check_param(param_type, value, args[1]):
            if param_type == "password":
                logger.error("input param's type is wrong,info:key is %s" % key)
            else:
                logger.error("input param's type is wrong,info:key is %s,value is %s" % (key, value))
                args[0].update(update_error_msg(key, checked_params_dict))
    logger.info("Basic_Parameters ebackup param check end.")
    return host_num


def check_management_plane(tmp_dict, error_msg_dict, datamover):
    if not check_nic(tmp_dict.get('management_plane')):
        logger.error("management_plane is invalid.")
        error_msg_dict.update(update_error_msg('management_plane', ODK_PARAMS_DICT_ALL))

    if tmp_dict.get('production_storage_plane', '') == tmp_dict.get('management_plane'):
        logger.error("production_storage_plane can not be same as management_plane.")
        error_msg_dict.update(update_error_msg('production_storage_plane', ODK_PARAMS_DICT_ALL))

    if tmp_dict.get('backup_storage_plane') == tmp_dict.get('management_plane'):
        logger.error("backup_storage_plane can not be same as management_plane.")
        error_msg_dict.update(update_error_msg('backup_storage_plane', ODK_PARAMS_DICT_ALL))
        return True

    if tmp_dict.get('production_storage_plane', ''):
        if not check_nic(tmp_dict.get('production_storage_plane')):
            logger.error("production_storage_plane is invalid.")
            error_msg_dict.update(update_error_msg('production_storage_plane', ODK_PARAMS_DICT_ALL))

        if tmp_dict.get('production_storage_plane') == tmp_dict.get('backup_storage_plane'):
            logger.error("backup_storage_plane can not be same as management_plane.")
            error_msg_dict.update(update_error_msg('backup_storage_plane', ODK_PARAMS_DICT_ALL))
    else:
        if "datamover_producestorage_iplist" in datamover and datamover.get('datamover_producestorage_iplist'):
            logger.error("production_storage_plane can not be empty,as datamover_producestorage_iplist in "
                         "Basic_Parameters is not empty.")
            error_msg_dict.update(update_error_msg('production_storage_plane', ODK_PARAMS_DICT_ALL))

    if tmp_dict.get('backup_storage_plane') and not check_nic(tmp_dict.get('backup_storage_plane')):
        logger.error("backup_storage_plane is invalid.")
        error_msg_dict.update(update_error_msg('backup_storage_plane', ODK_PARAMS_DICT_ALL))

    if not tmp_dict.get('backup_storage_plane') and "datamover_backupstorage_iplist" in datamover \
            and datamover.get('datamover_backupstorage_iplist'):
        logger.error("backup_storage_plane can not be empty,as datamover_backupstorage_iplist in "
                     "Basic_Parameters is not empty.")
        error_msg_dict.update(update_error_msg('backup_storage_plane', ODK_PARAMS_DICT_ALL))
    return True


def ip_judge(ip_list, ip_range):
    judgement = 0
    if ip_range:
        for ip_int in ip_list:
            if IP(ip_range[0]).int() <= ip_int <= IP(ip_range[1]).int():
                judgement += 1
    return judgement


def check_producestorage_ip(kwargs, error_msg_dict, project_id):
    service_storage_data_range = []
    management_storage_data_range = []
    datamover_producestorage_iplist_int = []
    datamover_producestorage_iplist = []

    for tmp_dict in kwargs["params"]["param_list"]:
        param_key = tmp_dict.get("param_key")
        if param_key == "datamover_producestorage_iplist":
            datamover_producestorage_iplist = tmp_dict.get("param_value", " ").split(",")
        elif param_key == "service_storage_data_range" and tmp_dict.get("param_value") is not None:
            service_storage_data_range = tmp_dict.get("param_value").split("-")
        elif param_key == "management_storage_data_range" and tmp_dict.get("param_value") is not None:
            management_storage_data_range = tmp_dict.get("param_value").split("-")
    if get_project_condition_boolean(project_id, "ProjectDeploy"):
        check_produce_storage_ip_range(kwargs, datamover_producestorage_iplist, error_msg_dict)
    for ips in datamover_producestorage_iplist:
        datamover_producestorage_iplist_int.append(IP(ips).int())

    iret = ip_judge(datamover_producestorage_iplist_int, service_storage_data_range)
    if iret != 0:
        logger.error(
            f"input param is invalid, info: datamover producestorage IPs {datamover_producestorage_iplist} "
            f"conflict with service storage data range {service_storage_data_range}")
        error_msg_dict.update(update_error_msg('datamover_producestorage_iplist', ODK_PARAMS_DICT_ALL))
    iret = ip_judge(datamover_producestorage_iplist_int, management_storage_data_range)
    if iret != 0:
        logger.error(
            f"input param is invalid, info: datamover producestorage IPs {datamover_producestorage_iplist} "
            f"conflict with management storage data range {management_storage_data_range}")
        error_msg_dict.update(update_error_msg('datamover_producestorage_iplist', ODK_PARAMS_DICT_ALL))


def check_produce_storage_ip_range(kwargs, datamover_producestorage_iplist, error_msg_dict):
    logger.info("Start to check the validity of produce_storage_ip")
    check_data = []
    management_storage_data_gateway = ""
    service_storage_data_gateway = ""
    management_storage_data_netmask = ""
    service_storage_data_netmask = ""

    for tmp_dict in kwargs["params"]["param_list"]:
        param_key = tmp_dict.get("param_key")
        if param_key == "management_storage_data_gateway":
            management_storage_data_gateway = tmp_dict.get("param_value", "")
        if param_key == "service_storage_data_gateway":
            service_storage_data_gateway = tmp_dict.get("param_value", "")
        if param_key == "management_storage_data_netmask":
            management_storage_data_netmask = tmp_dict.get("param_value", "")
        if param_key == "service_storage_data_netmask":
            service_storage_data_netmask = tmp_dict.get("param_value", "")

    if service_storage_data_gateway and service_storage_data_netmask:
        check_data.extend([service_storage_data_gateway, service_storage_data_netmask, datamover_producestorage_iplist])
    else:
        check_data.extend([management_storage_data_gateway, management_storage_data_netmask,
                           datamover_producestorage_iplist])

    try:
        ip_segment_result = check_ip_segment(check_data)
    except Exception as err:
        logger.error(f"Failed to check if ip is in the correct segment, and the reason is {err}.")
        return

    if not ip_segment_result:
        if service_storage_data_gateway:
            logger.error(f"The ip in datamover_producestorage_iplist{datamover_producestorage_iplist} is not"
                         f"in the same plane of Service_Storage_Data")
            error_msg_dict.update(update_error_msg('datamover_producestorage_iplist_storage_data', ODK_PARAMS_DICT_ALL))
        else:
            logger.error(f"The ip in datamover_producestorage_iplist{datamover_producestorage_iplist} is not"
                         f"in the same plane of management_storage_data")
            error_msg_dict.update(update_error_msg('datamover_producestorage_iplist_storage_data', ODK_PARAMS_DICT_ALL))


def check_ebackup_install_params(is_auto_install_os, kwargs, error_msg_dict, ip_version, datamover):
    odk_host_num = 0
    if not is_auto_install_os:
        return odk_host_num

    for tmp_dict in kwargs["params"]["ebackup_install_params_dict_list"]:
        odk_host_num = odk_host_num + 1
        management_plane_ports = tmp_dict.get(
            'management_plane').split(',') if tmp_dict.get('management_plane', '') else []
        production_storage_plane_ports = tmp_dict.get(
            'production_storage_plane').split(',') if tmp_dict.get('production_storage_plane', '') else []
        backup_storage_plane_ports = tmp_dict.get(
            'backup_storage_plane').split(',') if tmp_dict.get('backup_storage_plane', '') else []
        for ports in management_plane_ports:
            if ports in production_storage_plane_ports:
                logger.error("input param is invalid, info: "
                             "management plane and production storage plane have used the same port value, "
                             f"({management_plane_ports} and {production_storage_plane_ports}) conflict.")
                error_msg_dict.update(update_error_msg('plane_ports', ODK_PARAMS_DICT_ALL))
            elif ports in backup_storage_plane_ports:
                logger.error("input param is invalid, info: "
                             "management plane and backup storage plane have used the same port value, "
                             f"{management_plane_ports} and {production_storage_plane_ports} conflict.")
                error_msg_dict.update(update_error_msg('plane_ports', ODK_PARAMS_DICT_ALL))
        for ports in production_storage_plane_ports:
            if ports in backup_storage_plane_ports:
                logger.error("input param is invalid, info: "
                             "production storage plane and backup storage plane have used the same port value, "
                             f"({production_storage_plane_ports} and {backup_storage_plane_ports}) conflict.")
                error_msg_dict.update(update_error_msg('plane_ports', ODK_PARAMS_DICT_ALL))

        if tmp_dict.get('equipment_model') == '':
            logger.error("input param's type is wrong,info:key is %s,value is %s" % (
                'equipment_model', tmp_dict.get('equipment_model')))
            error_msg_dict.update(update_error_msg('equipment_model', ODK_PARAMS_DICT_ALL))

        if not paramCheckUnit.check_param_ip(tmp_dict.get('bmc_ip'), ip_version):
            logger.error(
                "input param's type is wrong,info:key is %s,value is %s" % ('bmc_ip', tmp_dict.get('bmc_ip')))
            error_msg_dict.update(update_error_msg('bmc_ip', ODK_PARAMS_DICT_ALL))

        if not paramCheckUnit.check_param_string(tmp_dict.get('bmc_name')):
            logger.error(
                "input param's type is wrong,info:key is %s,value is %s" % ('bmc_name', tmp_dict.get('bmc_name')))
            error_msg_dict.update(update_error_msg('bmc_name', ODK_PARAMS_DICT_ALL))

        if not tmp_dict.get('bmc_passwd'):
            logger.error("input param's type is wrong,info:key is %s" % 'bmc_passwd')
            error_msg_dict.update(update_error_msg('bmc_passwd', ODK_PARAMS_DICT_ALL))

        if tmp_dict.get('management_plane') == '':
            logger.error("management_plane can not be empty.")
            error_msg_dict.update(update_error_msg('management_plane', ODK_PARAMS_DICT_ALL))
        else:
            check_management_plane(tmp_dict, error_msg_dict, datamover)
    return odk_host_num


def params_check(**kwargs):
    """插件参数校验接口函数
    :param kwargs:
    :return: 成功：True, ''， 失败：False，错误信息，格式：{key1:err_msg1, key2:err_msg2}
    """
    error_msg_dict = {}
    datamover = {}
    external_om_info = {}
    is_auto_install_os = False
    project_id = kwargs.get("project_id")
    ip_version = 6 if ProjectApi().is_ipv6_project(project_id) else 4

    key_list = ['datamover_externalom_iplist', 'datamover_producestorage_iplist', 'datamover_management_float_ip',
                'datamover_internal_float_ip', 'datamover_backupstorage_iplist', 'business_storage_urllist',
                'backup_storageunit_path', 'cifs_storageunit_username', 'cifs_storageunit_password',
                'existed_datamover_externalom_iplist']
    for tmp_dict in kwargs.get("params", dict()).get("param_list", []):
        if tmp_dict["key"] in key_list:
            datamover[tmp_dict["key"]] = tmp_dict["value"]
        elif tmp_dict["key"] in ['external_om_gateway', 'external_om_netmask']:
            external_om_info[tmp_dict["key"]] = tmp_dict["value"]
        elif tmp_dict["key"] == 'datamover_backupstorage_netmask':
            is_auto_install_os = True

    user_params = UserParams(kwargs.get("params"), ip_version)

    user_params.check_datamover_ip(datamover)
    if user_params.error_log:
        error_msg_dict.update(user_params.error_log)

    datamover['mdb_backup_type'] = get_mdb_backup_type(project_id)

    host_num = checked_params_dict_keys(error_msg_dict, ip_version, external_om_info, datamover, project_id, **kwargs)

    check_producestorage_ip(kwargs, error_msg_dict, project_id)

    odk_host_num = check_ebackup_install_params(is_auto_install_os, kwargs, error_msg_dict, ip_version, datamover)

    if is_auto_install_os:
        if host_num != odk_host_num:
            logger.error("odk host num is not equal to datamover_externalom_iplist.")
            error_msg_dict.update(update_error_msg('eBackup_install_os_params', ODK_PARAMS_DICT_ALL))

    if not error_msg_dict:
        logger.info("eBackup check params result: PASS")
        return True, ''
    else:
        logger.info("eBackup check params result: %s" % error_msg_dict)
        return False, error_msg_dict


def check_param(param_type, value, ip_version):
    check_result = True
    if param_type == "string":
        check_result = check_param_string(value)
    if param_type == "integer":
        check_result = paramCheckUnit.check_param_integer(value)
    if param_type == "ip":
        check_result = paramCheckUnit.check_param_ip(value, ip_version)
    if param_type == "netmask2":
        check_result = paramCheckUnit.is_netmask2(value)
    if param_type == "netmask":
        check_result = paramCheckUnit.check_param_netmask(value)
    if param_type == "email":
        check_result = paramCheckUnit.check_param_email(value)
    if param_type == "password":
        check_result = paramCheckUnit.check_param_password(value)
    if param_type == "domain_name":
        check_result = check_param_domain_name(value)
    if param_type in ["s3_ak_string", "s3_sk_string"]:
        check_result = bool(value)
    if param_type == "external_password":
        check_result = bool(value)
    return check_result


def check_param_string(param_name):
    pattern = r"^[a-zA-Z0-9:/_\.-]{1,}$"
    match_obj = re.match(pattern, param_name)
    return bool(match_obj)


def check_url_list(value, ip_version):
    for sub_url in value.split(','):
        if ip_version == 4:
            urlinfo = sub_url.split(':')
        else:
            urlinfo = []
            start_index = sub_url.find('[')
            if start_index != 0:
                return False
            end_index = sub_url.find(']')
            if end_index == -1:
                return False
            urlinfo.append(sub_url[start_index + 1:end_index])
            if len(sub_url) < (end_index + 3):
                return False
            if sub_url[end_index + 1] != ':':
                return False
            urlinfo.append(sub_url[(end_index + 2):])

        if len(urlinfo) != 2:
            return False
        if not paramCheckUnit.check_param_ip(urlinfo[0], ip_version):
            return False
        if not paramCheckUnit.check_param_integer(urlinfo[1]):
            return False
    return True


def check_system_backup_path(path):
    pattern = r"^[/][a-zA-Z0-9\.#%\-=@\{_\}~\/:]{0,253}[a-zA-Z0-9\.#%\-=@\{_\}~:]$"
    match_obj = re.match(pattern, path, re.I)
    return bool(match_obj)


def check_nic(nic_str):
    pattern = r"[0-9]{1,}\.[0-9]{1,},[0-9]{1,}\.[0-9]{1,}$"
    match_obj = re.match(pattern, nic_str, re.I)
    return bool(match_obj)


def get_backup_path_info_of_ipv6(pathinfo, path):
    start_index = path.find('[')
    if start_index != 0:
        return False
    end_index = path.find(']')
    if end_index == -1:
        return False
    pathinfo.append(path[start_index + 1:end_index])
    if len(path) < (end_index + 3):
        return False
    if path[end_index + 1] != ':':
        return False
    pathinfo.append(path[(end_index + 2):])
    return True


def check_backup_fs_path(fs_type, pathinfo):
    if fs_type == 'eBackup_BSTU_NFS' or fs_type == 'FTPS' or \
            fs_type == 'NFS':
        pattern = r"^[/][a-zA-Z0-9\.#%\-=@\{_\}~\/:]{0,253}" \
                  r"[a-zA-Z0-9\.#%\-=@\{_\}~:]$"
        match_obj = re.match(pattern, pathinfo[1], re.I)
        if not match_obj:
            return False
        if "//" in pathinfo[1]:
            return False
    elif fs_type == 'SFTP' or fs_type == 'FTP':
        pattern = r"^[/][a-zA-Z0-9\.\-_/]{3,255}$"
        match_obj = re.match(pattern, pathinfo[1], re.I)
        if not match_obj:
            return False
        if pathinfo[1].find("//") > 0:
            return False
        if pathinfo[1][-1] == '/':
            return False
    else:
        pattern = r"^[/][a-zA-Z0-9\.\-_]{3,255}$"
        match_obj = re.match(pattern, pathinfo[1], re.I)
        if not match_obj:
            return False
    return True


def check_backup_path(fs_type, path, ip_version):
    if fs_type == 'not_BSTU':
        return True
    if fs_type != 'eBackup_BSTU_CIFS':
        if ip_version == 4:
            pathinfo = path.split(':')
            if len(pathinfo) != 2:
                return False
        else:
            pathinfo = []
            if not get_backup_path_info_of_ipv6(pathinfo, path):
                return False
        if not paramCheckUnit.check_param_ip(pathinfo[0], ip_version):
            if not check_param_domain_name(pathinfo[0]):
                return False
        if not check_backup_fs_path(fs_type, pathinfo):
            return False
    else:
        if path[0:2] != '//':
            return False
        tmp_path = path[2:]
        pos = tmp_path.find('/')
        if pos < 0:
            return False
        backup_path_ip = tmp_path[0:pos]
        if not paramCheckUnit.check_param_ip(backup_path_ip, ip_version) \
                or not check_param_domain_name(backup_path_ip):
            return False
        tmp_dir = tmp_path[pos + 1:]
        pattern = r"^[a-zA-Z0-9\.#%\-=@\{_\}~\/:]{0,253}" \
                  r"[a-zA-Z0-9\.#%\-=@\{_\}~:]$"
        match_obj = re.match(pattern, tmp_dir, re.I)
        if not match_obj:
            return False
    return True


def check_param_domain_name(param_name):
    pattern = r"^[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(\.[a-zA-Z0-9]" \
              r"[-a-zA-Z0-9]{0,62})+$"
    match_obj = re.match(pattern, param_name, re.I)
    return bool(match_obj)


def check_openstack_region(param_name):
    return check_param_string(param_name)


def check_openstack_region_sub(param_name):
    return check_param_string(param_name)


def check_ip_segment(check_params):
    ip_gateway = check_params[0]
    ip_netmask = check_params[1]
    check_ip_list = check_params[2]
    tmp_list = [bin(int(x)).count("1") for x in ip_netmask.split(".")]
    ip_mask_num = str(sum(tmp_list))

    ip_and_mask = ip_gateway + '/' + ip_mask_num

    for check_ip in check_ip_list:
        result = ipaddress.ip_address(check_ip) in ipaddress.ip_network(
            ip_and_mask, strict=False)
        logger.info(
            "check ip:%s in the same segmeng as gateway, result is %s." %
            (check_ip, result))
        if result:
            return True
        else:
            logger.info(
                "check ip:%s is not in the %s network"
                " segment." % (check_ip, ip_and_mask))
            return False

# the code has been updated to python3.7
