# -*- coding:utf-8 -*-
import os

from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_conditions
from utils.business.project_util import ProjectApi
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.scripts.common import ebackup_util as ebackup_handle
from plugins.eBackup.common.ebackup_node import EBackupNode

logger.init("eBackup Install UltraPath")


class InstallUltraPath(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.params = ParamUtil()
        self.ssh_obj = Ssh()
        self.project_api = ProjectApi()
        self.__config_dict = self.params.get_service_cloud_param(
            pod_id, "eBackup")
        self.hcp_passwd = self.__config_dict['hcp_ssh_password']
        self.root_passwd = self.__config_dict['eBackup_os_root_password']

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """

    def main_handle(self, project_id, pod_id):
        condition_dic = get_project_conditions(self.project_id)
        ip_list = self.__config_dict['datamover_externalom_iplist'].lower()
        is_bc_installed = condition_dic.get('BCInstalled', '')
        if is_bc_installed:
            ip_list = self.__config_dict['existed_datamover_externalom_iplist'].lower()
            for node_ip in ip_list.split(','):
                account_info = EBackupNode(pod_id, project_id).get_account_info("", externalom_ips=node_ip)
                self.hcp_passwd = account_info.get('hcp')
                self.root_passwd = account_info.get('root')
                self.install_ultrapath_package(node_ip)
        else:
            self.install_ultrapath_package(ip_list)
        return Message(200)

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return self.main_handle(project_id, pod_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error("install ultra path failed:%s" % str(err))
            return Message(500, err, err)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        pass

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def install_ultrapath_package(self, ip_list):
        path, name = API.find_file("OceanStor_UltraPath", "LinuxOther.zip")
        if name == "":
            raise HCCIException(
                653004, "OceanStor_UltraPath......LinuxOther.zip")
        file_path = os.path.join(path, name)
        logger.info("find ultra path package succ.")
        ebackup_handle.check_compress_file([file_path])
        ssh_user_name = "hcp"
        ssh_passwd = self.hcp_passwd
        for datamover_ip in ip_list.split(','):
            try:
                self.ssh_obj.put_file(
                    datamover_ip, ssh_user_name,
                    ssh_passwd, file_path, ".", 22)
            except HCCIException as err:
                logger.info("%s" % str(err))
                ssh_user_name = "root"
                ssh_passwd = self.root_passwd
                self.ssh_obj.put_file(
                    datamover_ip, ssh_user_name, ssh_passwd,
                    file_path, ".", 22)
            except Exception as err:
                ssh_user_name = "root"
                ssh_passwd = self.root_passwd
                logger.error("ssh login failed, error:%s" % str(err))
                self.ssh_obj.put_file(
                    datamover_ip,
                    ssh_user_name,
                    ssh_passwd,
                    file_path, ".", 22)
            self.ssh_obj.put_file(
                datamover_ip, ssh_user_name, ssh_passwd,
                "/opt/cloud/hcci/src/HCCI"
                "/plugins/eBackup/shell_tool/InstallUltraPath.sh",
                ".", 22)
            logger.info("upload ultra path package succ.")
            self.execute_install_ultra_path(ssh_user_name, datamover_ip,
                                            ssh_passwd)

    def execute_install_ultra_path(self, ssh_user_name, datamover_ip,
                                   ssh_passwd):
        cmd = "chmod 550 InstallUltraPath.sh;dos2unix" \
              " InstallUltraPath.sh;chown hcp:hcp InstallUltraPath.sh;" \
              "/bin/bash InstallUltraPath.sh;" \
              "rm -rf /home/hcp/InstallUltraPath.sh"
        if ssh_user_name != "root":
            result = self.ssh_obj.ssh_cmds(
                datamover_ip, cmd, ssh_user_name,
                ssh_passwd, self.root_passwd, "", "")
        else:
            ssh_client = self.ssh_obj.ssh_create_client(
                datamover_ip, ssh_user_name, ssh_passwd)
            result = self.ssh_obj.ssh_exec_command_return(ssh_client, cmd)
            self.ssh_obj.ssh_close(ssh_client)
        if "install ultrapath succ" in result or \
                "install ultrapath succ\n" in result:
            logger.info("execute install ultra path successful")
        else:
            logger.error("execute install ultra path failed")
            raise HCCIException(653013, cmd)
# the code has been updated to python3.7
