# -*- coding:utf-8 -*-
import json

import utils.common.log as logger
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.ebackup_rest import EbackupRest

logger.init("eBackup connect manageone")


class VulnerabilityManagement(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup connect to manageone")
        self.project_id = project_id
        self.pod_id = pod_id
        self.manageone = ManageOneUtil2()
        self.params = ParamUtil()
        self.rest_api = EbackupRest()
        self.err_code = 0
        self.oc_token = None
        self.__db_param_dict = self.params.get_service_cloud_param(
            pod_id, "eBackup")

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            if get_project_condition_boolean(project_id, 'eBackup_Driver') or \
                    get_project_condition_boolean(project_id, 'eBackup_Proxy'):
                return Message(200)
        except Exception as err:
            logger.error("Failed to get project condition, the reason is %s" % err)
            return Message(500, err, err)

        try:
            # 获取版本号
            path, name = API.find_file("OceanStor", "eBackup_DMK.zip")
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error("get project condition or Find the file failed' ,the reason is %s" % err)
            return Message(500, err, err)

        if name == "":
            raise HCCIException(653004, "OceanStor......eBackup_DMK.zip")
        current_version = API.get_package_version(name)

        service_system_info = []
        datamover_float_ip = self.get_datamover_float_ip()
        datamover_system_info = {"name": self.__db_param_dict["region_id"] + '_' + "eBackup-Server",
                                 "alias": self.__db_param_dict["region_id"] + '_' + "eBackup-Server",
                                 "product_name": "eBackup",
                                 "product_version": current_version,
                                 "product_patch_version": [],
                                 "ipaddress": datamover_float_ip}
        service_system_info.append(datamover_system_info)

        try:
            self.exec_report_system_info(service_system_info)
            return Message(200)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error(f"ebackup Vulnerability Management report to ManageOne failed the reason is {err}.")
            return Message(500, err, err)

    def exec_report_system_info(self, service_system_info):
        alarm_ip, alarm_user, alarm_pwd = self.get_oc_info()
        self.login(alarm_ip, alarm_user, alarm_pwd)
        self.report_system_info(service_system_info, alarm_ip)
        self.logout(alarm_ip)

    def get_datamover_float_ip(self):
        externalom_ip_lower = self.__db_param_dict[
            'datamover_externalom_iplist'
        ].lower()
        iplist = externalom_ip_lower.split(',')
        datamover_iplist = [ipStr.lower() for ipStr in iplist]
        if len(datamover_iplist) == 1:
            datamover_float_ip = datamover_iplist[0]
        else:
            datamover_float_ip = self.__db_param_dict[
                'datamover_management_float_ip'
            ].lower()
        return datamover_float_ip

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_oc_info(self):
        if_ipv6 = 0
        domain_name = 0
        pro_dic = get_project_conditions(self.project_id)
        if 'ManageIPV6' in pro_dic:
            if_ipv6 = pro_dic["ManageIPV6"]
        om_info = ManageOneUtil2().getMoOmInfo(self.pod_id)
        alarm_ip = om_info[0]
        if om_info.__len__() >= 5:
            domain_name = 1
            alarm_ip = om_info[4]
        if if_ipv6 and domain_name == 0:
            alarm_ip = '[%s]' % alarm_ip

        alarm_user, alarm_pwd = self.manageone.getThirdpartyInfor(self.pod_id)[:2]
        if alarm_user is None or alarm_user == '':
            logger.error("alarm user is empty.")
            raise HCCIException(653077, "third party user is empty.")
        if alarm_pwd == '' or alarm_pwd == '':
            logger.error("alarm pwd is empty.")
            raise HCCIException(653077, "third party user pwd is empty.")
        return alarm_ip, alarm_user, alarm_pwd

    def login(self, alarm_ip, alarm_user, alarm_pwd):
        logger.info("Begin to login to manageone to get session.")
        url = "https://%s:26335/rest/plat/smapp/v1/sessions" % alarm_ip
        header = {
            "Accept": "application/json",
            "Content-Type": "application/json;charset=UTF-8"
        }
        data = {
            "grantType": "password",
            "userName": alarm_user,
            "value": alarm_pwd
        }

        rsp = self.rest_api.put(url=url, headers=header, data=json.dumps(data))
        if rsp.status_code == 200:
            rsp_data = rsp.json()
            if 'accessSession' not in list(rsp_data.keys()):
                logger.error(
                    "Get session failed.The response body is:" + str(rsp_data))
                raise HCCIException(
                    653077,
                    "get session failed.The response body is:" + str(rsp_data))
            self.oc_token = rsp_data["accessSession"]
            logger.info("Login to manageone to get session successfully.")
        else:
            logger.error(
                "Get session failed,the status_code is "
                "" + str(rsp.status_code))
            raise HCCIException(
                653077, "get session failed.")

    def logout(self, alarm_ip):
        logger.info("Begin to logout manageone.")
        url = "https://{}:26335/rest/plat/smapp/v1/sessions".format(
            alarm_ip)
        header = {
            "Accept": "application/json",
            "Content-Type": "application/json;charset=UTF-8",
            "X-Auth-Token": self.oc_token
        }
        rsp = self.rest_api.delete(url=url, headers=header)
        if rsp.status_code == 200:
            logger.info("Logout manageone successfully.")
        else:
            logger.error("Get session failed,the status_code is:" + str(
                rsp.status_code))

    def report_system_info(self, service_info, alarm_ip):
        url = "https://%s:26335/rest/cmdb/v2/system-infos" % alarm_ip
        header = {
            "Accept": "application/json",
            "Content-Type": "application/json;charset=UTF-8",
            "X-Auth-Token": self.oc_token
        }
        data = {"system_infos": service_info}

        rsp = self.rest_api.put(url=url, headers=header, data=json.dumps(data))
        if rsp.status_code == 200:
            rsp_data = rsp.json()
            if int(rsp_data["resultCode"]) != 0:
                msg = "Report system info failed.The error code is  %s " \
                      "The error msg is:%s" \
                      % (str(rsp_data["resultCode"]), rsp_data["resultData"])
                logger.error(msg)
                raise HCCIException(
                    653077,
                    "report system info failed, the error msg is:" + str(
                        rsp_data["resultData"]))
            logger.info("Report system info successfully.")
        else:
            logger.error("report system info failed,the status_code is " + str(
                rsp.status_code))
            raise HCCIException(
                653077,
                "report system info failed")

# the code has been updated to python3.7
