# -*- coding:utf-8 -*-
import threading
import os
import shutil

import utils.common.log as logger

from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.model import AuthFtp
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import SFS_CERT_TYPE
from plugins.eReplication.common.constant import NOVA_CERT_TYPE
from plugins.eReplication.common.constant import MO_CERT_TYPE
from plugins.eReplication.common.client.mo_client import API as MO_API


class API(object):

    def __init__(self, auth_provider):
        self.host = auth_provider.host
        self.ssh_user = auth_provider.ssh_user
        self.ssh_pwd = auth_provider.ssh_pwd
        self.sudo_user = auth_provider.sudo_user
        self.sudo_pwd = auth_provider.sudo_pwd
        self.sudo_client = SSH_API.get_sudo_ssh_client(auth_provider)

    def __del__(self):
        if self.sudo_client:
            SSH_API.close_ssh(self.sudo_client)
        logger.info(f"SSH channel is closed: {self.host}.")

    @classmethod
    def get_instance(cls, *args, **kwargs):
        return cls(*args, **kwargs)

    @classmethod
    def get_curl_cmd(cls, project_id, pod_id):
        mo_api = MO_API(project_id, pod_id)
        vdc_admin_token = mo_api.get_vdc_admin_token()
        params = Params(project_id, pod_id)
        region_info = params.project_region_id
        global_domain_name = mo_api.global_domain(is_central_cloud=False)
        download_sfs_cert_url = f"https://sfs-api.{region_info}.{global_domain_name}:8000/cps/openapi/cert/download_ca"
        return rf'curl -k -X POST {download_sfs_cert_url} -H "X-Auth-Token: {vdc_admin_token}"\
         -H "Content-Type: application/json" 2>/dev/null'

    @classmethod
    def import_sys_cert(cls, project_id, pod_id, cert_type):
        logger.info(f"Start import sys cert, cert type is {cert_type}.")
        if cert_type not in [SFS_CERT_TYPE, NOVA_CERT_TYPE, MO_CERT_TYPE]:
            logger.warning(f"The cert type is incorrect.")
            return

        nodes = Nodes(project_id, pod_id)
        cert_funcs = list()
        thread_name = threading.current_thread().name
        for host in nodes.all_hosts:
            logger.info(f"Start import {cert_type} certs to {host}.")
            auth_provider = Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
            cert_func = (
                cls._import_sys_cert_to_bcm, thread_name,
                (auth_provider, project_id, pod_id, cert_type), {}
            )
            cert_funcs.append(cert_func)
        ExcThread.exec_func_in_thread(cert_funcs)
        if os.path.exists(Path.LOCAL_TEMP_PATH):
            shutil.rmtree(Path.LOCAL_TEMP_PATH)
        logger.info("Import The sys cert is complete.")

    @classmethod
    def _import_sys_cert_to_bcm(
            cls, auth_provider, project_id, pod_id, cert_type):
        logger.info(f"Start import sys cert to node {auth_provider.host}, cert type is {cert_type}.")
        host = auth_provider.host
        ssh_user = auth_provider.ssh_user
        ssh_pwd = auth_provider.ssh_pwd
        sudo_user = auth_provider.sudo_user
        sudo_pwd = auth_provider.sudo_pwd
        ins = cls.get_instance(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
        cert_lst = cls._get_cert_list(project_id, pod_id, cert_type)
        for cert_alias, cert_name in cert_lst.items():
            cmd = f"[ -f '{Path.NODE_TMP_PATH}/{cert_name}' ] && echo CMD_RESULT=$?"
            result = SSH_API.exec_command_return_list(ins.sudo_client, cmd)
            if "CMD_RESULT=0" not in str(result):
                logger.info(f"{cert_type} cert {cert_name} not exists, now to get it.")
                cls._upload_sys_certs(host, ssh_user, ssh_pwd, cert_type)
            cmd = f"chown -Rh {ssh_user}:LEGO {Path.NODE_TMP_PATH};chmod -R 750 {Path.NODE_TMP_PATH}"
            SSH_API.send_command(ins.sudo_client, cmd)
            # 导入证书
            run_cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh import_cert.sh; echo CMD_RESULT=$?"
            SSH_API.send_command(ins.sudo_client, run_cmd, "Please input the cert alias: ")
            SSH_API.send_command(ins.sudo_client, cert_alias, "Please input the cert file path: ")
            result = SSH_API.send_command(ins.sudo_client, f"{Path.NODE_TMP_PATH}/{cert_name}")
            if "CMD_RESULT=0" not in str(result):
                err_msg = f"{cert_type} cert {cert_name} failed to be imported to the node {host}."
                logger.error(err_msg)
                raise Exception(err_msg)
            logger.info(f"{cert_type} cert {cert_name} is imported to the node {host}.")
        else:
            logger.info(f"Import {cert_type} certs to node {host} success.")
            cmd = f"rm -rf {Path.NODE_TMP_PATH}"
            SSH_API.send_command(ins.sudo_client, cmd)

    @classmethod
    def _get_cert_list(cls, project_id, pod_id, cert_type):
        current_region_id = Params(project_id, pod_id).project_region_id
        if cert_type == SFS_CERT_TYPE:
            cert_lst = {f"{current_region_id}.sfs": "haproxy_tenant_ca-cert.pem"}
        elif cert_type == NOVA_CERT_TYPE:
            cert_lst = {f"{current_region_id}.nova": Path.NOVA_CERT_NAME}
        else:
            cert_lst = {
                "iamcert": "trust_apimlb_iam.cer", "sccert": "trust_apimlb_sc.cer",
                "occert": "trust_apimlb_oc.cer", "logcert": "cert.pem"}
        return cert_lst

    @classmethod
    def _upload_sys_certs(cls, host, ssh_user, ssh_pwd, cert_type):
        if cert_type == SFS_CERT_TYPE:
            sfs_file = f"{Path.LOCAL_TEMP_PATH}/haproxy_tenant_ca-cert.pem"
            cert_file_lst = [sfs_file]
        elif cert_type == NOVA_CERT_TYPE:
            cert_file_lst = [f"{Path.LOCAL_TEMP_PATH}/{Path.NOVA_CERT_NAME}"]
        else:
            iam_file = f"{Path.LOCAL_TEMP_PATH}/trust_apimlb_iam.cer"
            sc_file = f"{Path.LOCAL_TEMP_PATH}/trust_apimlb_sc.cer"
            oc_file = f"{Path.LOCAL_TEMP_PATH}/trust_apimlb_oc.cer"
            log_file = f"{Path.LOCAL_TEMP_PATH}/cert.pem"
            cert_file_lst = [iam_file, sc_file, oc_file, log_file]

        for cert_file in cert_file_lst:
            upload_res = SSH_API.put_file(host, ssh_user, ssh_pwd, cert_file, Path.NODE_TMP_PATH)
            if not upload_res:
                err_msg = f"Upload cert file to {host} failed."
                logger.error(err_msg)
                raise Exception(err_msg)
        logger.info(f"Upload {cert_type} certs to {host} success.")

    @classmethod
    def get_nova_cert(cls, project_id, pod_id):
        logger.info("Start copy remote nova_ca.crt to temp path.")
        if not os.path.exists(Path.LOCAL_TEMP_PATH):
            os.mkdir(Path.LOCAL_TEMP_PATH)
        # 复制到fsp目录
        if not cls._make_cert_file(project_id, pod_id):
            raise Exception("Copy remote nova_ca.crt failed.")
        # 从cps节点拷贝到本地
        params = Params(project_id, pod_id)
        reverse_ip = params.openstack_reverse_ip
        fsp_pwd = params.openstack_fsp_pwd
        auth_ftp = AuthFtp(reverse_ip, "fsp", fsp_pwd).auth_provider
        SSH_API.get_file(auth_ftp, Path.TEMP_NOVA_CERT_FILE,
                         f"{Path.LOCAL_TEMP_PATH}/{Path.NOVA_CERT_NAME}")
        logger.info("Copy remote nova_ca.certs to temp path success.")

    @classmethod
    def import_sfs_cert(cls, project_id, pod_id):
        logger.info(f"Start import sfs cert.")
        nodes = Nodes(project_id, pod_id)
        cert_funcs = list()
        thread_name = threading.current_thread().name
        for host in nodes.all_hosts:
            auth_provider = Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
            cert_func = (cls._import_sfs_cert_to_bcm, thread_name, (auth_provider, project_id, pod_id), {})
            cert_funcs.append(cert_func)
        ExcThread.exec_func_in_thread(cert_funcs)
        if os.path.exists(Path.LOCAL_TEMP_PATH):
            shutil.rmtree(Path.LOCAL_TEMP_PATH)
        logger.info("Import The sfs cert is complete.")

    @classmethod
    def _import_sfs_cert_to_bcm(cls, auth, project_id, pod_id):
        logger.info(f"Start import sfs cert to node {auth.host}.")
        sudo_client = SSH_API.get_sudo_ssh_client(auth)
        try:
            cert_lst = cls._get_cert_list(project_id, pod_id, SFS_CERT_TYPE)
            curl_cmd = cls.get_curl_cmd(project_id, pod_id)
            for cert_alias, cert_name in cert_lst.items():
                # 生成证书
                sfs_cert_name = f"{Path.DR_USER_PATH}/{cert_name}"
                cmd = rf'{curl_cmd} | openssl x509 > {sfs_cert_name}'
                SSH_API.send_command(sudo_client, cmd)
                cmd = f"chown {auth.ssh_user}:LEGO {sfs_cert_name};chmod 660 {sfs_cert_name}"
                SSH_API.send_command(sudo_client, cmd)
                # 导入证书
                run_cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh import_cert.sh; echo CMD_RESULT=$?"
                SSH_API.send_command(sudo_client, run_cmd, "Please input the cert alias: ")
                SSH_API.send_command(sudo_client, cert_alias, "Please input the cert file path: ")
                result = SSH_API.send_command(sudo_client, sfs_cert_name)
                if "CMD_RESULT=0" not in str(result):
                    err_msg = f"Sfs cert {cert_name} failed to be imported to the node {auth.host}."
                    logger.error(err_msg)
                    raise Exception(err_msg)
                logger.info(f"Sfs cert {cert_name} imported to the node {auth.host}.")
        finally:
            SSH_API.close_ssh(sudo_client)

    @classmethod
    def _make_cert_file(cls, project_id, pod_id):
        logger.info('Start copying nova_ca.crt and change owners.')
        params = Params(project_id, pod_id)
        reverse_ip = params.openstack_reverse_ip
        root_pwd = params.openstack_sudo_pwd
        fsp_pwd = params.openstack_fsp_pwd
        logger.info(f'Ssh to cps node:{reverse_ip}.')
        ssh_client = SSH_API.get_sudo_ssh_client(Auth(reverse_ip, "fsp", fsp_pwd, "root", root_pwd))
        cmd = fr"\cp -rf {Path.CPS_NOVA_CERT_FILE} {Path.TEMP_NOVA_CERT_FILE} && echo CMD_RESULT=$?"
        logger.info(f'Cmd:{cmd}.')
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            raise Exception("Copy nova_ca.crt failed.")
        cmd = f"chown -h fsp:fsp {Path.TEMP_NOVA_CERT_FILE} && echo CMD_RESULT=$?"
        logger.info(f'Cmd:{cmd}.')
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            raise Exception("Change owners failed.")
        logger.info("Copying nova_ca.crt and change owners success.")
        return True
