# -*- coding:utf-8 -*-
import base64
import json
import os
import threading
import shutil
from collections import defaultdict
from time import sleep

import requests
import utils.common.log as logger
from IPy import IP
from utils.business.apig_util import APIGApi
from utils.business.dns_util import DNSApi
from utils.business.dns_util import DNSNodeType
from utils.business.param_util import CheckResult
from utils.business.project_util import ProjectApi
import utils.common.software_package_util as file_util
from utils.common.compat import is_legacy_hcsu_scene
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.model import ApiRegisterInfo
from plugins.eReplication.common.lib.model import SSHClientInfo
from plugins.eReplication.common.lib.model import AuthFtp
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import check_param_domain_name
from plugins.eReplication.common.lib.utils import check_param_ip
from plugins.eReplication.common.lib.utils import check_value_null
from plugins.eReplication.common.lib.utils import get_install_services
from plugins.eReplication.common.request_api import RequestApi
from plugins.eReplication.common.os_api import API as OS_API


class API(object):

    def __init__(self, auth_provider):
        self.host = auth_provider.host
        self.ssh_user = auth_provider.ssh_user
        self.ssh_pwd = auth_provider.ssh_pwd
        self.sudo_user = auth_provider.sudo_user
        self.sudo_pwd = auth_provider.sudo_pwd
        self.sudo_client = SSH_API.get_sudo_ssh_client(auth_provider)

    def get_server_float_ip(self):
        logger.info("Begin to get server float ip.")
        cmd = \
            f"cat {Path.TOMCAT_WEB_XML_PATH} | grep sso.address.client | " \
            "grep https | awk -F'https://' '{print $2}' | " \
            "awk -F':9443' '{print $1}'"
        result = SSH_API.exec_command_return_list(self.sudo_client, cmd)
        if not result:
            raise Exception("Get server float ip return None.")
        server_float_ip = result[0].strip()
        logger.info(f"Get server float ip return {server_float_ip}.")
        if not check_value_null(server_float_ip) and check_param_ip(
                server_float_ip):
            return server_float_ip
        raise Exception("Get server float ip return None.")

    def get_current_region_id(self):
        logger.info("Begin to get current region id.")
        file_name = "node.properties"
        cmd = f"cat {Path.NODE_PROPERTIES_PATH} " \
              "| grep region | awk -F= '{print $2}'"
        result = SSH_API.exec_command_return_list(self.sudo_client, cmd)
        logger.info(f"Get current region id return {result}.")
        if not result:
            raise HCCIException("663604", file_name)
        return result[0]

    @classmethod
    def switch_dr_mode(cls, project_id, pod_id, region_id):
        nodes = Nodes(project_id, pod_id)
        physical, listen_ip = cls.get_server_node(
            nodes.all_hosts, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
            nodes.sudo_pwd)
        ins = cls.get_instance(Auth(physical, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd))
        result = SSH_API.exec_command_return_list(
            ins.sudo_client,
            f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh configFailoverMode.sh -a | "
            f"grep {region_id}")
        if region_id not in str(result):
            logger.error(f"Query failover mode failed: {result}.")
            raise Exception(f"Query failover mode failed: {result}.")
        if "manual" in str(result):
            logger.info(
                "The failover mode is already manual mode, "
                "no need change again.")
            return
        SSH_API.send_command(
            ins.sudo_client,
            f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh configFailoverMode.sh",
            "want to continue")
        SSH_API.send_command(ins.sudo_client, "yes", "Enter Region-id")
        SSH_API.send_command(
            ins.sudo_client, region_id, "2. manual switchover")
        SSH_API.send_command(ins.sudo_client, "2", "Configuration succeeded")
        logger.info("Config dr switch mode successfully.")

    @classmethod
    def get_server_node(
            cls, all_hosts, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        for host in all_hosts:
            logger.info(f"Start query listening ip on {host}.")
            ins = cls.get_instance(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
            find_tomcat_cmd = \
                "ps -ef | grep Tomcat | grep BCManager &>/dev/null;" \
                "echo CMD_RESULT=$?"
            find_tomcat_result = SSH_API.exec_command_return_list(
                ins.sudo_client, find_tomcat_cmd)
            if "CMD_RESULT=0" not in str(find_tomcat_result):
                logger.info(f"Node {host} is not active.")
                continue
            cat_install_file_cmd = \
                f"cat {Path.RD_INSTALLED_PATH} | grep '<floatip>' | " \
                f"awk -rF'<|>' '{{print $3}}'"
            cat_install_file_result = SSH_API.exec_command_return_list(
                ins.sudo_client, cat_install_file_cmd)
            logger.debug(
                f"Get float ip from {Path.RD_INSTALLED_PATH} return "
                f"{cat_install_file_result}.")
            if not cat_install_file_result or check_value_null(
                    cat_install_file_result[0]) or not \
                    check_param_ip(cat_install_file_result[0]):
                logger.info(
                    f"Get listen ip from {Path.RD_INSTALLED_PATH} failed, "
                    f"start search it from {Path.NGINX_SERVER_CONF_PATH}.")
                cat_server_conf_cmd = \
                    f"cat {Path.NGINX_SERVER_CONF_PATH} | " \
                    "grep -E 'listen.*9443' | " \
                    "awk -F' ' '{print $2}' | awk -F':' '{print $1}'"
                cat_server_xml_file_result = SSH_API.exec_command_return_list(
                    ins.sudo_client, cat_server_conf_cmd)
                logger.debug(
                    f"Get float ip from {Path.NGINX_SERVER_CONF_PATH} return "
                    f"{cat_server_xml_file_result}.")
                # 从CMD_RESULT=x LISTEN_IP=x.x.x.x=中解析服务监听IP
                if not cat_server_xml_file_result or check_value_null(
                        cat_server_xml_file_result[0]) or not \
                        check_param_ip(cat_server_xml_file_result[0]):
                    # 从nginx中获取服务ip失败，继续尝试从tomcat的配置中获取
                    server_ip = cls._get_float_ip_from_tomcat(ins.sudo_client)
                    return host, server_ip
                return host, cat_server_xml_file_result[0]
            return host, cat_install_file_result[0]

    @classmethod
    def _get_float_ip_from_tomcat(cls, sudo_client):
        # 从nginx中获取服务ip失败，继续尝试从tomcat的配置中获取
        logger.info(
            f"Get listen ip from {Path.NGINX_SERVER_CONF_PATH} failed, "
            f"start search it from {Path.TOMCAT_SERVER_XML_PATH}.")
        cat_server_xml_cmd = \
            f"cat {Path.TOMCAT_SERVER_XML_PATH} | " \
            "grep keystorePass | " \
            "grep -w address | awk -rF 'address=\"|\"/>' '{print $2}'"
        cat_server_xml_file_result = SSH_API.exec_command_return_list(
            sudo_client, cat_server_xml_cmd)
        logger.debug(
            f"Get float ip from {Path.TOMCAT_SERVER_XML_PATH} return "
            f"{cat_server_xml_file_result}.")
        # 从CMD_RESULT=x LISTEN_IP=x.x.x.x=中解析服务监听IP
        if not cat_server_xml_file_result or check_value_null(
                cat_server_xml_file_result[0]) or not \
                check_param_ip(cat_server_xml_file_result[0]):
            raise Exception("Get server node return None.")
        return cat_server_xml_file_result[0]

    def config_active_standby(self, s_role, local_ip, peer_ip):
        cmd = f"export localRole={s_role} IS_ACTIVE_STANDBY_AUTO=true " \
              f"local_IP={local_ip} peer_IP={peer_ip} && sudo -u ICUser -s sh " \
              f"{Path.BCM_SCRIPTS_PATH}configSystem.sh && echo CMD_RESULT=$?"
        result = SSH_API.send_command(
            self.sudo_client, cmd, "Config Active-Standby successfully", 300)
        if "CMD_RESULT=0" not in str(result):
            logger.error(f"Config active-standby mode in {self.host} failed.")
            raise Exception(
                f"Config active-standby mode in {self.host} failed.")
        logger.info(f"Config active-standby mode in {self.host} successfully.")

    def config_watch_dog(self):
        if self._check_watch_dog():
            logger.info(f"Watchdog has already config in {self.host}.")
            return
        logger.info(f"Start config watchdog in {self.host}.")
        script_file = os.path.join(os.path.dirname(
            os.path.dirname(__file__)), "shell_tools", "config_watchdog.sh")
        SSH_API.put_file(
            self.host, self.ssh_user, self.ssh_pwd, script_file, Path.NODE_TMP_PATH, 22)
        cmd = f"dos2unix {Path.NODE_TMP_PATH}/config_watchdog.sh && " \
              f"sh {Path.NODE_TMP_PATH}/config_watchdog.sh;rm -f {Path.NODE_TMP_PATH}/config_watchdog.sh"
        SSH_API.send_command(
            self.sudo_client, cmd, "config watchdog successfully")
        # 重启虚拟机
        try:
            SSH_API.send_command(self.sudo_client, "reboot", timeout=20)
        except Exception as err:
            logger.debug(f"The result of reboot vm is {err}.")

    def _check_watch_dog(self):
        logger.info(f"Start check watchdog in {self.host}.")
        # 如果已经设置了，表示该虚拟机已经增加了看门狗
        cmd = "cat /etc/fstab | grep errors=panic && echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(self.sudo_client, cmd)
        return "CMD_RESULT=0" in result

    def do_confirm_tool_env_config(self):
        local_file = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        remote_file = f"{Path.DR_USER_PATH}/env.ini"
        result = SSH_API.exec_command_return_list(
            self.sudo_client, f"[ -f '{remote_file}' ];echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in result:
            logger.info(f"Remote file {remote_file} not exists.")
            return
        remote_contents = SSH_API.exec_command_return_list(
            self.sudo_client, f"cat {remote_file}")
        finally_content = remote_contents[0]
        for index, content in enumerate(remote_contents):
            if index == 0:
                continue
            if content.startswith("[") and content.endswith("]"):
                finally_content += f"\n\n{content}"
            else:
                finally_content += f"\n{content}"
        with FILE_API.open_file_manager(local_file) as file:
            local_contents = file.read()
        if finally_content == local_contents:
            logger.info("The file content is the same.")
            return
        with FILE_API.open_file_manager(local_file, "w") as file:
            file.writelines(finally_content)
        logger.info("Do confirm env config success.")

    @classmethod
    def sync_env_config_to_remote(cls, host, ssh_user, ssh_pwd, config_path):
        try:
            SSH_API.put_file(
                host, ssh_user, ssh_pwd, config_path, Path.DR_USER_PATH)
        except Exception as exception:
            logger.error(f"Put file to {host} failed: {exception}")
            raise exception
        logger.info(f"Put file to {host} success.")

    @classmethod
    def execute_bcm_service(cls, project_id, pod_id, is_start: bool):
        """
        批量停止/启动 eReplication服务
        :param project_id:
        :param pod_id:
        :param is_start:
        :return:
        """
        nodes = Nodes(project_id, pod_id)
        operator_type = "start" if is_start else "stop"
        logger.info(f"Start to {operator_type} bcm service on host")
        exec_func = cls.start_bcm_service if is_start else cls.stop_bcm_service
        _, funcs = cls._get_threads_func_args(
            nodes.all_hosts, Auth('', nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                  nodes.sudo_pwd), exec_func)
        ExcThread.exec_func_in_thread(funcs)

    @classmethod
    def stop_bcm_service(cls, ssh_client):
        stop_cmd = f"echo y|sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh"
        stop_result = SSH_API.exec_command_return_list(ssh_client, stop_cmd, timeout=120)
        if "successfully" not in str(stop_result):
            logger.error(
                f"Stop bcm service failed, cmd execute return {stop_result}")
            raise Exception("Stop bcm service failed.")
        logger.info("Stop bcm service success on host.")

    @classmethod
    def start_bcm_service(cls, ssh_client):
        start_cmd = f"echo y|sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}startSystem.sh"
        start_result = SSH_API.exec_command_return_list(ssh_client, start_cmd, timeout=400)
        if "completely" not in str(start_result) \
                and "System is running" not in str(start_result):
            logger.error(
                f"Start bcm service failed, cmd execute return {start_result}")
            raise Exception("Start bcm service failed.")
        logger.info("Start bcm service success on host.")

    @classmethod
    def sync_dr_system_data_to_standby(cls, project_id, pod_id):
        nodes = Nodes(project_id, pod_id)
        params = Params(project_id, pod_id)
        ins = cls.get_instance(Auth(nodes.primary_ip, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd))
        # 备份主节点数据
        data_back_file = cls.backup_dr_system_data(ins.sudo_client, params.data_encrypt_pwd)
        cls._mkdir_local_temp_path()
        cls._download_local_from_primary_tmp(nodes.primary_ip, nodes.ssh_user, nodes.ssh_pwd, data_back_file)
        # 从主节点拷贝数据文件到备节点
        thread_name = threading.current_thread().name
        funcs = list()
        for dest_host in nodes.hosts:
            auth_provider = Auth(dest_host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
            func = (cls._restore_system_data_to_dest_host, thread_name,
                    (auth_provider, data_back_file, params.data_encrypt_pwd), {})
            funcs.append(func)
        # 导入数据
        ExcThread.exec_func_in_thread(funcs)
        cls._clear_local_temp_path()
        logger.info("Sync DR system data to standby success.")

    @classmethod
    def copy_bcm_files_to_standby(cls, project_id, pod_id):
        logger.info(f"Start copy certs and config file to standby")
        nodes = Nodes(project_id, pod_id)
        ins = cls.get_instance(Auth(nodes.primary_ip, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd))
        # 备份主节点内容
        cls._backup_dr_system_cert(ins.sudo_client)
        cls._backup_dr_system_conf(ins.sudo_client)
        # 压缩文件
        cls._compress_backup_file(ins.sudo_client)
        cls._mkdir_local_temp_path()
        cls._download_local_from_primary_tmp(nodes.primary_ip, nodes.ssh_user, nodes.ssh_pwd, Path.TMP_BACKUP_FILE_NAME)
        # 从主节点拷贝证书文件到备节点
        funcs = list()
        thread_name = threading.current_thread().name
        auth_provider = Auth(nodes.primary_ip, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
        for dest_host in nodes.hosts:
            if OS_API.check_result_in_os(dest_host, nodes.sudo_user, nodes.sudo_pwd):
                logger.info(f"RDInstalled.xml find in {dest_host}, no need copy.")
                continue
            func = (cls._copy_certs_and_conf_to_dest_host, thread_name, (auth_provider, dest_host), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        cls._clear_local_temp_path()
        logger.info(f"Sync certs and config file to standby success.")

    @classmethod
    def _compress_backup_file(cls, sudo_client):
        """压缩主节点证书文件和配置文件，由于更换了put_file，只能传单个文件，故把所有文件压缩成一个传到备region节点
        :param sudo_client:
        :return:
        """
        logger.info(f"Start compress certs and config file")
        cert_path = os.path.basename(Path.BCM_CERT_DEST_PATH)
        conf_file_path = os.path.basename(Path.DR_TEMP_CONF_DEST_PATH)
        SSH_API.send_command(sudo_client, f"cd {Path.DR_USER_PATH}")
        # 备端创建拷贝成功标志文件(对应eReplication安装脚本内的判断标志)
        SSH_API.send_command(sudo_client, f"touch {Path.SYNC_FILE_FLAG}")
        cmd = f"tar -zcvf {Path.TMP_BACKUP_FILE_NAME} '{cert_path}/' '{conf_file_path}/' {Path.SYNC_FILE_FLAG}"
        SSH_API.send_command(sudo_client, cmd)
        SSH_API.send_command(sudo_client, f"chown -h DRManager:LEGO {Path.TMP_BACKUP_FILE_NAME}")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.BCM_CERT_DEST_PATH}")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.DR_TEMP_CONF_DEST_PATH}")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.SYNC_FILE_FLAG}")
        logger.info(f"Compress certs and config file success")

    @classmethod
    def backup_dr_system_data(cls, ssh_client, encrypt_pwd, is_upgrade=False):
        """
        备份指定节点容灾服务数据
        :param ssh_client:
        :param encrypt_pwd:
        :param is_upgrade:
        :return:
        """
        logger.info(f"Start get DR config data file.")
        result = SSH_API.exec_command_return_list(
            ssh_client, f"echo y|sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh")
        if "successfully" not in str(result):
            logger.error(f"Shutdown service failed: {result}.")
            raise Exception(f"Shutdown service failed: {result}.")
        current_version = cls._get_bcm_version_from_env()
        if is_upgrade and current_version[0:3] < "8.3":
            back_cmd = f"sh {Path.BCM_SCRIPTS_PATH}backupData.sh"
        else:
            back_cmd = f"sh {Path.BCM_ROOT_SCRIPTS_PATH}backupData.sh"
        SSH_API.send_command(
            ssh_client, back_cmd, 'Please enter the password')
        dest_str = "Export system configuration data successfully"
        cmd_out = SSH_API.send_command(ssh_client, encrypt_pwd, dest_str, 180)
        data_back_file = ""
        for result in cmd_out:
            if dest_str in result and '/' in result:
                result = result.replace('\n', '')
                data_back_file = result[result.index('/'):]
        if check_value_null(data_back_file):
            logger.error(f"Get database config file failed: {cmd_out}.")
            raise Exception(f"Get database config file failed: {cmd_out}.")
        logger.info(f"Get DR config data file successfully: {data_back_file}.")
        return os.path.basename(data_back_file)

    @classmethod
    def _get_bcm_version_from_env(cls):
        config_tool = cls.get_env_config()
        return config_tool.get_value_by_key_and_sub_key(Component.REPLICATION, 'current_version')

    @classmethod
    def get_env_config(cls):
        config_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        return FILE_API(config_path)

    @classmethod
    def _clear_local_temp_path(cls):
        """删除本地临时目录"""
        if os.path.exists(Path.LOCAL_TEMP_PATH):
            shutil.rmtree(Path.LOCAL_TEMP_PATH)
        logger.info("Clear local temporary file success.")

    @classmethod
    def _mkdir_local_temp_path(cls):
        """创建本地临时目录"""
        if not os.path.exists(Path.LOCAL_TEMP_PATH):
            os.mkdir(Path.LOCAL_TEMP_PATH)
        logger.info("Create local temporary path success.")

    @classmethod
    def _backup_dr_system_cert(cls, ssh_client):
        """
        备份指定节点证书文件
        :param ssh_client:
        :return:
        """
        SSH_API.send_command(
            ssh_client,
            rf"\cp -rf {Path.BCM_CERT_SOURCE_PATH} "
            rf"{Path.DR_USER_PATH} && echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            ssh_client, f"chown -Rh DRManager:LEGO {Path.BCM_CERT_DEST_PATH} && "
                        f"echo CMD_RESULT=$?", "CMD_RESULT=0")
        logger.info("Backup DR system keystore file success.")

    @classmethod
    def _copy_system_data_to_dest_host(
            cls, auth_provider, dest_host, data_back_file):
        ssh_user = auth_provider.ssh_user
        ssh_pwd = auth_provider.ssh_pwd
        try:
            # 上传文件到目标主机
            file_name = os.path.basename(data_back_file)
            local_back_path = f"{Path.LOCAL_TEMP_PATH}/{file_name}"
            SSH_API.put_file(dest_host, ssh_user, ssh_pwd, local_back_path, Path.DR_USER_PATH)
            logger.info(f"Copy DR system data to dest host: {dest_host} success.")
        except Exception as err:
            err_msg = f"Copy DR system data to dest host: {dest_host} failed."
            logger.error(err_msg)
            raise Exception(err_msg) from err
        return data_back_file

    @classmethod
    def _copy_certs_and_conf_to_dest_host(
            cls, auth_provider, dest_host):
        logger.info(f"Begin to copy certs and configs to {dest_host}.")
        ssh_user = auth_provider.ssh_user
        ssh_pwd = auth_provider.ssh_pwd
        ssh_client = None
        try:
            # 上传文件到目标主机
            local_back_path = f"{Path.LOCAL_TEMP_PATH}/{Path.TMP_BACKUP_FILE_NAME}"
            SSH_API.put_file(dest_host, ssh_user, ssh_pwd, local_back_path, Path.DR_USER_PATH)
            ssh_client = SSH_API.get_ssh_client(SSHClientInfo(dest_host, ssh_user, ssh_pwd))
            # 解压文件
            SSH_API.execute_command(ssh_client, f"cd {Path.DR_USER_PATH}")
            SSH_API.execute_command(ssh_client, f"tar -zxvf {Path.TMP_BACKUP_FILE_NAME}")
            SSH_API.execute_command(ssh_client, rf"\rm -rf {Path.DR_USER_PATH}/{Path.TMP_BACKUP_FILE_NAME}")
            logger.info(f"Copy certs and configs to {dest_host} success.")
        except Exception as err:
            err_msg = f"Copy certs and configs to {dest_host} failed."
            logger.error(err_msg)
            raise Exception(err_msg) from err
        finally:
            SSH_API.close_ssh(ssh_client)

    @classmethod
    def _restore_system_data_to_dest_host(cls, auth_provider, file_name, encrypt_pwd):
        dest_host = auth_provider.host
        logger.info(f"Start restore DR system data to dest host: {dest_host}.")
        # 上传文件到目标主机
        local_back_path = f"{Path.LOCAL_TEMP_PATH}/{file_name}"
        SSH_API.put_file(dest_host, auth_provider.ssh_user, auth_provider.ssh_pwd, local_back_path, Path.DR_USER_PATH)
        ins = cls.get_instance(auth_provider)
        cmd = f"chmod 640 {Path.DR_USER_PATH}/{file_name}"
        SSH_API.send_command(ins.sudo_client, cmd)

        # 加固 导入数据前停止一把服务
        cls.stop_bcm_service(ins.sudo_client)
        restore_data_cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh import.sh"
        # 恢复备份数据文件
        dest_data_file = f"{Path.DR_USER_PATH}/{file_name}"
        SSH_API.send_command(ins.sudo_client, restore_data_cmd, 'do you want to continue')
        SSH_API.send_command(ins.sudo_client, "y", 'Please input the restore file')
        SSH_API.send_command(ins.sudo_client, dest_data_file, "encrypt password")
        SSH_API.send_command(ins.sudo_client, encrypt_pwd, "Restore successfully", timeout=300)
        logger.info(f"Restore DR system data to dest host: {dest_host} success.")

    @classmethod
    def clear_environment(cls, project_id, pod_id, hosts: list):
        funcs = list()
        thread_name = threading.current_thread().name
        for host in hosts:
            func = (cls._clear_environment, thread_name,
                    (project_id, pod_id, host), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    @classmethod
    def _clear_environment(cls, project_id, pod_id, host):
        nodes = Nodes(project_id, pod_id)
        ins = cls.get_instance(Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd))
        clear_cmd = \
            f"rm -rf {Path.BCM_DATA_BACKUP_PATH} " \
            f"{Path.BCM_CERT_DEST_PATH} && echo CMD_RESULT=$?"
        SSH_API.send_command(ins.sudo_client, clear_cmd, "CMD_RESULT=0")
        logger.info(f"Clear environment on {host} success.")

    def get_current_version(self):
        """获取当前eReplication的版本号信息

        :return:
        """
        version = ""
        results = SSH_API.exec_command_return_list(
            self.sudo_client,
            f"echo version=`cat {Path.LEGO_DEPLOY_FILE_PATH} | "
            "grep version= | awk '{print $NF}'`")
        logger.info(f"Get current version on {self.host} return {results}.")
        if "version" not in str(results):
            logger.error(f"Get current version failed: {results}.")
            raise Exception(f"Get current version failed: {results}.")
        for res in results:
            if 'version=' not in res or "echo version" in res:
                continue
            version = res.replace('\n', '').replace(' ', '').split('=')[1]
            break
        if not version:
            logger.error(f"Get current version failed.")
            raise Exception(f"Get current version failed.")
        return version

    @classmethod
    def get_bcm_version(cls, project_id, pod_id):
        if cls._is_installed_bcm_service(project_id):
            pkg_version = PKG_API.get_pkg_version()
            logger.debug(f"Get pkg version return {pkg_version}.")
            return pkg_version
        else:
            nodes = Nodes(project_id, pod_id)
            for host in nodes.all_hosts:
                dr_api = cls.get_instance(Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd))
                system_version = dr_api.get_current_version()
                logger.debug(f"Get system version return {system_version}.")
                return system_version

    @staticmethod
    def _is_installed_bcm_service(project_id):
        condition = Condition(project_id)
        if not condition.is_primary and not (condition.is_sub and condition.is_global_con_dr):
            return False
        if not condition.install_dr_service:
            return False
        return not is_legacy_hcsu_scene(project_id)

    def get_patch_version(self):
        """获取当前eReplication的补丁版本号信息

        :return:
        """

        cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh showSystemInfo.sh | " \
              "grep 'Patch Version' | awk -F' ' '{print $NF}'"
        results = SSH_API.exec_command_return_list(self.sudo_client, cmd)
        logger.info(f"Get patch version on {self.host} return {results}.")
        if not results:
            return ''
        return results[0]

    @classmethod
    def get_instance(cls, *args, **kwargs):
        return cls(*args, **kwargs)

    @classmethod
    def get_cloud_dr_service_installed_list(
            cls, project_id, pod_id, region_id):
        """
        查询当前环境的Endpoint
        通过Endpoint获取当前环境安装的容灾服务
        """
        csdr_existed, csha_existed, vha_existed = (
            "False", "False", "False")
        if is_legacy_hcsu_scene(project_id):
            endpoints_zh = MO_API(project_id, pod_id).get_endpoints(region_id)
        else:
            endpoints_zh = cls.get_endpoints(project_id, pod_id, region_id)
        endpoints_zh_str = str(endpoints_zh)
        if Component.CSDR_U in endpoints_zh_str or Condition(
                project_id).is_hcs_global:
            csdr_existed = "True"
        if Component.CSHA_U in endpoints_zh_str:
            csha_existed = "True"
        if Component.VHA_U in endpoints_zh_str:
            vha_existed = "True"
        return [{"key": Component.CSDR_U, "value": csdr_existed},
                {"key": Component.CSHA_U, "value": csha_existed},
                {"key": Component.VHA_U, "value": vha_existed}]

    @classmethod
    def get_endpoints(cls, project_id, pod_id, region_id):
        """
        获取Endpoint列表
        """
        er_ip, er_port = MO_API(project_id, pod_id).get_mo_er_ip_and_port()
        base_url = cls._get_http_url(er_ip, er_port)
        url = f"{base_url}/silvan/rest/v1.0/endpoints?start=0&limit=0&" \
              f"region={region_id}"
        endpoints_ret = cls._auth_get(url, headers={"X-Language": "zh-cn"})
        endpoints_info = json.loads(endpoints_ret.text)
        return endpoints_info

    @classmethod
    def _get_http_url(cls, mo_ip, mo_port):
        if IP(mo_ip).version() == 6:
            mo_ip = f"[{mo_ip}]"
        return f"https://{mo_ip}:{mo_port}"

    @classmethod
    def _auth_get(cls, *vargs, **kwargs):
        kwargs = cls._add_auth_headers(kwargs)
        response = requests.get(*vargs, **kwargs)
        cls._check_response(response)
        return response

    @classmethod
    def _add_auth_headers(cls, kwargs):
        header_data = {
            'Content-Type': 'application/json', 'Accept': 'application/json;charset=utf8',
            'User-Agent': 'python-cinderclient'
        }
        kwargs["headers"] = dict(kwargs.get("headers", {}), **header_data)
        kwargs["verify"] = False
        return kwargs

    @classmethod
    def _check_response(cls, response):
        """
        校验请求响应状态码
        """
        if response.status_code != 200:
            error = f"Get endpoint request failed: {response.status_code}."
            logger.error(error)
            raise Exception(error)
        logger.info("Get endpoint request return true.")

    @classmethod
    def register_cmdb(cls, project_id, pod_id, region_id, cloud_service_info):
        logger.info("Start to edit extendInfos")
        mo_api = MO_API(project_id, pod_id)
        logger.info(f"cloud_service_info is {cloud_service_info}")
        mo_api.set_cloud_service_info(region_id, cloud_service_info)
        logger.info("Set cloud service info to CMDB successfully")
        mo_api.refresh_dr_params()

    @classmethod
    def set_iam_service_name_info(cls, project_id, pod_id):
        logger.info("Rewrite server config file start.")
        service_lst = get_install_services(project_id)
        iam_accounts = \
            [f"{service.lower()}_service" for service in service_lst]
        nodes = Nodes(project_id, pod_id)
        _, funcs = cls._get_threads_func_args(
            nodes.all_hosts, Auth('', nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                  nodes.sudo_pwd), cls.do_set_iam_service_name_info, iam_accounts)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Set iam service name to lego.properties success.")

    @classmethod
    def do_set_iam_service_name_info(cls, ssh_client, accounts):
        result = SSH_API.exec_command_return_list(
            ssh_client, f"cat {Path.LEGO_FILE_PATH} | grep 'IAM.service.name'")
        service_iam_accounts = []
        if result:
            service_iam_accounts = result[0].strip().split("=")[-1].split(",")
        for account in accounts:
            if account in service_iam_accounts:
                continue
            service_iam_accounts.append(account)
        iam_accounts = ",".join(service_iam_accounts)
        SSH_API.send_command(
            ssh_client,
            f"sudo -u ICUser -s sed -i '/IAM.service.name.*/d' {Path.LEGO_FILE_PATH} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            ssh_client,
            r"sudo -u ICUser -s sed -i '/IAM.service.password.*/i"
            rf"\IAM.service.name={iam_accounts}' "
            rf"{Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?", "CMD_RESULT=0")

    @classmethod
    def set_hcs_domain_sys_config(cls, project_id, pod_id):
        logger.info("Init hcs domain sys configuration process.")
        global_domain_name = MO_API(project_id, pod_id).global_domain()
        check_param_domain_name(global_domain_name)
        nodes = Nodes(project_id, pod_id)
        _, funcs = cls._get_threads_func_args(
            nodes.all_hosts, Auth('', nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                  nodes.sudo_pwd), cls.do_set_hcs_domain_sys_config,
            global_domain_name)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Set hcs domain to lego.properties success.")

    @classmethod
    def do_set_hcs_domain_sys_config(cls, ssh_client, global_domain_name):
        result = SSH_API.exec_command_return_list(
            ssh_client,
            f"sudo -u ICUser -s cat {Path.LEGO_FILE_PATH} | grep 'config.hcs.domain';"
            "echo CMD_RESULT=$?")
        if "CMD_RESULT=0" in str(result) and \
                global_domain_name in str(result):
            logger.info("Config hcs domain already exists in lego.properties.")
            return
        write_cmd = \
            r"sudo -u ICUser -s sed -i '/fusioncloud.version/a\config.hcs.domain" \
            rf"={global_domain_name}' {Path.LEGO_FILE_PATH};echo CMD_RESULT=$?"
        SSH_API.send_command(ssh_client, write_cmd, "CMD_RESULT=0")

    @classmethod
    def set_multi_cloud_adapter_switch_status(cls, project_id, pod_id):
        thread_name = threading.current_thread().name
        funcs = list()
        nodes = Nodes(project_id, pod_id)
        for host in nodes.all_hosts:
            logger.info(f"Do set multi cloud adapter switch to {host} in thread.")
            sudo_client = SSH_API.get_sudo_ssh_client(
                host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
            func = (cls._do_set_multi_cloud_adapter_switch, thread_name, (sudo_client,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    @staticmethod
    def _do_set_multi_cloud_adapter_switch(sudo_client):
        logger.info("Do execute set multi cloud adapter switch task.")
        clear_cmd = "sudo -u ICUser -s sed -i -e '/#是否是多云合一场景.*/d' -e '/multi.cloud.adapter.switch.*/d'" \
                    f" {Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?"
        SSH_API.send_command(sudo_client, clear_cmd, "CMD_RESULT=0")
        rewrite_cmd = r"sudo -u ICUser -s sed -i '/#是否支持console.*/i\#是否是多云合一场景\n" \
                      rf"multi.cloud.adapter.switch=true' {Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?"
        SSH_API.send_command(sudo_client, rewrite_cmd, "CMD_RESULT=0")
        logger.info("Set multi cloud adapter switch success.")

    @classmethod
    def set_sdr_sys_config(cls, project_id, pod_id, service_type):
        service_lst = get_install_services(project_id)
        sdr_services = "&".join([service.lower() for service in service_lst])
        sdr_float_ip = Params(project_id, pod_id).sdr_float_ip
        global_domain_name = MO_API(project_id, pod_id).global_domain()
        region_info = Params(project_id, pod_id).project_region_id
        check_param_domain_name(global_domain_name)
        sdr_info = f"{region_info}@{sdr_float_ip}"
        exists_sdr_infos = cls.get_exists_sdr_info(project_id, pod_id)
        nodes = Nodes(project_id, pod_id)
        sdr_infos = [
            f"{_sdr_info}&{service_type}" if
            sdr_info in _sdr_info and service_type
            not in _sdr_info else _sdr_info for _sdr_info
            in exists_sdr_infos]
        if sdr_info not in ",".join(sdr_infos):
            sdr_infos.append(f"{sdr_info}#{sdr_services}")
        sdr_infos_str = ",".join(sdr_infos)
        _, funcs = cls._get_threads_func_args(
            nodes.all_hosts, Auth('', nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                  nodes.sudo_pwd), cls.do_set_sdr_sys_config, sdr_infos_str,
            global_domain_name)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Set sdr sys config to lego.properties success.")

    @classmethod
    def do_set_sdr_sys_config(cls, ssh_client, sdr_infos, global_domain_name):
        SSH_API.send_command(
            ssh_client,
            "sudo -u ICUser -s sed -i -e '/#SDR服务地址.*/d' -e '/sdr.service.address.*/d' -e "
            f"'/config.dr.domain.*/d' {Path.LEGO_FILE_PATH} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            ssh_client,
            rf"sudo -u ICUser -s sed -i '/#是否支持console.*/i\#SDR服务地址\n"
            rf"sdr.service.address={sdr_infos}\n"
            rf"config.dr.domain=cdrs.{global_domain_name}' "
            rf"{Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?", "CMD_RESULT=0")

    @classmethod
    def get_exists_sdr_info(cls, project_id, pod_id):
        nodes = Nodes(project_id, pod_id)
        host, listen_ip = cls.get_server_node(
            nodes.all_hosts, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
            nodes.sudo_pwd)
        auth_provider = Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user, nodes.sudo_pwd)
        dr_api = cls(auth_provider)
        result = SSH_API.exec_command_return_list(
            dr_api.sudo_client,
            f"sudo -u ICUser -s cat {Path.LEGO_FILE_PATH} | grep 'sdr.service.address'")
        if "sdr.service.address" in str(result):
            sdr_infos = result[0].strip().split("=")[-1].split(",")
            return sdr_infos
        return []

    @classmethod
    def config_cloud_init(cls, project_id, pod_id):
        ssh_client = None
        try:
            ssh_client = cls._do_config_cloud_init(pod_id, project_id)
        except Exception as err:
            logger.error(f"Config cloud init failed: {err}")
            raise err
        finally:
            SSH_API.close_ssh(ssh_client)

    @classmethod
    def _do_config_cloud_init(cls, pod_id, project_id):
        logger.info("Start config cloud init.")
        params = Params(project_id, pod_id)
        env_os_password = params.env_os_pwd
        reverse_ip = params.openstack_reverse_ip
        root_pwd = params.openstack_sudo_pwd
        fsp_pwd = params.openstack_fsp_pwd
        ssh_client = SSH_API.get_sudo_ssh_client(Auth(reverse_ip, "fsp", fsp_pwd, "root", root_pwd))
        SSH_API.send_command(
            ssh_client, "source set_env", "please choose:")
        SSH_API.send_command(ssh_client, '1', "Enter OS_PASSWORD=")
        SSH_API.send_command(ssh_client, env_os_password)
        cmd = "cps template-params-update --service nova nova-client " \
              "--parameter bcm_primary_vm=True;cps commit"
        SSH_API.send_command(ssh_client, cmd)
        logger.info("Config cloud init successfully.")
        return ssh_client

    @classmethod
    def register_api(cls, project_id, pod_id, service_type):
        """不带internal是旧版本的，防止有人用，继续注册"""
        logger.info("Start to register api.")
        global_domain_name = MO_API(project_id, pod_id).global_domain()
        param = Params(project_id, pod_id)
        domain = f"{service_type}.{global_domain_name}"
        address = f"{domain}:{param.server_port}"
        region_id = param.project_region_id
        cls._register_api(pod_id, ApiRegisterInfo(service_type, domain, address, "dr_apig.yml", region_id))
        cls.register_internal_api(project_id, pod_id, service_type, region_id, param.server_port)
        logger.info("Register api successful.")

    @classmethod
    def register_internal_api(cls, project_id, pod_id, service_type, region_id, server_port):
        """注册api，规范格式:{服务名}-internal.{region_id}.{global_domain}"""
        logger.info("Start to register internal api.")
        global_domain_name = MO_API(project_id, pod_id).global_domain()
        address = f"{service_type}.{global_domain_name}:{server_port}"
        internal_service_type = f"{service_type}-internal"
        internal_domain = f"{internal_service_type}.{region_id}.{global_domain_name}"
        api_register_info = ApiRegisterInfo(internal_service_type, internal_domain,
                                            address, "dr_internal_apig.yml", region_id)
        cls._register_api(pod_id, api_register_info)
        logger.info("Register internal api successful.")

    @classmethod
    def _register_api(cls, pod_id, api_info):
        logger.info(f"Start to execute register api of {api_info.api_file_name}.")
        config_info = FILE_API.get_config_content(os.path.join(
            os.path.join(os.path.dirname(os.path.dirname(__file__)), "conf"),
            api_info.api_file_name)) % (api_info.service_type, api_info.domain, api_info.address)
        # 注册前先执行一次删除, 确保信息清空
        try:
            logger.info(f"Start delete old {api_info.service_type} client info.")
            APIGApi().register_api_by_content(pod_id, api_info.service_type, config_info, "Remove Api")
        except Exception as err:
            # 移除出错不抛异常
            logger.error(f"Remove client failed, error={str(err)}")
        logger.info(f"Start register {api_info.service_type} client info.")
        # 开始注册API
        result_as = APIGApi().register_api_by_content(pod_id, api_info.service_type, config_info, "Register Api")
        if not result_as:
            raise Exception("Execute_dmk_deployment return false")
        logger.info(f"Execute register api of {api_info.api_file_name} successful.")

    @classmethod
    def register_domain(cls, project_id, pod_id, service_type):
        """注册我们服务域名到dns，规范格式:{服务名}-internal.{region_id}.{global_domain}"""
        logger.info(f"Start to register domain info for {service_type}.")
        service_ip = Nodes(project_id, pod_id).service_ip
        cls._register_domain(project_id, pod_id, service_type, service_ip)
        param = Params(project_id, pod_id)
        cls.register_domain_with_apig(project_id, pod_id, service_type,
                                      param.project_region_id, param.apig_internal_float_ip)
        logger.info(f"Register domain info for {service_type} success.")

    @classmethod
    def register_domain_with_apig(cls, project_id, pod_id, service_type, region_id, apig_internal_float_ip):
        """注册服务域名映射到apig浮动ip，能在apig找到我们服务"""
        logger.info(f"Start to register domain info to apig.")
        cls._register_domain(project_id, pod_id, f"{service_type}-internal.{region_id}", apig_internal_float_ip)
        logger.info(f"Register domain info to apig successful.")

    @classmethod
    def _register_domain(cls, project_id, pod_id, service_type, service_ip):
        logger.info(f"Start to execute register domain info for {service_type}.")
        dns_type = "AAAA" if ":" in service_ip else "A"
        dns_domain = MO_API(project_id, pod_id).global_domain()
        rrsets = [{'name': 'cdrs', 'type': dns_type, 'records': [service_ip]}]
        if not Condition(project_id).is_hcs_global:
            rrsets.append({'name': service_type, 'type': dns_type,
                           'records': [service_ip]})
        DNSApi().add_rrsets(dns_domain, rrsets, DNSNodeType.OM, pod_id)
        logger.info(f"Execute register domain info for {service_type} success.")

    @classmethod
    def register_elb_domain(cls, project_id, pod_id):
        logger.info("Start register elb dns domain process.")
        params = Params(project_id, pod_id)
        region_id = params.project_region_id
        dns_domain = params.external_global_domain
        apig_float_ip = Params(project_id, pod_id).apig_float_ip
        dns_type = "AAAA" if ":" in apig_float_ip else "A"
        rrsets = [{'name': f'vpc.{region_id}', 'type': dns_type,
                   'records': [apig_float_ip]}]
        DNSApi().add_rrsets(dns_domain, rrsets, DNSNodeType.OM, pod_id)
        logger.info("Register elb dns domain success.")

    @classmethod
    def get_dr_request_api(cls, project_id, pod_id):
        nodes = Nodes(project_id, pod_id)
        if is_legacy_hcsu_scene(project_id):
            physical, float_ip = cls.get_server_node(
                nodes.all_hosts, nodes.ssh_user, nodes.ssh_pwd,
                nodes.sudo_user, nodes.sudo_pwd)
        else:
            float_ip = nodes.service_ip
        return RequestApi(
            float_ip, nodes.service_name, nodes.service_pwd,
            nodes.service_port)

    @classmethod
    def _get_config_oc_data(cls, project_id, pod_id):
        logger.info("Get config oc data.")
        region_id = Params(project_id, pod_id).project_region_id
        mo_api = MO_API(project_id, pod_id)
        oc_host, oc_port = mo_api.get_oc_domain_info(region_id)
        user, pwd = mo_api.get_third_party_info()
        logger.info(f"Get oc info return: {oc_host}, {oc_port}, {user}.")
        body_data = {"domain": oc_host, "port": oc_port, "username": user,
                     "password": pwd}
        return body_data

    @classmethod
    def config_oc(cls, project_id, pod_id, rest_client):
        nodes = Nodes(project_id, pod_id)
        xaas_ips = nodes.float_ip if Condition(project_id). \
            global_con_dr_install_dr_service else nodes.service_ip
        logger.info(f"Start config oc white list {[xaas_ips]}.")
        logic_region = Params(project_id, pod_id).region_display_name
        # 添加白名单
        MO_API(project_id, pod_id).mo_util2.addXaasWhiteList(
            pod_id, "eReplication_", [xaas_ips], None, None, logic_region,
            restProtocol="1", snmpProtocol="",
            floatIp="", point="9443", productname="XAAS")
        body_data = cls._get_config_oc_data(project_id, pod_id)
        logger.info(f"Start config oc server in [{rest_client.node_ip}].")
        ex = None
        for count in range(0, 10):
            # 当前OC有个问题, 注册XAAS白名单过后,OC有个10分钟的轮巡时间,
            # 注册后, 如果马上调用对接操作,此时可能会报用户名或密码错误,
            # 因此,注册后采用重试机制进行对接,如果失败,则等待60s后再试一次,
            # 有个时间差问题, 所以此处重试10次
            logger.info(f"Start retry times: {count}.")
            q_state, q_body = rest_client.send_get_request(
                "ws/system/thirdservice/oc")
            if q_body and len(dict(q_body)) != 0 and dict(q_body).get(
                    "domain", None):
                logger.info("The oc already existed, no need to execute.")
                return
            sleep(60)
            state, body = rest_client.send_put_request(
                body_data, "ws/system/thirdservice/oc")
            if state == 200:
                logger.info(f"Config oc successfully in {rest_client.node_ip}.")
                return
            else:
                logger.error(f"Config oc failed: {body}")
                err_code = str(dict(body).get("errorCode", None))
                if err_code == "1073947691":
                    ex = HCCIException('663700')
                    continue
                if err_code == "1073947407":
                    raise HCCIException(
                        '663704', body_data.get("domain", None),
                        f"eReplication: {rest_client.node_ip}", "OC")
                raise HCCIException(
                    '663703', body_data.get("domain", None),
                    body_data.get("port", None), body, rest_client.node_ip)
        logger.error("The number of retries has reached the maximum.")
        raise ex

    @classmethod
    def _get_config_cts_data(cls, project_id, pod_id):
        logger.info("Get config cts data start.")
        sc_domain, port = MO_API(project_id, pod_id).get_sc_domain_info()
        logger.info(f"Get cts info return: {sc_domain}, {port}.")
        body_data = {"logType": "tenantlog", "isEnabled": True,
                     "protocalType": "CTS",
                     "protocalProps":
                         f"{{\"ip\":\"{sc_domain}\", \"port\":\"{port}\"}}"}
        return body_data

    @classmethod
    def config_cts(cls, project_id, pod_id, rest_client):
        body_data = cls._get_config_cts_data(project_id, pod_id)
        sc_domain = \
            json.loads(body_data.get("protocalProps")).get("ip")
        port = json.loads(body_data.get("protocalProps")).get("port")
        logger.info(f"Start config cts in {rest_client.node_ip}: {body_data}.")
        q_state, q_body = rest_client.send_get_request(
            "ws/system/action/querylognotify?logType=talentlog")
        if not cls.is_need_config(q_body):
            logger.info("The cts already existed, no need execute.")
            return
        state, body = rest_client.send_put_request(
            body_data, "ws/system/action/setlognotify")
        if state == 200:
            logger.info(f"Config cts successfully in {rest_client.node_ip}.")
            return
        else:
            logger.error(f"Config cts failed: {body}")
            err_code = str(dict(body).get("errorCode", None))
            if err_code == "1073947407":
                raise HCCIException(
                    '663704', f"CTS: {sc_domain}",
                    f"eReplication: {rest_client.node_ip}", "CTS")
            raise HCCIException(
                '663702', sc_domain, port, body, rest_client.node_ip)

    @staticmethod
    def is_need_config(q_body):
        if not q_body:
            return True
        if len(q_body) == 0:
            return True
        if len(dict(q_body[0])) == 0:
            return True
        return not dict(q_body[0]).get("isEnabled")

    @classmethod
    def _get_config_syslog_data(cls, project_id, pod_id):
        logger.info("Get config syslog data start.")
        mo_api = MO_API(project_id, pod_id)
        lvs_domian = mo_api.get_lvs_ip()
        tls_port = mo_api.get_tls_port()
        body_data = {"logType": "administratorlog", "isEnabled": True,
                     "protocalType": "syslog",
                     "protocalProps":
                         f"{{\"ip\":\"{lvs_domian}\", \"port\":\""
                         f"{tls_port}\", \"encryptionMode\":\"TLS_TCP\"}}"}
        return body_data

    @classmethod
    def config_syslog(cls, project_id, pod_id, rest_client):
        body_data = cls._get_config_syslog_data(project_id, pod_id)
        host_ip = json.loads(body_data.get("protocalProps")).get("ip")
        port = json.loads(body_data.get("protocalProps")).get("port")
        logger.info(f"Start config syslog in {rest_client.node_ip}: {host_ip}, {port}.")
        q_state, q_body = rest_client.send_get_request(
            "ws/system/action/querylognotify?logType=administratorlog")
        if not cls.is_need_config(q_body):
            logger.info("The syslog already existed, no need execute.")
            return
        body = dict()
        for _ in range(0, 5):
            state, body = rest_client.send_put_request(
                body_data, "ws/system/action/setlognotify")
            if state == 200:
                logger.info(f"Config syslog successfully in {rest_client.node_ip}.")
                return
            else:
                logger.info("Retry set syslog after 60 seconds")
                sleep(60)
        logger.error(f"Config syslog failed: {body}.")
        err_code = str(dict(body).get("errorCode", None))
        if err_code == "1073947407":
            raise HCCIException(
                '663704', f"syslog: {host_ip}",
                f"eReplication: {rest_client.node_ip}", "syslog")
        raise HCCIException('663701', host_ip, port, body, rest_client.node_ip)

    @classmethod
    def create_site(cls, rest_client):
        logger.info("Start create rd site.")
        server_id = cls._get_server_id_with_retry(rest_client)
        site_id = cls._check_site(rest_client, server_id)
        if site_id:
            logger.info("The site is already exists, no need to create.")
            return site_id, server_id

        site_id = cls._create_dr_site(rest_client, server_id)
        return site_id, server_id

    @classmethod
    def _check_site(cls, rest_client, server_id):
        body = cls._query_dr_sites_by_id(rest_client, server_id)
        for site_info in body:
            site_name = dict(site_info).get("name")
            site_id = dict(site_info).get("siteId")
            if not check_value_null(site_name) and site_name == "autoSite":
                return site_id
        logger.error(f"Check site return: {body}")
        return ''

    @classmethod
    def get_dest_device_id(cls, project_id, pod_id, rest_client, server_id):
        iam_address = "" if is_legacy_hcsu_scene(project_id) else Params(
            project_id, pod_id).iam_address
        body = cls._query_dr_sites_by_id(rest_client, server_id)
        for site_info in body:
            site_id = dict(site_info).get("siteId")
            body = cls._query_fusionsphere_info_by_site_id(rest_client, site_id)
            for fsp_info in body:
                address = dict(fsp_info).get("ipAddress")
                devsn = dict(fsp_info).get("deviceSn")
                # iam address参数只在安装部署场景下需要从外部输入，升级场景直接返回设备ID
                if not iam_address:
                    return devsn
                # 当前一个server上只能发现一套IAM, 因此， 如果address不为空,此时也直接返回
                if address == iam_address or not check_value_null(address):
                    logger.info(
                        f"Get dest device id successfully: {address}, {devsn}")
                    return devsn
        logger.error(f"Get dest device id failed: {server_id}, {iam_address}.")
        raise HCCIException('663706', rest_client.node_ip, iam_address)

    @classmethod
    def get_server_id(cls, rest_client):
        state, body = rest_client.send_get_request("ws/drmservers")
        if state != 200:
            logger.error(f"Get drm server info failed, "
                         f"status={state}, response={str(body)}.")
            raise Exception(f"Get drm server info failed, "
                            f"status={state}, response={str(body)}.")
        server_id = dict(body[0]).get('uuid')
        logger.info(f"DRM server uuid is [{server_id}]")
        return server_id

    @classmethod
    def discover_openstack(
            cls, project_id, pod_id, rest_client, site_id, service_type):
        # 封装发现openstack资源的请求体
        params = Params(project_id, pod_id)
        iam_user = f"{service_type}_service"
        iam_pwd = params.iam_account_preset_pwd
        iam_address = params.iam_address
        iam_port = params.iam_port
        iam_domain = params.iam_domain
        params_map = {
            "authType": "IAM", "devicetype": "FusionSphereOpenstack",
            "version": "6.1", "port": iam_port, "username": iam_user,
            "password": iam_pwd, "domain": iam_domain, "discoveryregions": ""}
        body_data = {
            "siteId": site_id, "ip": iam_address,
            "deviceType": "ism.drm.deviceType.fusionsphere.mainType",
            "deviceSubType":
                "ism.drm.deviceType.fusionsphereopenstack.subType",
            "paramMap": params_map}
        # 开始发现openstack
        state, body = rest_client.send_post_request(
            [body_data], "ws/resources/action/discoverResource")
        result = dict(body).get("result")
        fail_count = dict(body).get("failCount")
        # 无返回结果,或有结果但是失败个数不等于0则认为失败
        if check_value_null(result) or check_value_null(fail_count) or \
                fail_count != 0:
            logger.error(f"Discovery openstack failed: {state}, {body}.")
            raise Exception(body)
        # 下发任务成功后获取下发请求后的后台任务ID
        task_id = dict(result[0]).get("taskId")
        if check_value_null(task_id) or int(task_id) <= 0:
            logger.error(f"Discovery OpenStack task {task_id} failed.")
            raise Exception(body)
        # 间隔20秒检查一次后台任务是否成功,总共检查20次
        for _ in range(0, 20):
            sleep(20)
            state, body = rest_client.send_get_request(
                f"ws/backtasks/{task_id}")
            task_result = dict(body).get("taskResult")
            # 若返回结果为空则继续
            if check_value_null(task_result):
                continue
            # 返回结果等于1表示发现失败
            if task_result == 1:
                logger.error(f"Discovery openstack failed: {body}.")
                # 如果发现失败,检查是否是已经被发现了, 如果是，则认为是发现成功了
                task_detail = dict(body).get("taskDetail")
                if task_detail == "1073947431":
                    # 设备已经被发现,直接返回成功
                    logger.info("The openstack already existed.")
                    return True
                raise Exception(body)
            # 等于-1表示正在发现
            elif task_result == -1:
                logger.debug("The task is in progress, please wait.")
                continue
            else:
                logger.info("Discovery openstack successfully.")
                return True
        logger.error("Discovery openstack task timeout.")
        raise Exception("timeout")

    @classmethod
    def refresh_device(cls, rest_client, ops_uuid):
        base64_encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        refresh_url = f"ws/fusionsphere/{base64_encode_uuid}/action/refresh"
        state, body = rest_client.send_put_request(
            None, refresh_url)
        if state != 200:
            if 'errorCode' in dict(body) and (
                    str(dict(body).get('errorCode')) == "1073947428" or
                    str(dict(body).get('errorCode')) == "606"):
                # 正在刷新中
                logger.info("The device is being refresh, no need execute.")
                sleep(20)
                return True
            raise Exception(f'Failed to refresh device: {body}.')
        logger.info("Refresh_device cmd send successfully.")
        sleep(30)
        return True

    @classmethod
    def config_region(cls, project_id, pod_id, rest_client, devsn):
        params = Params(project_id, pod_id)
        region_info = params.project_region_id
        openstack_region = params.openstack_region
        openstack_domain = params.openstack_domain
        apicom_service_uri = params.apicom_service_uri
        global_domain_name = MO_API(project_id, pod_id).global_domain(is_central_cloud=False)
        external_global_domain_name = params.external_global_domain
        compute = \
            f"https://compute.{openstack_region}.{openstack_domain}:443/v2.1/"
        cinder = \
            f"https://volume.{openstack_region}.{openstack_domain}:443/v2/"
        network = f"https://network.{openstack_region}.{openstack_domain}:443/"
        drextend = \
            f"https://drextend.{openstack_region}.{openstack_domain}:443/"
        computeext = f"https://compute-ext.{openstack_region}." \
                     f"{openstack_domain}:443/v1.0/"
        metering = \
            f"https://metering.{openstack_region}.{openstack_domain}:443/"
        image = f"https://image.{openstack_region}.{openstack_domain}:443/"
        apicom = f"https://{apicom_service_uri}/v1/"

        resturi = f"ws/openstack/{devsn}/regions"
        service_endpoint = [
            {"serviceType": "compute", "serviceUrl": compute},
            {"serviceType": "volume", "serviceUrl": cinder},
            {"serviceType": "network", "serviceUrl": network},
            {"serviceType": "agent", "serviceUrl": drextend},
            {"serviceType": "compute-ext", "serviceUrl": computeext},
            {"serviceType": "metering", "serviceUrl": metering},
            {"serviceType": "image", "serviceUrl": image},
            {"serviceType": "apicom", "serviceUrl": apicom}]
        service_endpoint = cls._update_optional_service_endpoint(
            project_id, service_endpoint, region_info, global_domain_name,
            external_global_domain_name)
        service_endpoint = cls._init_config_region_endpoint_info(
            rest_client, devsn, service_endpoint, region_info)
        req_body = {"regionId": region_info, "services": service_endpoint}
        state, body = rest_client.send_post_request(
            [req_body], resturi)
        if state != 200:
            logger.error(f"Add region info failed: {state}, {body}.")
            raise HCCIException(
                '663709', rest_client.node_ip, openstack_region, openstack_domain,
                body)
        cls._do_config_region_check(
            rest_client, devsn, service_endpoint, region_info)
        logger.info("Add region info successfully.")

    @classmethod
    def _update_optional_service_endpoint(
            cls, project_id, service_endpoint, region_info,
            global_domain_name, external_global_domain_name):
        condition = Condition(project_id)
        manila = \
            f"https://sfs-api.{region_info}.{global_domain_name}:8786/"
        filemeter = \
            f"https://sfs-api.{region_info}.{global_domain_name}:8787/"
        if condition.is_nas_dr:
            service_endpoint.extend(
                [{"serviceType": "manila", "serviceUrl": manila},
                 {"serviceType": "filemeter", "serviceUrl": filemeter}])
        return service_endpoint

    @classmethod
    def _do_config_region_check(
            cls, rest_client, devsn, service_endpoint, region_info):
        logger.info("Start to check config region result.")
        exists_endpoint_info = \
            cls._get_current_region_exists_endpoint_info(
                rest_client, devsn, region_info)
        if not exists_endpoint_info:
            logger.error("Get exists endpoint info return None.")
            raise Exception("Get exists endpoint info return None.")
        for endpoint in service_endpoint:
            if endpoint not in exists_endpoint_info:
                logger.error(
                    f'{endpoint} not exists in region info, '
                    'please check task result.')
                raise Exception(
                    f'{endpoint} not exists in region info, '
                    'please check task result.')
        logger.info("Check config region result success.")

    @classmethod
    def _init_config_region_endpoint_info(
            cls, rest_client, devsn, service_endpoint, region_info):
        exists_endpoint_info = \
            cls._get_current_region_exists_endpoint_info(
                rest_client, devsn, region_info)
        exists_service = \
            [exists_endpoint.get("serviceType") for exists_endpoint in
             exists_endpoint_info]
        for endpoint in service_endpoint:
            if endpoint.get("serviceType") in exists_service:
                continue
            exists_endpoint_info.append(endpoint)
        return exists_endpoint_info

    @classmethod
    def _get_current_region_exists_endpoint_info(
            cls, rest_client, devsn, region_info):
        resturi = f"ws/openstack/{devsn}/regions"
        state, body = rest_client.send_get_request(resturi)
        if state != 200:
            logger.error(f"Get region info failed: {state}, {body}.")
            raise Exception(f"Get region info failed: {state}, {body}.")
        exists_endpoint_info = \
            [region.get("services", []) for region in body if
             region.get("description", "") == region_info]
        return exists_endpoint_info[0] if exists_endpoint_info else []

    @classmethod
    def config_tenant_api(cls, project_id, pod_id, rest_client):
        tenant_address, port = MO_API(project_id, pod_id).get_tenant_info()
        logger.info(f"Start config tenant client: {tenant_address}, {port}.")
        # ws/system/thirdservice/manageone
        req_body = {"domain": tenant_address, "port": port}
        state, body = rest_client.send_put_request(
            req_body, "ws/system/thirdservice/manageone")
        if state != 200:
            logger.error(
                f"Failed to config tenant northbound client[{body}].")
            raise Exception(body)
        logger.info("Config telnet client successfully.")

    @classmethod
    def _config_vha_az_map(cls, rest_client, region_id, az1_id, devsn, iam_ip):
        req_body = {
            "primary_az": az1_id, "primary_az_id": az1_id,
            "primary_region": region_id, "primary_region_id": region_id,
            "type": "vha"}
        cls._do_config_az_map(rest_client, devsn, iam_ip, req_body)

    @classmethod
    def _do_config_az_map(cls, rest_client, devsn, iam_ip, req_body):
        region1 = req_body.get("primary_region_id")
        region2 = req_body.get("second_region_id")
        az1 = req_body.get("primary_az_id")
        az2 = req_body.get("second_az_id")
        region_msg = ""
        second_region_and_az_exist = region1 and region2 and az1 and az2
        second_region_not_exist = region1 and not region2 and az1 and az2
        second_region_and_az_not_exist = region1 and not region2 and az1 and not az2
        if second_region_and_az_exist:
            region_msg = f"{region1},{az1}<->{region2},{az2}"
        elif second_region_not_exist:
            region_msg = f"{region1},{az1}<->{region1},{az2}"
        elif second_region_and_az_not_exist:
            region_msg = f"{region1},{az1}"
        state, body = rest_client.send_post_request(
            req_body, f"ws/openstack/{devsn}/azpairs/action/create")
        if state != 200:
            err_code = dict(body).get("errorCode", None)
            if str(err_code) == "1073948128":
                logger.info(f"AZ map already exists: {body}.")
                return
            logger.error(f"Config az map failed: {body}")
            raise HCCIException(
                '663711', rest_client.node_ip, iam_ip, region_msg, body)
        logger.info("Config AZ map successfully.")

    @classmethod
    def _do_config_volume_map(cls, rest_client, devsn, iam_ip, req_body):
        region_az1 = req_body.get("primary_az_id")
        region_az2 = req_body.get("second_az_id")
        volume1 = req_body.get("primary_volType")
        volume2 = req_body.get("second_volType")
        volume_msg = f"{region_az1},{volume1}<->{region_az2},{volume2}"
        state, body = rest_client.send_post_request(
            [req_body], f"ws/openstack/{devsn}/voltypepairs/action/create")
        if state != 200:
            err_code = dict(body).get("errorCode", None)
            if str(err_code) in ["1073948123", "1073948156"]:
                logger.info(f"Volume already exists: {body}")
                return
            logger.error(f"Config volume type map failed: {body}")
            raise HCCIException(
                '663712', rest_client.node_ip, iam_ip, volume_msg, body)
        logger.info("Config volume type successfully.")

    @classmethod
    def _do_config_share_type_map(cls, rest_client, devsn, iam_ip, req_body):
        region_az1 = req_body.get("primary_az_id")
        region_az2 = req_body.get("second_az_id")
        share_type1 = req_body.get("primary_volType")
        share_type2 = req_body.get("second_volType")
        share_type_msg = \
            f"{region_az1},{share_type1}<->{region_az2},{share_type2}"
        state, body = rest_client.send_post_request(
            [req_body], f"ws/openstack/{devsn}/voltypepairs/action/create")
        if state != 200:
            err_code = dict(body).get("errorCode", None)
            if str(err_code) in ["1073948123", "1073948156"]:
                logger.info(f"Volume already exists: {body}")
                return
            logger.error(f"Config volume type map failed: {body}")
            raise HCCIException(
                '663712', rest_client.node_ip, iam_ip, share_type_msg, body)
        logger.info("Config volume type successfully.")

    @classmethod
    def config_az_volume_mapping_for_csdr(
            cls, project_id, pod_id, rest_client, devsn):
        params = Params(project_id, pod_id)
        region_az_info = params.csdr_service_map_info
        iam_address = params.iam_address
        for map_info in region_az_info.replace(' ', '').split('#'):
            value = map_info.split('&')[0].split('|')
            region1_id = value[0].split(',')[0]
            az1_id = value[0].split(',')[1]
            az1_volume_type = value[0].split(',')[2]
            region2_id = value[1].split(',')[0]
            az2_id = value[1].split(',')[1]
            az2_volume_type = value[1].split(',')[2]
            logger.info(f"Start config: region1_id[{region1_id}], az1_id[{az1_id}], "
                        f"az1_volume_type[{az1_volume_type}], region2_id[{region2_id}], "
                        f"az2_id[{az2_id}], az2_volume_type[{az2_volume_type}]")
            az_req_body = {
                "primary_az": az1_id,
                "primary_az_id": az1_id,
                "primary_region": region1_id,
                "primary_region_id": region1_id,
                "second_az": az2_id,
                "second_az_id": az2_id,
                "second_region": region2_id,
                "second_region_id": region2_id,
                "type": "csdr"
            }
            cls._do_config_az_map(rest_client, devsn, iam_address, az_req_body)
            volume_req_body = {
                "primary_volType": az1_volume_type,
                "second_volType": az2_volume_type,
                "primary_az_id": f"{region1_id}_@_{az1_id}",
                "second_az_id": f"{region2_id}_@_{az2_id}",
                "type": "csdr"}
            if len(region_az_info.replace(' ', '').split('#')) == 1:
                volume_req_body["priority"] = 1
            else:
                volume_req_body["priority"] = map_info.split('&')[1]
            cls._do_config_volume_map(rest_client, devsn, iam_address, volume_req_body)
        logger.info("Config az and volume map successful.")

    @classmethod
    def config_az_share_type_mapping_for_csdr(
            cls, project_id, pod_id, rest_client, devsn):
        params = Params(project_id, pod_id)
        region_az_info = params.csdr_service_map_info_for_nas
        iam_address = params.iam_address
        for map_info in region_az_info.replace(' ', '').split('#'):
            value = map_info.replace(' ', '').split('|')
            region1_id = value[0].split(',')[0]
            az1_id = value[0].split(',')[1]
            az1_share_type = value[0].split(',')[2]
            region2_id = value[1].split(',')[0]
            az2_id = value[1].split(',')[1]
            az2_share_type = value[1].split(',')[2]
            logger.info(f"Start config: region1_id[{region1_id}], az1_id[{az1_id}], "
                        f"az1_share_type[{az1_share_type}], region2_id[{region2_id}], "
                        f"az2_id[{az2_id}], az2_share_type[{az2_share_type}]")
            csdr_req_body = {
                "primary_az": az1_id,
                "primary_az_id": az1_id,
                "primary_region": region1_id,
                "primary_region_id": region1_id,
                "second_az": az2_id,
                "second_az_id": az2_id,
                "second_region": region2_id,
                "second_region_id": region2_id,
                "object_type": "SFS",
                "type": "csdr"
            }
            cls._do_config_az_map(rest_client, devsn, iam_address, csdr_req_body)
            csha_req_body = {
                "primary_volType": az1_share_type,
                "second_volType": az2_share_type,
                "primary_az_id": f"{region1_id}_@_{az1_id}",
                "second_az_id": f"{region2_id}_@_{az2_id}",
                "type": "csdr", "object_type": "SFS", "priority": 1}
            cls._do_config_share_type_map(rest_client, devsn, iam_address, csha_req_body)
            logger.info("Config az and volume map successful.")

    @classmethod
    def config_az_volume_mapping_for_csha(
            cls, project_id, pod_id, rest_client, devsn):
        params = Params(project_id, pod_id)
        region_id = params.project_region_id
        region_az_info = params.csha_service_map_info
        iam_address = params.iam_address
        for map_info in region_az_info.replace(' ', '').split('#'):
            value = map_info.split('|')
            az1_id = value[0].split(',')[0]
            az1_volume_type = value[0].split(',')[1]
            az2_id = value[1].split(',')[0]
            az2_volume_type = value[1].split(',')[1]
            logger.info(f"start config: region_id[{map_info}], az1_id[{az1_id}], "
                        f"az1_volume_type[{az1_volume_type}], az2_id[{az2_id}], a"
                        f"z2_volume_type[{az2_volume_type}]")
            az_req_body = {
                "primary_az": az1_id, "primary_az_id": az1_id,
                "primary_region": region_id, "primary_region_id": region_id,
                "second_az": az2_id, "second_az_id": az2_id, "type": "csha"}
            cls._do_config_az_map(rest_client, devsn, iam_address, az_req_body)
            volume_req_body = {
                "primary_volType": az1_volume_type,
                "second_volType": az2_volume_type,
                "primary_az_id": f"{region_id}_@_{az1_id}",
                "second_az_id": f"{region_id}_@_{az2_id}",
                "type": "csha"
            }
            cls._do_config_volume_map(rest_client, devsn, iam_address, volume_req_body)
        logger.info("Config az and volume map successful.")

    @classmethod
    def config_az_volume_mapping_for_vha(
            cls, project_id, pod_id, rest_client, devsn):
        """
        配置VHA AZ/卷类型信息映射
        :return:
        """
        params = Params(project_id, pod_id)
        value_info = params.vha_service_map_info
        iam_address = params.iam_address
        logger.info("Get value_info is {}.".format(value_info))
        for volume in value_info.replace(' ', '').split('#'):
            data_lis = volume.split('|')
            region_id = data_lis[0]
            az_id = data_lis[1]
            vol_type = data_lis[2]
            # 配置AZ映射
            cls._config_vha_az_map(
                rest_client, region_id, az_id, devsn, iam_address)
            # 配置卷类型映射
            value_list = vol_type.split(',')
            logger.info(f"Start config volume type mapping[{value_list[0]}, {value_list[1]}].")
            req_body = {
                "primary_az_id": f"{region_id}_@_{az_id}",
                "second_az_id": f"{region_id}_@_{az_id}",
                "primary_volType": value_list[0], "second_volType": value_list[1],
                "type": "vha"}
            cls._do_config_volume_map(rest_client, devsn, iam_address, req_body)
        logger.info("Config az and volume map successfully.")

    @classmethod
    def change_network_and_shared_switch(
            cls, project_id, pod_id, rest_client, devsn):
        """
        修改switch开关
        :return:
        """
        logger.info("Start change network and shared switch.")
        region = Params(project_id, pod_id).project_region_id
        shared_switch_status = True
        network_switch_status = False
        if ProjectApi().get_project_info(project_id).get(
                'region_type') == "Type3":
            # Type场景需要开启网卡切换开关
            network_switch_status = True
        logger.info(
            f"Change network_switch_status: {network_switch_status}, "
            f"shared_switch_status: {shared_switch_status}.")
        req_body = {
            "isSwitchNetwork": network_switch_status,
            "supportSharedVolumes": shared_switch_status
        }
        uri = f"ws/openstack/{devsn}/regions/{region}/configs"
        state, body = rest_client.send_put_request(req_body, uri)
        if state != 200:
            logger.error(f"Change network switch failed: {body}.")
            raise HCCIException("663713", rest_client.node_ip, body)
        logger.info("Change network switch successful.")

    @classmethod
    def create_dmk_login_user(cls, dmk_info):
        DMK_API.login_dmk(dmk_info.dmk_float_ip, dmk_info.sys_user, dmk_info.sys_pwd)
        team_id, user_id = DMK_API.create_dmk_user(
            dmk_info.deploy_user, dmk_info.user_preset_pwd, DMK_API.dmk_role(), dmk_info.deploy_group)
        if team_id and user_id:
            ret = DMK_API.update_user_to_team(
                user_id, DMK_API.dmk_role(), ["OTD", dmk_info.deploy_group])
            if not ret:
                logger.error("Update user to OTD group failed.")
                raise HCCIException(
                    "663612", Component.REPLICATION,
                    "Update user return false.")
        DMK_API.login_dmk(dmk_info.dmk_float_ip, dmk_info.deploy_user, dmk_info.user_preset_pwd)
        is_success = DMK_API.change_dmk_user_passwd(user_id, dmk_info.user_preset_pwd, dmk_info.user_new_pwd)
        if not is_success:
            logger.error("Failed to update user password")
            raise HCCIException(
                "663612", Component.REPLICATION,
                "Change password return false.")
        DMK_API.login_dmk(dmk_info.dmk_float_ip, dmk_info.deploy_user, dmk_info.user_new_pwd)
        return team_id

    @classmethod
    def create_remote_account(cls, team_id, hosts, dmk_account_info):
        account_id = None
        try:
            account_id = DMK_API.get_dmk_account_id(dmk_account_info.user, dmk_account_info.ssh_user)
            DMK_API.validate_account(dmk_account_info.ssh_user, account_id, hosts)
        except Exception as err:
            logger.debug(f"Catch exception: {err}")
            if account_id:
                DMK_API.delete_dmk_account(account_id)
            DMK_API.add_account_to_dmk(dmk_account_info.ssh_user, team_id, dmk_account_info.ssh_pwd,
                                       dmk_account_info.sudo_pwd)
            ret = DMK_API.update_account_to_team(
                dmk_account_info.user, dmk_account_info.ssh_user, ["OTD", dmk_account_info.group],
                dmk_account_info.ssh_pwd, dmk_account_info.sudo_pwd)
            if not ret:
                logger.error("Update account to OTD group failed.")
                raise HCCIException(
                    "663612", Component.REPLICATION,
                    "Update account return false.")

    @classmethod
    def check_whether_backup_file_exists(
            cls, ssh_client, host, backup_path, result_check_list):
        result = SSH_API.exec_command_return_list(
            ssh_client, f"[ -f '{backup_path}' ] && echo CMD_RESULT=$?")
        if "CMD_RESULT=0" in str(result):
            logger.error(
                "The backup data is already exists, please confirm.")
            exception = HCCIException('665003', backup_path, host)
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查备份文件[{host}]",
                    itemname_en=f"Check the backup file[{host}]",
                    status="failure", error_msg_cn=exception))
        return result_check_list

    @classmethod
    def check_whether_disk_free_space_is_enough(
            cls, ssh_client, host, disk_zone, capacity, result_check_list):
        cmd = f"echo result=`df -m {disk_zone} | sed -n '2p' | " \
              f"awk '{{print $4,$6}}' | sed 's/ /:/g'`="
        result_list = SSH_API.exec_command_return_list(ssh_client, cmd)
        for result in result_list:
            if "result=" in str(result) and "df -m" not in str(result):
                info = str(result).split("=")[1].split(":")
                ava_size = info[0]
                if int(ava_size) < capacity * Capacity.ONE_THOUSAND:
                    logger.error(
                        f"Check disk space result: "
                        f"available size: {ava_size}, disk: {disk_zone}.")
                    exception = HCCIException('675010', host, disk_zone, "4")
                    result_check_list.append(
                        CheckResult(
                            itemname_ch="检查磁盘空间",
                            itemname_en="Check disk space",
                            status="failure", error_msg_cn=exception))
                else:
                    result_check_list.append(
                        CheckResult(
                            itemname_ch=f"检查磁盘空间[{host}]",
                            itemname_en=f"Check disk space[{host}]",
                            status="success"))
                break
        return result_check_list

    @classmethod
    def check_whether_disk_can_be_writen(
            cls, ssh_client, host, disk_zone, result_check_list):
        cmd = f"echo result=`lsattr -d {disk_zone} | sed 's/ /:/g'`="
        result_list = SSH_API.exec_command_return_list(ssh_client, cmd)
        for result in result_list:
            if "result=" in str(result) and "lsattr -d" not in str(result):
                info = str(result).split("=")[1].split(":")
                permissions = info[0]
                if "i" in permissions:
                    write_permission = False
                    logger.error(
                        f"Check disk permissions result: "
                        f"permissions: {write_permission}, disk: {disk_zone}.")
                    exception = HCCIException('675031', host, disk_zone)
                    result_check_list.append(
                        CheckResult(
                            itemname_ch="检查磁盘是否可写",
                            itemname_en="Check disk writable status",
                            status="failure", error_msg_cn=exception))
                else:
                    result_check_list.append(
                        CheckResult(
                            itemname_ch=f"检查磁盘是否可写[{host}]",
                            itemname_en=f"Check disk writable status[{host}]",
                            status="success"))
                break
        return result_check_list

    @classmethod
    def get_csdr_installed_info(cls, project_id, pod_id, region_id):
        csdr_installed = False
        block_supported, nas_supported = "False", "False"
        extend_infos = cls.get_cloud_dr_service_installed_list(
            project_id, pod_id, region_id)
        for extend_info in extend_infos:
            if extend_info.get("key") == Component.CSDR_U and extend_info.get(
                    "value") == "True":
                csdr_installed = True
        if not csdr_installed:
            return block_supported, nas_supported
        condition = Condition(project_id)
        if condition.is_sub or is_legacy_hcsu_scene(project_id):
            block_supported, nas_supported = cls._get_csdr_installed_info(
                project_id, pod_id, region_id)
        else:
            if not condition.is_current_dr_installed:
                block_supported = "True" if condition.is_block_dr else "False"
                nas_supported = "True" if condition.is_nas_dr else "False"
            else:
                config_path = os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'conf',
                    'env.ini')
                env = FILE_API(config_path)
                block_supported = "True" if condition.is_block_dr else \
                    env.get_value_by_key_and_sub_key(
                        "CSDR_supported_type", "block")
                nas_supported = "True" if condition.is_nas_dr else \
                    env.get_value_by_key_and_sub_key(
                        "CSDR_supported_type", "nas")
        return block_supported, nas_supported

    @classmethod
    def _get_csdr_installed_info(cls, project_id, pod_id, region_id):
        block_supported, nas_supported = "False", "False"
        request_api = cls.get_dr_request_api(project_id, pod_id)
        uuid = cls.get_server_id(request_api)
        devsn = cls.get_dest_device_id(project_id, pod_id, request_api, uuid)
        query_nas_uri = f"ws/openstack/{devsn}/voltypepairs?object_type=SFS"
        query_block_uri = \
            f"ws/openstack/{devsn}/voltypepairs?object_type=ECS_BMS"
        nas_state, nas_body = request_api.send_get_request(query_nas_uri)
        if nas_state != 200:
            logger.error(
                f"Get share type pairs info failed: {nas_state}, {nas_body}.")
            raise Exception(
                f"Get share type info pairs failed: {nas_state}, {nas_body}.")
        for share_type in nas_body:
            if share_type.get("type") == Component.CSDR_L and \
                    share_type.get("object_type") == "SFS" and \
                    region_id in str(share_type):
                nas_supported = "True"
        block_state, block_body = request_api.send_get_request(query_block_uri)
        if block_state != 200:
            logger.error(
                f"Get volume type pairs info failed: {block_state}, "
                f"{block_body}.")
            raise Exception(
                f"Get volume type pairs info failed: {block_state}, "
                f"{block_body}.")
        for share_type in block_body:
            if share_type.get("type") == Component.CSDR_L and \
                    region_id in str(share_type):
                block_supported = "True"
        return block_supported, nas_supported

    @classmethod
    def _get_threads_func_args(
            cls, hosts, auth_provider, func, *args,
            **kwargs):
        client_pool = defaultdict()
        thread_name = threading.current_thread().name
        for host in hosts:
            auth_provider.host = host
            client_pool[host] = cls(auth_provider)
        funcs = list()
        for host in client_pool:
            _func = (func, thread_name,
                     (client_pool[host].sudo_client, *args), kwargs)
            funcs.append(_func)
        return client_pool, funcs

    @classmethod
    def get_sdr_info(cls, cmdb):
        supported_dr_region_ids = cls._get_all_supported_dr_region_ids(cmdb)
        sdr_infos = list()
        for region_id in supported_dr_region_ids:
            service_lst = set()
            cloud_service_info = cmdb.get_cloud_service_info(
                region_id, Component.REPLICATION)
            extend_info = cloud_service_info[0].get("extendInfos", [])
            for extend_ in extend_info:
                extend_key = extend_.get("key")
                extend_value = extend_.get("value")
                if extend_key in [Component.CSDR_U, Component.CSHA_U,
                                  Component.VHA_U] and \
                        extend_value == "True":
                    service_lst.add(extend_key.lower())
            sdr_ctrl_float_ip = cls._get_sdr_ctrl_float_ip_from_new_cmdb(
                cmdb, region_id)
            sdr_info = \
                f"{region_id}@{sdr_ctrl_float_ip}#{'&'.join(service_lst)}"
            sdr_infos.append(sdr_info)
        return ",".join(sdr_infos)

    @classmethod
    def _get_all_supported_dr_region_ids(cls, cmdb):
        region_ids = cls._get_all_business_region_ids(cmdb)
        supported_dr_region_ids = \
            [region_id for region_id in region_ids if
             cmdb.get_cloud_service_info(region_id, Component.REPLICATION)]
        return supported_dr_region_ids

    @classmethod
    def _get_all_business_region_ids(cls, cmdb):
        regions = cmdb.get_region_info()
        condition = Condition(cmdb.project_id)
        region_ids = set()
        for region in regions:
            region_id = region.get("regionCode")
            region_type = region.get("regionType")
            if region_type in ["master", "dr_standby"] and (
                    condition.is_hcs_global or condition.is_primary_hcs_global
            ):
                continue
            region_ids.add(region_id)
        return region_ids

    @classmethod
    def _get_sdr_ctrl_float_ip_from_new_cmdb(cls, cmdb, region_id):
        sdr_machine_name_lst = ["PUB-SRV-03", "PUB-SRV-04", "PUB-SRV-DC1-02",
                                "PUB-SRV-DC2-02"]
        sdr_deploy_node_info = cmdb.get_deploy_node_info(
            region_id, Component.SDR)
        for node in sdr_deploy_node_info:
            if node.get("name") not in sdr_machine_name_lst:
                continue
            sdr_float_ip_info = node.get("floatIpAddresses")
            for float_ip_info in sdr_float_ip_info:
                if float_ip_info.get("cloudServiceIndex") != Component.SDR:
                    continue
                sdr_ctrl_float_ip = float_ip_info.get("floatIp")
                logger.info(
                    f"Get sdr ctrl float ip for {region_id} "
                    f"return {sdr_ctrl_float_ip}")
                return sdr_ctrl_float_ip

    @classmethod
    def _backup_dr_system_conf(cls, ssh_client):
        """
        备份指定配置文件
        :param ssh_client:
        :return: 临时路径
        """
        source_file_lst = ["/opt/BCManager/Runtime/LegoRuntime/conf/lego.properties",
                           "/opt/BCManager/Runtime/NodeAgent/conf/nodeAgent-pwd.properties",
                           "/opt/BCManager/Runtime/LegoRuntime/conf/ha.properties",
                           "/opt/BCManager/Runtime/bin/config/conf/config.properties",
                           "/opt/BCManager/Runtime/bin/config/conf/config.properties.sign",
                           "/opt/BCManager/Runtime/Tomcat6/conf/server.xml",
                           "/opt/BCManager/Runtime/LegoRuntime/conf/wcc/crypt"]
        cmd = f"[ -d '{Path.DR_TEMP_CONF_DEST_PATH}' ]; echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            cmd = f"mkdir -p {Path.DR_TEMP_CONF_DEST_PATH}"
            SSH_API.send_command(ssh_client, cmd)
        for source_file in source_file_lst:
            SSH_API.send_command(
                ssh_client,
                rf"\cp -rf {source_file} "
                rf"{Path.DR_TEMP_CONF_DEST_PATH} && echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            ssh_client, f"chown -Rh DRManager:LEGO {Path.DR_TEMP_CONF_DEST_PATH} && "
                        f"echo CMD_RESULT=$?", "CMD_RESULT=0")
        logger.info("Backup DR system conf file success.")

    @classmethod
    def register_adaptation_package(cls, project_id, pod_id, region_id):
        logger.info(f"Start to register eReplication adaptation package on {region_id}.")
        try:
            path, name, version = file_util.find_software_package_by_name_re(Pkg.ADAPTATION_PKG_REG)
        except Exception as exception:
            logger.error(f"Find adaptation_pkg_re:{Pkg.ADAPTATION_PKG_REG} failed. Reason is:{exception}.")
            raise HCCIException(663620, exception) from exception
        logger.info(f"Find adaptation_pkg[{path}] success, name[{name}, version is[{version}].")
        mo_client = MO_API(project_id, pod_id)
        file_path = os.path.join(path, name)
        mo_client.register_union_adaptation_pkg(region_id, file_path)
        logger.info(f"Succeeded to register eReplication adaptation package on {region_id}.")

    def __del__(self):
        if self.sudo_client:
            SSH_API.close_ssh(self.sudo_client)
        logger.info(f"SSH channel is closed: {self.host}.")

    @classmethod
    def _get_server_id_with_retry(cls, rest_client):
        logger.info("Start get server id with retry.")
        # 先从server服务器查询容灾站点的UUID, 循环查询5次, 每次间隔10秒, fsp环境上服务可能存在无法访问情况
        server_id = ""
        for count in range(5):
            server_id = cls.get_server_id(rest_client)
            if not check_value_null(server_id):
                break
            logger.error(f"Get drm info failed, retry {count + 1}")
            sleep(10)
            continue
        if check_value_null(server_id):
            err_msg = "Get server id failed."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info("Get server id with retry successfully.")
        return server_id

    @classmethod
    def _query_dr_sites_by_id(cls, rest_client, server_id):
        state, body = rest_client.send_get_request(f"ws/sites?serverUuid={server_id}")
        if state != 200:
            err_msg = f"Get site info failed, status={state}, response={str(body)}."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Get site info success.")
        return body

    @classmethod
    def _create_dr_site(cls, rest_client, server_id):
        logger.info(f"Start create rd site, server_id: {server_id}.")
        request_body = {"name": "autoSite", "description": "", "serverUuid": server_id}
        state, body = rest_client.send_post_request(request_body, "ws/sites")
        if state != 200:
            err_msg = f"Create site info failed, status={state}, response={str(body)}."
            logger.error(err_msg)
            raise Exception(err_msg)
        site_id = str(dict(body).get('siteId'))
        if check_value_null(site_id):
            err_msg = f"Create site failed, body={body}."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Create rd site successfully: {site_id}.")
        return site_id

    @classmethod
    def _query_fusionsphere_info_by_site_id(cls, rest_client, site_id):
        state, body = rest_client.send_get_request(f"ws/fusionsphere?siteId={site_id}&cloudType=")
        if state != 200:
            err_msg = f"Get fusionsphere info failed, status={state}, response={str(body)}."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Get fusionsphere info success.")
        return body

    @classmethod
    def _download_local_from_primary_tmp(cls, primary_ip, ssh_user, ssh_pwd, file_name):
        logger.info(f"Start from node({primary_ip}) download certs and configs.")
        ssh_client = None
        try:
            # 从源主机文件下载到本地
            primary_back_path = f"{Path.DR_USER_PATH}/{file_name}"
            local_back_path = f"{Path.LOCAL_TEMP_PATH}/{file_name}"
            auth_ftp = AuthFtp(primary_ip, ssh_user, ssh_pwd).auth_provider
            SSH_API.get_file(auth_ftp, primary_back_path, local_back_path)
            ssh_client = SSH_API.get_ssh_client(SSHClientInfo(primary_ip, ssh_user, ssh_pwd))
            SSH_API.execute_command(ssh_client, rf"\rm -rf {primary_back_path}")
        except Exception as err:
            err_msg = f"From node {primary_ip} download certs and configs failed."
            logger.error(err_msg)
            raise Exception(err_msg) from err
        finally:
            SSH_API.close_ssh(ssh_client)
        logger.info(f"From node({primary_ip})  download certs and configs success.")
