# -*- coding:utf-8 -*-
import threading
import traceback
from typing import List
from typing import Tuple


class ExcThread(threading.Thread):
    def __init__(self, target, thread_name, *args, **kwargs):
        super().__init__(name=thread_name)
        self.function = target
        self.args = args
        self.kwargs = kwargs
        self.exit_code = 0  # 添加一个flag标记，用于判断子线程是否出错
        self.exception = None
        self.exc_traceback = ''
        self.result = None

    def run(self):
        try:
            self.result = self.function(*self.args, **self.kwargs)
        except Exception as err:
            self.exit_code = 1
            self.exception = err
            self.exc_traceback = traceback.format_exc()

    def get_result(self):
        return self.result

    @classmethod
    def get_instance(cls, target, thread_name, args, kwargs):
        return ExcThread(target, thread_name, *args, **kwargs)

    @classmethod
    def exec_func_in_thread(cls, funcs: List or Tuple):
        """
        funcs: ((target, target_name, args, kwargs), ……)
        *target* is the callable object to be invoked by the run()
        *thread_name* is the name of the child thread
        *args* is the argument tuple for the target invocation.
        Defaults to (). --> Tuple object
        *kwargs* is a dictionary of keyword arguments for the target
        invocation. Defaults to {}. --> Dict object
        """
        threads = list()
        for func, thread_name, args, kwargs in funcs:
            thread = cls.get_instance(func, thread_name, args, kwargs)
            threads.append(thread)

        for _thread in threads:
            _thread.start()

        for _thread in threads:
            _thread.join()
            if _thread.exit_code != 0:
                raise Exception(_thread.exception)

    @classmethod
    def exec_func_in_thread_and_return(cls, funcs: List or Tuple):
        """
        funcs: ((target, thread_name, args, kwargs), ……)
        *target* is the callable object to be invoked by the run()
        *thread_name* is the name of the child thread
        *args* is the argument tuple for the target invocation.
        Defaults to (). --> Tuple object
        *kwargs* is a dictionary of keyword arguments for the target
        invocation. Defaults to {}. --> Dict object
        return: [result_thread1, result_thread2, ……]
        """
        threads = list()
        result = list()
        for func, thread_name, args, kwargs in funcs:
            thread = cls.get_instance(func, thread_name, args, kwargs)
            threads.append(thread)

        for _thread in threads:
            _thread.start()

        for _thread in threads:
            _thread.join()
            if _thread.exit_code != 0:
                raise Exception(_thread.exception)
            t_res = _thread.get_result()
            result.append(t_res)
        return result
