# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.Logic.dmk.dmk_data import Host
from utils.business.dmk_util import DmkApi
from utils.common.exception import HCCIException

from plugins.CSBS.common.model import Node
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import Params
from plugins.CSBS.common.constant import KARBOR_PROXY_ROOT_NAME
from plugins.CSBS.common.constant import KARBOR_PROXY_USER_NAME
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation

logger.init("CSBS")


class DmkTool(object):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.role = '2'
        self.db_param_dict = Params(self.project_id, self.pod_id).get_params()
        self.dmk_api = DmkApi()
        self.dmk_float_ip = self.db_param_dict.get('dmk_floatIp')
        self.dmk_ui_username = self.db_param_dict.get('dmk_ui_username')
        self.dmk_ui_password = self.db_param_dict.get('dmk_ui_password')

    def install_maglev_agent(self, node_list: list[Node], node_names: list[str]):
        dmk_username = self.db_param_dict.get('CBS_dmk_groupadminname', self.db_param_dict.get('dmk_account'))
        if not dmk_username:
            logger.error("Failed to obtain dmk account name.")
            raise False
        host_list = []
        for index, node in enumerate(node_list):
            host_list.append(
                Host(ip=node.node_ip, name=node_names[index], ssh_user=node.user, ssh_password=node.user_pwd,
                     root_password=node.root_pwd))
        logger.info(f'Login dmk, dmk_floatIp:{self.dmk_float_ip}, '
                    f'dmk_ui_username:{dmk_username}.')
        self.dmk_api.login_dmk(self.dmk_float_ip,
                               dmk_username,
                               self.db_param_dict.get('CBS_dmk_grouppasswd', self.db_param_dict.get('dmk_account_pwd')))
        logger.info(f"Start to install maglev agent in nodes: {', '.join(node.ip for node in host_list)}")
        try:
            self.dmk_api.install_agent(host_list)
        except Exception as e:
            logger.error(f"Install maglev agent failed! msg: {e}")
            return False
        return True

    def create_dmk_user(self, user_name, group_name, tmp_passwd, passwd):
        # 创建用户，使用dmk管理员账户
        logger.info(f'Login dmk, dmk_floatIp:{self.dmk_float_ip}, '
                    f'dmk_ui_username:{self.dmk_ui_username}.')
        # login by sysadmin
        self.dmk_api.login_dmk(self.dmk_float_ip,
                               self.dmk_ui_username,
                               self.dmk_ui_password)
        logger.info(f'Start creating dmk user, username:{user_name},'
                    f'team_name:{group_name}.')
        team_id, user_id = self.dmk_api.create_dmk_user(user_name,
                                                        tmp_passwd,
                                                        self.role,
                                                        group_name)

        # 初次修改用户密码
        logger.info('Start modifying user password firstly.')
        self.dmk_api.login_dmk(self.dmk_float_ip, user_name, tmp_passwd)
        self.dmk_api.change_dmk_user_passwd(user_id, tmp_passwd, passwd)
        return team_id, user_id

    def _create_dmk_account(self, admin_username, account_name, account_pwd, sudo_pwd, team_id):
        account_id = None
        try:
            account_id = self.dmk_api.get_dmk_account_id(
                admin_username, account_name)
        except HCCIException:
            pass
        except Exception as err1:
            logger.error(f"Execute error:{str(err1)}.")
            raise err1

        if account_id:
            self.dmk_api.update_account_to_multi_team(
                admin_username, account_name, [admin_username],
                account_pwd, sudo_pwd)
        else:
            self.dmk_api.add_account_to_dmk(account_name, team_id, account_pwd, sudo_pwd)

    def create_dmk_proxy_account(self, admin_username, admin_pwd):
        # CSBS_Proxy
        self.dmk_api.login_dmk(self.dmk_float_ip, admin_username, admin_pwd)
        team_id = self.dmk_api.get_dmk_team_id(admin_username)
        karborproxy_user_info = KarborProxyOperation(self.project_id).get_user_info_from_install()
        dpamanager_username = KARBOR_PROXY_USER_NAME
        dpamanager_pwd = karborproxy_user_info.user_pwd
        karbor_proxy_root_username = KARBOR_PROXY_ROOT_NAME
        karbor_proxy_root_pwd = karborproxy_user_info.root_pwd
        self._create_dmk_account(admin_username, dpamanager_username, dpamanager_pwd, karbor_proxy_root_pwd, team_id)
        self._create_dmk_account(admin_username, karbor_proxy_root_username, karbor_proxy_root_pwd,
                                 karbor_proxy_root_pwd, team_id)

    def create_dmk_account(self, admin_username, admin_pwd):
        cbs_admin_name = "cbs_admin"
        cbs_deploy_name = "cbs_deploy"
        cbs_deploy_pwd = self.db_param_dict.get('deploy_accountpasswd')
        cbs_admin_pwd = self.db_param_dict.get('admin_accountpasswd')

        # CBS
        self.dmk_api.login_dmk(self.dmk_float_ip, admin_username, admin_pwd)

        # 如果账号仓库中已经有djmanager、cbs_admin、cbs_deploy、root,
        # 则删除账号然后重新创建
        # 主要是解决用户在已经装过的环境中修改了这些账号的密码，再重新安装会出现bug
        self.dmk_api.login_dmk(self.dmk_float_ip, admin_username, admin_pwd)
        user_id = self.dmk_api.get_dmk_user_id()
        team_id = self.dmk_api.get_dmk_team_id(admin_username)
        karbor_util = KarborUtil(self.project_id, self.pod_id)
        karbor_node = karbor_util.get_karbor_node_list()[0]

        karbor_root_pwd = karbor_node.root_pwd

        self._create_dmk_account(admin_username, cbs_deploy_name, cbs_deploy_pwd, karbor_root_pwd, team_id)
        self._create_dmk_account(admin_username, cbs_admin_name, cbs_admin_pwd, karbor_root_pwd, team_id)
        self._create_dmk_account(admin_username, karbor_node.user, karbor_node.user_pwd, karbor_root_pwd,
                                 team_id)
        self._create_dmk_account(admin_username, karbor_node.root_name, karbor_root_pwd, karbor_root_pwd, team_id)

        team_list = [admin_username, 'OTD']
        self.dmk_api.login_dmk(self.dmk_float_ip, self.dmk_ui_username, self.dmk_ui_password)
        result_dmk = self.dmk_api.update_user_to_multi_team(user_id, self.role, team_list)
        if not result_dmk:
            logger.error("Update user to multi_team for CBS failed.")
            raise HCCIException(640004)

    def reset_dmk_first_login(self, dmk_username, dmk_passwd):
        self.dmk_api.login_dmk(self.dmk_float_ip, dmk_username, dmk_passwd)
        self.dmk_api.reset_first_login()
