# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
import utils.common.software_package_util as file_util
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common.code2Msg import code2Msg
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import CSBS_SERVICE_NAME, PKG_VERSION
from plugins.CSBS.common.error_code import UPLOAD_ADAPTIONPACKAGE_ERROR
from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.util import auto_retry, get_env_config

logger.init("CSBS-VBS")

# 十统一适配包上传成功状态
ADAPTATION_UPLOAD_SUCCESS_STATUS = 1
KARBOR_ADAPTATION_PKG_REG = f"^resource_10UnifiedAccess2MO_CSBS-OceanProtect_Karbor_{PKG_VERSION}\\.tar\\.gz$"
KARBOR_PROXY_ADAPTATION_PKG_REG = f"^resource_10UnifiedAccess2MO_CSBS-OceanProtect_KarborProxy_{PKG_VERSION}\\.tar\\.gz$"


class RegisterAdaptationPackage(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterAdaptationPackage, self).__init__(project_id, pod_id, regionid_list)
        # 根据当前场景获取对应的十统一适配包
        # 1. 当前工程中不存在安装KarborProxy resource_10UnifiedAccess2MO_CSBS-VBS_Karbor*
        # 2. 当前工程中存在安装KarborProxy resource_10UnifiedAccess2MO_CSBS-VBS_KarborProxy*
        self.adaptation_pkg_reg = KARBOR_ADAPTATION_PKG_REG
        if get_project_condition_boolean(self.project_id, "InstalledCSBSProxy"):
            self.adaptation_pkg_reg = KARBOR_PROXY_ADAPTATION_PKG_REG

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to register CSBS-VBS adaptation package.")
        try:
            path, name, version = file_util.find_software_package_by_name_re(self.adaptation_pkg_reg)
        except Exception as execute_error:
            err_msg = f"Find adaptation_pkg_re:{self.adaptation_pkg_reg} failed. Reason is:{execute_error}."
            logger.error(err_msg)
            return Message(500, err_msg)
        logger.info(f"Find adaptation_pkg[{path}] success, name[{name}, version is[{version}].")
        # 写入当前适配包版本用于后续回退
        self._pre_store_origin_adapter_version()

        file_path = os.path.join(path, name)
        result = self.register_union_adaptation_pkg(self.region_id, file_path)
        if not result:
            msg_cn, msg_en, _, _ = code2Msg(UPLOAD_ADAPTIONPACKAGE_ERROR)
            return Message(500, msg_cn, msg_en)
        logger.info("Succeeded to register CSBS-VBS adaptation package.")
        return Message(200)

    def _pre_store_origin_adapter_version(self):
        conf_env_file = get_env_config()
        old_version = conf_env_file.get_value_by_key_and_sub_key('AdaptationPackage', 'old_version')
        if old_version:
            return

        result = ManageOneUtil2().query_adaptation_package_record(
            self.project_id, CSBS_SERVICE_NAME, self.region_id)
        origin_version = None
        for pkg in result:
            if pkg.get('status') == ADAPTATION_UPLOAD_SUCCESS_STATUS:
                logger.info(f"Query adaptation pkg success: {pkg}.")
                origin_version = pkg.get("version")
                if origin_version:
                    break

        if not origin_version:
            query_err_msg = 'Query adaptation pkg failed.'
            logger.error(query_err_msg)
            raise HCCIException(UPLOAD_ADAPTIONPACKAGE_ERROR, query_err_msg)
        conf_env_file.set_value_by_key_and_sub_key('AdaptationPackage', 'old_version', origin_version)
        logger.info("Store origin adapter version to file successfully.")

    @auto_retry(max_retry_times=5, delay_time=60)
    def register_union_adaptation_pkg(self, region_id, file_path):
        package_infos = {
            "regionId": region_id,
            "file_path": file_path
        }
        logger.info(f"Start to register adaptation package, package_infos: {package_infos}.")
        result = ManageOneUtil2().upload_adaptation_package(self.project_id, self.pod_id, package_infos)
        if result.get("status") == "failed":
            message = result.get("msg")
            err_msg_en = f"Failed use manage service api to register adaptation package, " \
                         f"file path : {file_path}, err_msg:{message}."
            logger.error(err_msg_en)
            raise HCCIException(UPLOAD_ADAPTIONPACKAGE_ERROR, (err_msg_en, err_msg_en))
        logger.info("Succeeded to register adaptation package.")
        return True
