import utils.common.log as logger
from utils.business.arb_util import ArbApi
from utils.business.project_util import ProjectApi
from utils.common import param_check

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import check_string_param


class KarborArbiterCfg(node_installer.Installer):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        super(KarborArbiterCfg, self).__init__()
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)

    def install(self):
        karbor_client = SshClient()
        arb_info = ArbApi().get_arb_info(self.pod_id)
        if self.karbor_util.get_install_type() != "private_recover":
            return True
        karbor_node = self.karbor_util.get_karbor_node_list()[0]
        logger.info('Ssh to karbor node:{}'.format(karbor_node.node_ip))
        ssh_client = karbor_client.get_ssh_client(karbor_node)
        arbit_ip_list = arb_info.get("ips")
        # 校验参数
        ip_version = self.param_tool.get_ip_version()
        for _ip in arbit_ip_list:
            if not param_check.check_param_ip(_ip, ip_version):
                raise Exception(f"Invalid ip address, the parma is {_ip}.")

        local_dc, peer_dc = self.get_arbit_localdc_peerdc(karbor_node.node_ip, arb_info)
        if not check_string_param(local_dc) or not check_string_param(peer_dc):
            raise Exception("The obtained character string is invalid.")
        arbit_ip_string = ",".join(arbit_ip_list)
        cmds = f"bash /karbor/CSBS/install_scripts/config_arb.sh {arbit_ip_string} {local_dc} {peer_dc} "
        logger.info('Cmd: {}'.format(cmds))
        expect_str = "Enter password of user(arbiterClient):"
        result = karbor_client.ssh_send_command_expect(ssh_client, cmds, expect_str)
        if karbor_client.failed_to_return(result, expect_str, ssh_client):
            logger.error('Failed executing:{}, expect_str:{}, '
                         'result:{}'.format(cmds, expect_str, result))
            return False

        arbit_user_pwd = arb_info['user_info']['password']
        expect_str = "Config arbitration service successful."
        result = karbor_client.ssh_send_command_expect(ssh_client, arbit_user_pwd, expect_str)
        if karbor_client.failed_to_return(result, expect_str, ssh_client):
            logger.error('Failed, expect_str:{}, result:{}'.format(expect_str, result))
            return False
        return True

    def get_arbit_localdc_peerdc(self, ip_key, arb_info):
        """
        :param ip_key: karbor节点ip
        :return: 返回dc，例如：dc001,dc002
        """
        project_api = ProjectApi()
        replication_dict = project_api.get_project_ha_dr_status(self.project_id)
        karbor_list = self.karbor_util.get_karbor_ip_list()
        if replication_dict['RegionConHA'] == 1:
            arb_dc1_name = arb_info['dc_name']['dc1_name']
            if not arb_dc1_name:
                raise Exception(f"Get get_arb_dc1_name failed.")
            arb_dc2_name = arb_info['dc_name']['dc2_name']
            if not arb_dc2_name:
                raise Exception(f"Get arb_dc2_name failed.")
            local_dc_name = arb_dc1_name if karbor_list.index(ip_key) == 0 else arb_dc2_name
            peer_dc_name = arb_dc2_name if karbor_list.index(ip_key) == 0 else arb_dc1_name
            return local_dc_name, peer_dc_name
        else:
            return '', ''

    def check(self):
        return True

    def rollback(self):
        return True
