import utils.common.log as logger
import yaml
from utils.business.dmk_util import DmkApi

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool, Params
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import check_url_param


class SCServerConfig(node_installer.Installer):
    def __init__(self, project_id, pod_id, client):
        super(SCServerConfig, self).__init__()
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.karbor_node_list = self.karbor_util.get_karbor_node_list()
        self.sc_url = self.param_tool.get_sc_url()
        self.global_auth_url = self.param_tool.get_iam_url()
        params = Params(project_id, pod_id)
        self.ecs_endpoint = self.get_ecs_endpoint(params.get_params())
        self.karbor_client = client
        self.ssh_client = SshClient()

    def sc_server_config(self, karbor_node):
        logger.info('Ssh to karbor node: {}'.format(karbor_node.node_ip))
        if self.karbor_client:
            karbor_client = self.karbor_client
        else:
            karbor_client = self.ssh_client.get_ssh_client(karbor_node)
        if not karbor_client:
            return False
        # noinspection PyBroadException
        check_url_param(self.sc_url)
        sc_url_cmd = f"source /opt/huawei/dj/inst/utils.sh;set_karbor_endpoints --sc_endpoint {self.sc_url}"
        result = self.check_cmd_result(sc_url_cmd, karbor_client)
        if not result:
            return False
        check_url_param(self.global_auth_url)
        global_auth_cmd = f"source /opt/huawei/dj/inst/utils.sh;" \
                          f"set_auth_endpoint --global_auth_url {self.global_auth_url}/v3/"
        result = self.check_cmd_result(global_auth_cmd, karbor_client)
        if not result:
            return False
        # add ecs endpoint confid in 8.0.0
        check_url_param(self.ecs_endpoint)
        ecs_cmd = f"source /opt/huawei/dj/inst/utils.sh;set_karbor_endpoints --apicom_endpoint {self.ecs_endpoint}"
        result = self.check_cmd_result(ecs_cmd, karbor_client)
        if not result:
            return False
        return True

    def install(self):
        return self.sc_server_config(self.karbor_node_list[0])

    def rollback(self):
        return True

    def check(self):
        return True

    def get_ecs_endpoint(self, param_dict):
        try:
            return self._get_ecs_url(param_dict)
        except Exception as err:
            logger.error('Get ecs endpoint failed, '
                         'Execute error: {}'.format(err))
            raise err

    def _get_ecs_url(self, param_dict):
        # get apicom domain in dmk public config
        dmk_ui_user = param_dict['dmk_ui_username']
        dmk_ui_pwd = param_dict['dmk_ui_password']
        dmk_float_ip = param_dict['dmk_floatIp']
        logger.info('Get dmk info: [dmk_ui_username:{}, '
                    'dmk_floatIp:{}]'.format(dmk_ui_user, dmk_float_ip))
        dmk = DmkApi()
        dmk.login_dmk(dmk_float_ip, dmk_ui_user, dmk_ui_pwd)
        public_config = dmk.get_dmk_public_config()
        logger.info('Get dmk public config successfully')
        public_var = yaml.safe_load(public_config)
        api_com_domain = \
            public_var['g_current_region']['apicom']['service']
        ecs_url = "https://" + api_com_domain
        return ecs_url

    def check_cmd_result(self, url_cmd, ssh_client):
        logger.info('Cmd: {}'.format(url_cmd))
        result = self.ssh_client.ssh_exec_command_return(ssh_client, url_cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            self.ssh_client.ssh_close(ssh_client)
            logger.error('Failed Executing: {}'.format(url_cmd))
            return False
        return True
