import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import KAEBOR_EXT_PORT
from plugins.CSBS.common.constant import RegisterToMOInfo
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.register_silvan import RegisterSilvan
from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")


class CSBSAndVBSRegisterSilvan(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(CSBSAndVBSRegisterSilvan, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.csbs_id = "csbs"
        self.vbs_id = "vbs"
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.register_silvan = RegisterSilvan(self.project_id, self.pod_id)
        self.region_id = self.param_tool.get_region_id()
        self.console_domain_name = \
            self.param_tool.get_console_domain_name()
        self.register_info = RegisterToMOInfo(self.region_id, self.console_domain_name)
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            result = self._register_silvan()
            if result.status_code != 200:
                return result
            self._register_query_expenses_notification()
        except Exception as err:
            logger.error(str(err))
            return Message(500, HCCIException("640016", str(err)))
        return Message(200)

    def _register_silvan(self):
        if not self._register_endpoint():
            return Message(500, HCCIException("640017"))

        # Registering the association between regions and endpoints.
        csbs_float_ip = self.karbor_util.get_karbor_float_ip()
        result2_1 = self.register_silvan.register_region_and_endpoint_association(
            self.pod_id, self.region_id, self.csbs_id, csbs_float_ip, KAEBOR_EXT_PORT)
        if not result2_1:
            logger.error(f"Register {self.csbs_id} service of the association between region and endpoint failed.")
            return Message(500, HCCIException("640018"))

        result2_2 = self.register_silvan.register_region_and_endpoint_association(
            self.pod_id, self.region_id, self.vbs_id, csbs_float_ip, KAEBOR_EXT_PORT)
        if not result2_2:
            logger.error(f"Register {self.vbs_id} service of the association between region and endpoint failed.")
            return Message(500, HCCIException("640018"))

        # csbs detail links
        result3_2 = self.register_silvan.register_links_info(
            self.pod_id, "detail_url_csbs", self.csbs_id, "detail", self.register_info.csbs_detail_link, "product",
            region_id=self.region_id)
        if not result3_2:
            logger.error(f"Register {self.csbs_id} detail link failed.")
            return Message(500, HCCIException("640019"))

        # vbs detail links
        result3_4 = self.register_silvan.register_links_info(
            self.pod_id, "detail_url_vbs", self.vbs_id, "detail", self.register_info.vbs_detail_link, "product",
            region_id=self.region_id)
        if not result3_4:
            logger.error(f"Register {self.vbs_id} detail link failed.")
            return Message(500, HCCIException("640019"))

        is_succ, code = self._register_csbs_vbs_info()
        if not is_succ:
            return Message(500, HCCIException(code))
        return Message(200)

    def _register_csbs_vbs_info(self):
        # register_product_action
        for action in self.register_info.csbs_vbs_action_list:
            result4 = self.register_silvan.register_product_action(self.pod_id, action)
            if not result4:
                logger.error(f"Register product action failed, action info:{action}.")
                return False, "640021"

        # Register csbs product info.
        result4_1 = self.register_silvan.register_product_info(
            self.pod_id, self.region_id, self.csbs_id, self.register_info.csbs_product_info)
        if not result4_1:
            logger.error(f"Register {self.csbs_id} product info failed.")
            return False, "640020"

        # Register quota indicators.
        result5_1 = self.register_silvan.register_quota_indicators(
            self.pod_id, self.csbs_id, self.register_info.csbs_quota_indicators)
        if not result5_1:
            logger.error(f"Register {self.csbs_id} quota indicator info failed.")
            return False, "640088"

        # Register csbs meter metrics.
        result6_1 = self.register_silvan.register_meter_metrics(self.pod_id, self.register_info.metric_info)
        if not result6_1:
            logger.error(f"Register {self.csbs_id} metric info failed.")
            return False, "640089"
        return True, ""

    def _register_endpoint(self):
        return self.register_silvan.register_endpoint(self.csbs_id, self.register_info.csbs_endpoint_info_zh) \
               and self.register_silvan.register_endpoint(self.csbs_id, self.register_info.csbs_endpoint_info_en,
                                                          _type="en")

    def _register_query_expenses_notification(self):
        self.register_silvan.register_notifications(
            self.register_info.get_query_expenses_notification_info(self.csbs_id))
        self.register_silvan.register_notifications(
            self.register_info.get_query_expenses_notification_info(self.vbs_id))
