import json
import re

import requests
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.message import Message
from utils.common.check_result import CheckResult

from plugins.CSBS.common.karbor_cpshelper import CPSInfo, CpsHelper
from plugins.CSBS.common.ebackup_util import EBackupUtil
from plugins.CSBS.common.model import Account
from plugins.CSBS.common.util import check_ip
from plugins.CSBS.common.step_base import UpgradeBaseSubJob

logger.init("CSBS-VBS")


class PreCheckWorkflow(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.ebackup_util = EBackupUtil(project_id, pod_id, regionid_list)
        self.workflow_ip_list = self.ebackup_util.get_workflow_ip_list()
        self.param_util = ParamUtil()
        self.base_url = None
        self.admin_account = None
        self.headers = None
        self.check_results = []

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.precheck_workflow()
        except Exception as exception:
            logger.error(f"Execute error:{exception}.")
            return Message(500, f"Execute error:{exception}.")

    def precheck_workflow(self):
        self._init_workflow_param()
        self.check_login()
        self._check_permit_for_admin()
        self.check_az_ebackup_map_param()
        self.logout()
        logger.info('Login workflow check finished.')
        return Message(200, check_results=self.check_results)

    def _init_workflow_param(self):
        workflow_manage_ip = self._get_workflow_manage_ip()
        self._init_workflow_base_url(workflow_manage_ip)
        self._get_workflow_admin_account()

    def _check_permit_for_admin(self):
        logger.info("begin to check permit of admin account.")
        check_url = f'{self.base_url}/vbackup_server'
        rsp = requests.get(check_url, headers=self.headers, verify=False, timeout=30)
        if rsp.status_code != 200:
            logger.error(f"Request {check_url} return not 200.")
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            raise Exception("Errcode is empty.")
        if errcode == 1610664355:
            raise Exception("Account admin is abnormal.")
        elif errcode != 0:
            description = rsp.json()["error"]["description"]
            raise Exception(f"Login {check_url} failed,description:{description}.")
        else:
            logger.info("Check admin account successfully.")

    def check_login(self):
        logger.info("Begin to login workflow portal.")
        login_url = f'{self.base_url}/login'
        headers = {'Accept': 'application/json'}
        data = f'{{"scope":0,"username":"{self.admin_account.account_name}",' \
               f'"password":"{self.admin_account.account_pwd}"}}'
        try:
            rsp = requests.post(login_url, headers=headers, data=data, verify=False, timeout=30)
        except Exception as err:
            err_msg = f"Failed to login eBackup({login_url}) with admin, reason: {str(err)}."
            logger.error(err_msg)
            raise Exception(err_msg) from err
        if rsp.status_code != 200:
            logger.error(f"Request {login_url} return not 200.")
        errcode = rsp.json()["error"]["code"]
        if errcode == 0:
            logger.info("Login ebackup success.")
            token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            self.headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': token,
                'Cookie': f'language=en;{session_content};DEVICE_ID=dev;'
                          f'sessionIdleTime=60000; MACHINEROLE=0;CSRF_IBASE_TOKEN={token}'
            }
            logger.info("Login workflow portal successfully.")
            return
        description = rsp.json()["error"]["description"]
        err_msg = f"Login ebackup failed, login_url:{login_url}, description: {description}."
        logger.error(err_msg)
        raise Exception(err_msg)

    def logout(self):
        url = f"{self.base_url}/sessions"
        try:
            requests.delete(url, headers=self.headers, verify=False, timeout=30)
        except Exception as err:
            logger.warning(f"Failed to delete the session, err_msg:{err}.")
        logger.info("Logout success.")

    def _get_workflow_admin_account(self):
        ebackup_workflow_user = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                          'karbor_eBackup_admin')
        if not ebackup_workflow_user:
            ebackup_workflow_user = "admin"
        ebackup_workflow_user_pwd = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                              'karbor_eBackup_admin_pwd')
        if not ebackup_workflow_user_pwd:
            raise Exception("Get eBackup workflow user password failed.")
        self.admin_account = Account(ebackup_workflow_user, ebackup_workflow_user_pwd)

    def _get_workflow_manage_ip(self):
        workflow_manage_ip = None
        for workflow_ip in self.workflow_ip_list:
            workflow_login_url = f"https://{workflow_ip}:8088"
            logger.info(f"workflow_login_url is:{workflow_login_url}")
            try:
                rsp = requests.post(workflow_login_url, headers={'Accept': 'application/json'}, verify=False,
                                    timeout=30)
            except Exception as exception:
                logger.warning(f"Request {workflow_login_url} failed:{exception}.")
                continue
            redirect_history_list = rsp.history
            if not redirect_history_list:
                continue
            redirect_url = redirect_history_list[len(redirect_history_list) - 1].headers['Location']
            workflow_manage_ip = re.findall(r'[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+', redirect_url)[0]
            if workflow_manage_ip:
                break
        if not check_ip(workflow_manage_ip):
            logger.error("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
            raise Exception("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
        logger.info(f"workflow_node_ip is:{workflow_manage_ip}.")
        return workflow_manage_ip

    def _init_workflow_base_url(self, workflow_mg_float_ip):
        if not check_ip(workflow_mg_float_ip):
            raise Exception(f"Illegal workflow management float ip:{workflow_mg_float_ip}.")
        self.base_url = f"https://{workflow_mg_float_ip}:8088/rest/dev"

    def check_az_ebackup_map_param(self):
        if not self.base_url or not self.headers:
            self._init_workflow_param()
        az_ebackup_map_list = self._get_az_ebackup_map_list_from_workflow()
        self._check_ebackup_server(az_ebackup_map_list)

    def _get_az_ebackup_map_list_from_workflow(self):
        az_ebackup_setting_url = f"{self.base_url}/azebackup_settings"
        logger.info(f"Start to get azebackup_settings, url is {az_ebackup_setting_url}.")
        return self._get_az_ebackup_info(az_ebackup_setting_url)

    def _check_ebackup_server(self, az_ebackup_map_list):
        ebackup_server_check_conf_dic = self.params_store.get_check_config("CheckEbackupServer")
        az_check_conf_dic = self.params_store.get_check_config("CheckAZ")
        existence_az_dict = self._get_existence_business_az_info()
        for az_ebackup_map in az_ebackup_map_list:
            logger.info(f"The az and ebackup server map info:{az_ebackup_map}.")
            ebackup_mg_ip = az_ebackup_map["eBackupIp"]
            ebackup_az = az_ebackup_map["AZName"]
            self._handle_check_az_result(existence_az_dict, ebackup_az, az_check_conf_dic)
            self._handle_check_ebackup_ma_ip_result(ebackup_mg_ip, ebackup_server_check_conf_dic)

    def _handle_check_az_result(self, existence_az_dict, az_id, check_config_dic):
        if az_id in existence_az_dict:
            result = CheckResult(itemname_ch=check_config_dic.get("itemname_ch"),
                                 itemname_en=check_config_dic.get("itemname_en"),
                                 status="success")
        else:
            result = CheckResult(itemname_ch=check_config_dic.get("itemname_ch"),
                                 itemname_en=check_config_dic.get("itemname_en"),
                                 status="failure",
                                 error_msg_cn=check_config_dic.get("error_msg_cn").format(az_id=az_id),
                                 error_msg_en=check_config_dic.get("error_msg_en").format(az_id=az_id),
                                 suggestion_cn=check_config_dic.get("suggestion_cn"),
                                 suggestion_en=check_config_dic.get("suggestion_en"))
        self.check_results.append(result)

    def _check_server_accessable(self, manager_ip):
        if manager_ip in self.workflow_ip_list:
            return False
        url = f"https://{manager_ip}:8088"
        try:
            rsp = requests.post(url, headers={'Accept': 'application/json'}, verify=False, timeout=30)
        except Exception as err:
            logger.warning(f"Request {url} failed, err_msg:{err}.")
            return False
        redirect_history_list = rsp.history
        if not redirect_history_list:
            return False
        redirect_url = redirect_history_list[len(redirect_history_list) - 1].headers['Location']
        if "ebackup" in redirect_url.lower():
            return True
        return False

    def _handle_check_ebackup_ma_ip_result(self, ebackup_mg_ip, check_config_dic):
        if self._check_server_accessable(ebackup_mg_ip):
            result = CheckResult(itemname_ch=check_config_dic.get("itemname_ch"),
                                 itemname_en=check_config_dic.get("itemname_en"),
                                 status="success")
        else:
            result = CheckResult(itemname_ch=check_config_dic.get("itemname_ch"),
                                 itemname_en=check_config_dic.get("itemname_en"),
                                 status="failure",
                                 error_msg_cn=check_config_dic.get("error_msg_cn").format(ip=ebackup_mg_ip),
                                 error_msg_en=check_config_dic.get("error_msg_en").format(ip=ebackup_mg_ip),
                                 suggestion_cn=check_config_dic.get("suggestion_cn"),
                                 suggestion_en=check_config_dic.get("suggestion_en"))
        self.check_results.append(result)

    def _get_az_ebackup_info(self, azebackup_settings_url):
        rsp = requests.get(azebackup_settings_url, headers=self.headers, verify=False, timeout=30)
        self._check_rsp_when_req_url(rsp)
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode != 0:
            raise Exception(f"Get azebackup_settings failed, error code:{errcode}, description: {description}.")
        az_ebackup_mapinfo = rsp.json()["data"][0]["EBACKUPINFO"]
        if not az_ebackup_mapinfo:
            raise Exception("az_ebackup_mapinfo is None.")
        az_ebackup_map_list = json.loads(az_ebackup_mapinfo)["data"]
        if not isinstance(az_ebackup_map_list, list):
            raise Exception("az_ebackup_map_list is not list.")
        return az_ebackup_map_list

    @staticmethod
    def _check_rsp_when_req_url(rsp):
        if not isinstance(rsp, requests.Response):
            raise Exception("Request url failure.")
        if rsp.status_code != 200:
            raise Exception(f"Request url success, but http status code is not 200, status code:{rsp.status_code}.")

    def _get_existence_business_az_info(self):
        cps_node = CPSInfo(self.pod_id).get_cps_node()
        return CpsHelper(cps_node).get_business_az_info()
