# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util import ManageOneUtil
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import RegisterToMOInfo
from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.upgrade.constant import CSBS_SERVICE_NAME
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import auto_retry

logger.init("CSBS-VBS")


class RegisterMetric(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterMetric, self).__init__(project_id, pod_id, regionid_list)
        self.service_name = CSBS_SERVICE_NAME
        self.mo_util = ManageOneUtil()
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.param_tool = ParamsTools(project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to register meter metrics.")
        try:
            self._do_register_meter_metrics()
        except Exception as err:
            logger.error(f"Failed to register metrics, err_msg:{str(err)}.")
            return Message(500, HCCIException(645029))
        logger.info("Succeeded to register meter metrics.")
        return Message(200)

    def _do_register_meter_metrics(self):
        csbs_region_code_list, csbs_proxy_region_code_list = self._get_installed_csbs_region_code_list()
        for region_code in csbs_region_code_list:
            register_info = RegisterToMOInfo(region_code)
            metric_info = register_info.metric_info
            result = self._register_meter_metrics(self.pod_id, metric_info)
            if not result:
                raise Exception(f"Failed to register {metric_info.get('metrics')[0].get('name')} meter metrics, "
                                "please check.")
            if region_code not in csbs_proxy_region_code_list:
                continue

    @auto_retry(max_retry_times=5, delay_time=60)
    def _register_meter_metrics(self, pod_id, metric_info):
        return self.mo_util.register_meter_metrics(pod_id, metric_info)

    def _get_installed_csbs_region_code_list(self):
        all_region_code_list = self.param_tool.get_all_region_code()
        installed_csbs_region_code_list = []
        installed_csbs_proxy_region_code_list = []
        for region_code in all_region_code_list:
            logger.info(f"Start to obtain {self.service_name} service info from CMDB, region code:{region_code}.")
            cmdb_service_name = f"{self.service_name}_{region_code}"
            service_info_list = self.cmdb_util.get_cloud_service_info(
                region_code=region_code, index_name=self.service_name, name=cmdb_service_name)
            if not service_info_list:
                continue
            installed_csbs_region_code_list.append(region_code)
            service_info = service_info_list[0]
            for extend_info in service_info.get("extendInfos"):
                if extend_info.get("key") == "InstalledCSBS-VBS-Proxy" and extend_info.get("value") == "True" or \
                        extend_info.get("value") is True:
                    installed_csbs_proxy_region_code_list.append(region_code)
                    break
        logger.info(f"Succeeded to obtain installed CSBS Service region code, "
                    f"region code:{installed_csbs_region_code_list}, "
                    f"installed CSBS Proxy region code:{installed_csbs_proxy_region_code_list}.")
        return installed_csbs_region_code_list, installed_csbs_proxy_region_code_list
