#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import utils.common.log as logger

from utils.business.manageone_util import ManageOneUtil
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import RegisterToMOInfo
from plugins.CSBS.common.http_client import HttpClient
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.register_silvan import RegisterSilvan
from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import auto_retry


class RollbackRegisterOperation(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.csbs_id = "csbs"
        self.vbs_id = "vbs"
        self.register_silvan = RegisterSilvan(self.project_id, self.pod_id)
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.console_domain_name = self.param_tool.get_console_domain_name()
        self.register_info = RegisterToMOInfo(self.region_id, self.console_domain_name)
        self.http_client = HttpClient()
        self.mo_util = ManageOneUtil()
        self.param_util = ParamUtil()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._rollback_endpoints()
            self._delete_query_expenses_notification(self.csbs_id)
            self._delete_query_expenses_notification(self.vbs_id)
        except Exception as err:
            logger.error(f"Failed to rollback register operation, reason: {str(err)}.")
            return Message(500, HCCIException("645072", err))
        return Message(200)

    def _rollback_endpoints(self):
        logger.info("Start to rollback endpoints.")
        csbs_register_info = self.register_info.csbs_endpoint_info_zh
        csbs_register_info.update({'name': '云服务器备份'})
        result = self.register_silvan.update_endpoint(self.csbs_id, self.register_info.csbs_endpoint_info_zh,
                                                      self.register_info.csbs_endpoint_info_en)
        if not result:
            logger.error("Rollback endpoints failed!")
            raise HCCIException('645072')

    @auto_retry(max_retry_times=5, delay_time=60)
    def _delete_query_expenses_notification(self, service_id):
        result = self._delete_notifications(service_id, event_type="query_expenses")
        if result:
            logger.info("The notification is deleted successfully, "
                        f"service_id: {service_id}, event_type: query_expenses.")
            return True
        raise Exception(f"Failed to delete the notification, service_id: {service_id}, event_type: query_expenses.")

    def _delete_notifications(self, service_id, event_type="query_expenses"):
        logger.info("Start to delete notification.")
        headers = {'User-Agent': 'python-cinderclient',
                   'Content-Type': 'application/json',
                   'Accept': 'application/json;charset=utf8'}
        sc_nib_ip, sc_nib_port = self._get_sc_nbi_ip_and_port()
        vdc_admin_token = self.mo_util.get_vdc_admin_token(self.pod_id)
        headers.update({'x-auth-token': vdc_admin_token})
        sc_domain = self._get_sc_domain()
        headers.update({'host': sc_domain})
        self.http_client.headers = headers

        url01 = f'https://{sc_nib_ip}:{sc_nib_port}/rest/notification/v3.0/notifications?' \
                f'service_id={service_id}&cloud_resource_type={service_id}&' \
                f'event_type={event_type}&region_id={self.region_id}'
        _, body = self.http_client.get(url=url01)
        notifications = body.get('notifications', [])
        if not notifications:
            return True
        notification_id = notifications[0].get("id")
        url02 = f'https://{sc_nib_ip}:{sc_nib_port}/rest/notification/v3.0/notifications/{notification_id}'
        resp, _ = self.http_client.delete(url=url02)
        if resp.status_code == 204:
            logger.info("The notification is deleted successfully, "
                        f"service_id: {service_id}, event_type: {event_type}.")
            return True
        err_msg = f"Failed to delete the notification, response code: {resp.status_code}, " \
                  f"service_id: {service_id}, event_type: {event_type}." \
                  f"Solution: 1.Please check the request info; " \
                  f"2.Check to see if the SC floating IP address is correct or if the preset user exists."
        logger.error(err_msg)
        raise Exception(err_msg)

    def _get_sc_domain(self):
        logger.info("Start to obtain sc_domain.")
        global_domain_name = ParamsTools(self.project_id).get_global_domain(self.region_id)
        sc_domain = f'sc.{global_domain_name}'
        logger.info(f"The sc_domain is: {sc_domain}.")
        return sc_domain

    def _get_sc_nbi_ip_and_port(self):
        project_conditions = ProjectApi().get_project_conditions(self.project_id)
        if project_conditions.get("PrimaryRegion"):
            param_keylist = ["ManageOne_moTenantFloatIp", "ManageOne_webPort"]
        else:
            param_keylist = ["moTenantFloatIp", "webPort"]
        sc_nbi_ip = self.param_util.get_param_value(self.project_id, "ManageOne_public_params", param_keylist[0],
                                                    user_param_key=param_keylist[0])
        if not sc_nbi_ip:
            sc_nbi_ip = self.param_util.get_param_value(self.project_id, "ManageOne", param_keylist[0],
                                                        user_param_key=param_keylist[0])
        port = self.param_util.get_param_value(self.project_id, "ManageOne", param_keylist[1],
                                               user_param_key=param_keylist[1])
        if not port:
            port = self.param_util.get_param_value(self.project_id, "ManageOne", param_keylist[1],
                                                   user_param_key=param_keylist[1])
        if not (sc_nbi_ip and port):
            raise Exception("Get sc nbi ip or port failed.")
        return sc_nbi_ip, port
