import json
import random
import socket

import requests
import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.param_check import ping

from plugins.CSBS.common.karborproxy.agent_proxy_param import AgentProxyDPAInfos

logger.init("DPA Login Checking")
DEFAULT_REQUEST_TIMEOUT = 120


class DPALoginCheck:
    def __init__(self, project_id):
        self.agent_proxy = AgentProxyDPAInfos(project_id, project_id)
        self.headers = {
            "content-type": "application/json",
            "Accept": "application/json;version=2.3;charset=UTF-8"
        }

    def check_account_info(self):
        """
        通过登录来检查三类账号填写是否正确。
        """

        logger.info("Checking the dpa_sec_admin account")
        # 检查dpa_sec_admin账户可登陆性
        dpa_sec_admin_list = self.agent_proxy.dpa_sec_admin.split(',')
        dpa_sec_admin_pwd_list = self.agent_proxy.dpa_sec_admin_pwd.split(',')
        self.check_account_login(1, dpa_sec_admin_list, dpa_sec_admin_pwd_list)
        logger.info("Checking the dpa_general_user account")
        # 检查dpa_general_user账户可登陆性
        dpa_general_user_list = self.agent_proxy.dpa_general_user.split(',')
        dpa_general_user_pwd_list = self.agent_proxy.dpa_general_user_pwd.split(',')
        self.check_account_login(3, dpa_general_user_list, dpa_general_user_pwd_list)

        logger.info("Checking the dpa_admin account")
        # 检查dpa_admin账户可登陆性
        dpa_admin_list = self.agent_proxy.dpa_admin.split(',')
        dpa_admin_pwd_list = self.agent_proxy.dpa_admin_pwd.split(',')
        self.check_account_login(0, dpa_admin_list, dpa_admin_pwd_list)

    def check_account_login(self, role, account_list, pwd_list):
        """
        检查账号能否登录,登录成功则设置headers，失败则报错。
        Args:
            role:
            account_list:
            pwd_list:

        Returns:

        """
        if not self.agent_proxy.dpa_cluster_address_list or not account_list or not pwd_list:
            logger.error(f'The login checking args is {self.agent_proxy.dpa_cluster_address_list}, {account_list}')
            raise HCCIException('check account login has error args, please check.')
        for cluster_url, account_name, pwd in zip(self.agent_proxy.dpa_cluster_address_list, account_list, pwd_list):
            # 检查ip是否有效
            cluster_url_list = cluster_url.split(',')
            for url in cluster_url_list:
                self.can_arrived(url)
            login_url = random.choice(cluster_url_list)
            self.login_dpa(login_url, account_name, pwd)
            # 检查用户角色与用户类型是否匹配
            self.judge_role(login_url, role)
            cluster_url_list.remove(login_url)
            if len(cluster_url_list) == 0:
                continue
            for url in cluster_url_list:
                self.socket_test(url)

    def login_dpa(self, url, user, pwd):
        """
        登录dpa账号
        Args:
            url: 管理平面url
            user:
            pwd:

        Returns:

        """
        login_url = f"https://{url}/oauths/access_token"
        self.headers.update({
            "Referer": f'https://{url}',
        })
        # 构造登录请求体
        req = {
            "userName": user,
            "isEnc": False,
            "userPass": pwd,
            "validPwdExpire": "true"
        }
        res = requests.post(login_url, json=req, headers=self.headers,
                            timeout=DEFAULT_REQUEST_TIMEOUT,
                            verify=False)
        if res.status_code != 200:
            raise HCCIException(
                f"When a {user} logs in to the {url}, the returned status code is {res.status_code} but not 200..")
        result = json.loads(res.content)
        if result.get("status") == "success":
            # 设置请求头的cookies
            self.set_cookie_to_headers(res.cookies.get_dict())
        else:
            logger.error(f'The response status is {result.get("status")}')
            raise Exception(f"The {user} login {url} is successful, but the token fails to be set. Please check..")

    def set_cookie_to_headers(self, cookies_dict):
        csrftoken = cookies_dict.get("csrftoken")
        self.headers.update({"x-csrftoken": csrftoken})
        str_cookie = []
        for key, value in cookies_dict.items():
            str_cookie.append("{}={}".format(key, value))
        self.headers.update({"cookie": ";".join(str_cookie)})

    def judge_role(self, url, role):
        """
        Check whether the role matches the role type.
        Args:
            url:
            role:

        Returns:

        """
        person_url = f"https://{url}/oauths/profile"
        res = requests.get(
            person_url,
            timeout=DEFAULT_REQUEST_TIMEOUT,
            headers=self.headers,
            verify=False)
        if res.status_code != 200:
            raise HCCIException("Failed to query the type information of the account.")
        try:
            web_role = json.loads(res.content).get('responseData').get(
                'userInfo').get('roleType')
        except Exception as exception:
            raise HCCIException("The type of the account is not found.") from exception
        if role != web_role:
            raise Exception(
                "The account does not match the group to which the account belongs.")

    @staticmethod
    def can_arrived(url):
        """
        Check whether the IP address of the DPA is correct.
        Args:
            url:dpa填写的url

        Returns:

        """
        if len(url.split(':')) != 2:
            raise Exception(f"{url} is error. Please check.")
        dpa_ip = url.split(':')[0]
        result = ping(dpa_ip)
        if not result:
            raise Exception(f"{dpa_ip} IP address unreachable. Please check.")

    @staticmethod
    def socket_test(domain_url):
        if len(domain_url.split(":")) != 2:
            raise HCCIException(f"{domain_url} is error. Please check.")
        test_ip, port = domain_url.split(":")
        test_connect = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        result = test_connect.connect_ex((test_ip, int(port)))
        if result == 0:
            logger.info(f"The {domain_url} is normal")
        else:
            raise HCCIException(f"the {domain_url} is abnormal, Please check")
