import os
import threading

import utils.common.log as logger
from utils.common.ssh_util2 import Ssh as ssh_util

from plugins.CSBS.common.constant import TEMP_CERT_PATH
from plugins.CSBS.common.util import auto_retry

SSH_TIMEOUT = 60 * 60 * 2
KEEP_ALIVE = 30
EXEC_TIMEOUT = 60


class SshClient(object):

    def __init__(self):
        self.r_lock_outer = self.r_lock_inner = threading.RLock()

    @staticmethod
    def create_ssh_client(node_ip, username, passwd, port=22,
                          timeout=SSH_TIMEOUT, pub_key=None,
                          status_prompt=True) -> ssh_util:
        # 通过agent连接ssh
        return ssh_util.ssh_create_client(ip=node_ip, username=username, passwd=passwd, port=port, timeout=timeout,
                                          pub_key=pub_key,
                                          status=status_prompt)

    @staticmethod
    def is_expect_recv(expect_list, recvstr):
        for expect in expect_list:
            if recvstr.__contains__(expect):
                return True
        return False

    @staticmethod
    def ssh_exec_command_return(ssh_client, cmds, timeout=EXEC_TIMEOUT):
        """单独执行命令行，确保执行成功，并返回执行结果（拆分为列表）"""
        new_cmd = f"{cmds};echo last CMD_RESULT: $?"
        result = ssh_util.ssh_exec_command_return_list(ssh_client=ssh_client, cmds=new_cmd, timeout=timeout)
        if not result or not isinstance(result, list):
            return result
        ret = []
        for line in result:
            if line.__contains__(";echo last CMD_RESULT: $?"):
                continue
            ret.append(line)
        return ret

    @staticmethod
    def ssh_send_command_expect(ssh_client, cmds, expect_str="#", timeout=EXEC_TIMEOUT * 60):
        """执行一个命令,等待一段时间之后直接返回，用来处理无法输出结束符的情况， 带交互式"""
        return ssh_util.ssh_send_command(ssh_client=ssh_client, cmds=cmds, expect=expect_str, timeout=timeout)

    @staticmethod
    def ssh_close(ssh_client):
        ssh_util.ssh_close(ssh_client)

    @staticmethod
    def is_ssh_cmd_executed(listout):
        """判断上一个命令是否被系统执行了;检查执行命令的结果：True：成功    False：失败"""
        try:
            for line in listout:
                if line.__contains__("last CMD_RESULT: 0"):
                    return True
            return False
        except Exception as ex:
            err_msg = 'judge command  executed or not  failed'
            logger.error(err_msg)
            raise Exception(err_msg) from ex

    def put(self, host, user, passwd, srcdir, destdir, port=22):
        """上传文件"""
        with self.r_lock_outer:
            ssh_util.put_file(host=host, user=user, passwd=passwd, source=srcdir, destination=destdir, port=port)

    @staticmethod
    def scp_file(from_node, to_node, source_file_path='.', to_path='.'):
        """构建scp命令，完成远程拷贝"""
        file_name = os.path.basename(source_file_path)
        logger.info(f'Get scp file name: {file_name}.')
        if not file_name:
            raise Exception('get file name failed.')
        if not os.path.exists(TEMP_CERT_PATH):
            os.mkdir(TEMP_CERT_PATH)
        temp_file_path = f"{TEMP_CERT_PATH}/{file_name}"
        logger.info(f'Get file from {from_node.node_ip}.')
        ssh_util.get_file(from_node.node_ip, from_node.user, from_node.user_pwd, source=source_file_path,
                          destination=temp_file_path)
        logger.info(f'Put file to {to_node.node_ip}.')
        ssh_util.put_file(to_node.node_ip, to_node.user, to_node.user_pwd, source=temp_file_path,
                          destination=to_path)
        if os.path.exists(temp_file_path):
            os.remove(temp_file_path)
        return True

    def get_ssh_client_user_sudo_root(self, node):
        """从user登录后，使用sudo su root切换到root

        """
        ssh_client = self.create_ssh_client(node.node_ip, node.user, node.user_pwd)
        self.ssh_exec_command_return(ssh_client, "TMOUT=0")
        self.ssh_send_command_expect(ssh_client, "sudo su", expect_str="password for root:")
        result = self.ssh_send_command_expect(ssh_client, node.root_pwd)
        if self.failed_to_return(result, "# ", ssh_client):
            raise Exception("Sudo su root failed.")
        self.ssh_exec_command_return(ssh_client, "TMOUT=0")
        return ssh_client

    def get_ssh_client_user_su_root(self, node):
        """从user登录后，使用su root切换到root

        """
        ssh_client = self.create_ssh_client(node.node_ip, node.user, node.user_pwd)
        self.ssh_exec_command_return(ssh_client, "TMOUT=0")
        self.ssh_send_command_expect(ssh_client, "su", expect_str="Password: ")
        result = self.ssh_send_command_expect(ssh_client, node.root_pwd)
        if self.failed_to_return(result, "# ", ssh_client):
            raise Exception("Su root failed.")
        self.ssh_exec_command_return(ssh_client, "TMOUT=0")
        return ssh_client

    def get_ssh_client(self, node, sudo_type="sudo"):
        """
        :param node:
        :param sudo_type: 切换账号的方式 su/sudo
        :su 使用 su root切換root
        :sudo 使用 sudo su root切換 root
        :return:
        """
        try:
            if sudo_type.lower() == "sudo":
                return self.get_ssh_client_user_sudo_root(node)
            else:
                return self.get_ssh_client_user_su_root(node)
        except Exception as err:
            logger.warn(f'Normal user login node:{node.node_ip} failed, '
                        f'return message:{str(err)}, '
                        'use root relogin directly.')
            ssh_client = self.create_ssh_client(node.node_ip, "root", node.root_pwd)
            self.ssh_exec_command_return(ssh_client, "TMOUT=0")
            return ssh_client

    def failed_to_return(self, result_list, expect_str, ssh_client):
        for result in result_list:
            if self.is_expect_recv([expect_str], result):
                return False
        self.ssh_close(ssh_client)
        return True

    def success_to_return(self, result_list, expect_str):
        for result in result_list:
            if self.is_expect_recv([expect_str], result):
                return True
        return False

    def exec_cmd_can_retry(self, node, cmd, timeout=60, max_retry_times=5, delay_time=60):
        @auto_retry(max_retry_times=max_retry_times, delay_time=delay_time)
        def exec_cmd():
            result = self.ssh_exec_command_return(ssh_client, cmd, timeout=timeout)
            logger.info(f"The cmd: {cmd}, exec result: {result}.")
            if not self.is_ssh_cmd_executed(result):
                raise Exception(f"Failed to execute [{cmd}] on {node.ip} node.")

        ssh_client = None
        try:
            ssh_client = self.get_ssh_client(node)
            exec_cmd()
        except Exception as err:
            logger.error(f"Failed to execute [{cmd}], err_msg: {err}.")
            raise err
        finally:
            if ssh_client:
                self.ssh_close(ssh_client)
