import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_conditions
from utils.business.project_util import ProjectApi
from utils.business.vm_util import PublicServiceOM
from utils.common.exception import HCCIException

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.constant import KARBOR_IMAGE_NAME
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.os_util import OSUtil

logger.init("Karbor create vm")


class CbsCreateVM(node_installer.ThreadInstaller):
    def __init__(self, project_id, pod_id):
        self.pod_id = pod_id
        self.project_id = project_id
        self.karbor_util = KarborUtil(project_id, pod_id)
        node_list = self.karbor_util.get_karbor_node_list()
        super().__init__(node_list)
        self.pub_om = PublicServiceOM()
        self.network_name = "external_om"
        self.csha = self.__is_csha()
        self.vm_names = {}
        self.vm_params = {}
        self._init_params()
        self.file_name = OSUtil.get_kvm_image_file(self.project_id)

    def _init_params(self):
        ip_list = self.karbor_util.get_karbor_ip_list()
        group_names = ["CSBS", "CSBSdr"] if self.csha else ["CSBS", "CSBS2", "CSBS3"]
        self.vm_names = dict(zip(ip_list, group_names))
        params = ParamUtil()
        self.vm_params.setdefault("cpu", params.get_value_from_cloudparam(self.pod_id, "CSBS-VBS", "CSBS_cpu"))
        self.vm_params.setdefault("mem", params.get_value_from_cloudparam(self.pod_id, "CSBS-VBS", "CSBS_mem"))

    def install_thread(self, node):
        """
        创建虚拟机
        """
        logger.info(f"Start to create vm. Current CSHA state is {self.csha}")
        vm_name = self.vm_names.get(node.node_ip)
        cpu = self.vm_params.get('cpu')
        mem = self.vm_params.get('mem')
        logger.info(f"VM params, cpu: {cpu}, mem: {mem}, vm name: {vm_name}, ip: {node.node_ip}")
        if self.csha:
            if "CSBSdr" in vm_name:
                availability_zone = "dr-manage-az"
                volume_type = "dr.manager.fsa.csg"
            else:
                availability_zone = "manage-az"
                volume_type = "manager.fsa.csg"

            self.pub_om.create_servers_with_group(self.pod_id, "Service", vm_name,
                                                  cpu, mem, 0,
                                                  [node.node_ip],
                                                  self.network_name, vm_image_name=KARBOR_IMAGE_NAME, sys_disk=150,
                                                  image_file_name=self.file_name, availability_zone=availability_zone,
                                                  image_template_type=0, volume_type=volume_type,
                                                  ip_list_ex=None, network_name_ex=None)
        else:
            self.pub_om.create_servers_with_group(self.pod_id, "Service", vm_name,
                                                  cpu, mem, 0, [node.node_ip],
                                                  self.network_name, vm_image_name=KARBOR_IMAGE_NAME, sys_disk=150,
                                                  image_file_name=self.file_name)
        logger.info(f"{node.node_ip} vm create success.")
        self.install_result[node.node_ip] = True

    def rollback_thread(self, node):
        """
        删除虚拟机
        """
        vm_name = self.vm_names.get(node.node_ip)
        logger.info(f"Start to delete vm {node.node_ip}, vm_name: {vm_name}")
        PublicServiceOM.delete_servers_with_group(self.pod_id, "Service", vm_name,
                                                  [node.node_ip], self.network_name)
        logger.info(f"{node.node_ip} vm delete success.")
        self.rollback_result[node.node_ip] = True

    def __is_csha(self):
        project_api = ProjectApi()
        replication_dict = project_api.get_project_ha_dr_status(self.project_id)
        csha = replication_dict['CSHA']
        if replication_dict['RegionConHA'] == 1:
            csha = 1
        if not isinstance(csha, int):
            raise HCCIException(640067)
        return csha
