# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.message import Message

from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import check_url_param

logger.init("CSBS")


class ConfigIAMAccount(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(ConfigIAMAccount, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = "CSBS-VBS"
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.param_util = ParamUtil()
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to config IAM account.")
        try:
            self._config_iam_account()
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, f"Failed to config IAM account, err_msg:{err}")
        logger.info("Succeed to config IAM account.")
        return Message(200)

    def _config_iam_account(self):
        self._config_op_service_account_for_karbor()
        self._config_kms_endpoint()

    def _config_op_service_account_for_karbor(self):
        op_service_account = self.iam_util.get_iam_account_info()
        logger.info(f"Succeed to get IAM account: {op_service_account.account_name}.")
        # 如果是默认密码则不用给karbor做配置
        if op_service_account.is_default:
            return
        self.karbor_util.set_karbor_account(op_service_account.account_name, op_service_account.account_pwd)
        logger.info("Successfully configured account to karbor.")

    def _config_kms_endpoint(self):
        logger.info(f"Start to config kms endpoint for karbor node.")
        karbor_client = SshClient()
        op_svc_account = self.iam_util.get_iam_account_info(account_type="op_svc_csbs")
        kms_endpoint = self.param_util.get_param_value(self.pod_id, self.service_name, "kms_address")
        if not kms_endpoint:
            raise Exception("Failed to obtain the kms_endpoint.")

        # 校验endpoint
        check_url_param(kms_endpoint)

        logger.info(f"Start to set_kms_config, kms endpoint: {kms_endpoint}, "
                    f"IAM op_svc account name: {op_svc_account.account_name}.")
        cmd = f"source /opt/huawei/dj/inst/utils.sh;set_kms_config --endpoint {kms_endpoint}"
        ssh_client = karbor_client.get_ssh_client(self.karbor_util.get_karbor_node_list()[0])
        karbor_client.ssh_send_command_expect(ssh_client, cmd, "account", 30)
        karbor_client.ssh_send_command_expect(ssh_client, op_svc_account.account_name, "password", 30)
        karbor_client.ssh_send_command_expect(ssh_client, op_svc_account.account_pwd, "password", 30)
        result = karbor_client.ssh_send_command_expect(ssh_client, op_svc_account.account_pwd, "successfully", 30)
        logger.info(result)
        karbor_client.ssh_close(ssh_client)
        if karbor_client.failed_to_return(result, "successfully", ssh_client):
            raise Exception("Failed to config kms endpoint for karbor node.")
        logger.info("Succeed to config kms endpoint for karbor node.")
