#!/bin/bash
if [[ ${UID} -eq 0 ]];then
    sudo -u openstack bash "$(readlink -f "$0")" "$@";exit $?
elif [[ ${UID} -ne 51001 ]];then
    echo "Please switch to user(openstack) for run." && exit 1
fi
source /opt/huawei/dj/inst/utils.sh
export LD_LIBRARY_PATH=/opt/huawei/dj/bin/gaussdb/app/lib:${LD_LIBRARY_PATH}
export PATH=/opt/huawei/dj/bin/gaussdb/app/bin:${PATH}
export GAUSSHOME=/opt/huawei/dj/bin/gaussdb/app
export PGDATA=/opt/huawei/dj/bin/gaussdb/data
export PGSSLCERT=/opt/huawei/dj/bin/gaussdb/cert/client.crt
export PGSSLKEY=/opt/huawei/dj/bin/gaussdb/cert/client.key
export PGSSLROOTCERT=/opt/huawei/dj/bin/gaussdb/cert/ca-cert.pem
export PGSSLMODE=require
export GAUSSLOG=/var/log/huawei/dj/services/system/gaussdb
export LOG_TAG="gaussdb_helper"
START_FAIL_RECORD="/opt/huawei/dj/bin/gaussdb/data/cert.startGS.fail"
BACKUP_PATH="/opt/huawei/djbackup/db/rmandatabackup"
# 从数据库中查出的数据库状态值
gs_s_normal="Normal"                         # 表示数据库双机状态正常。
gs_s_unknown="Unknown"                       # 表示未知的数据库双机状态。
gs_s_needrepair="Needrepair"                 # 表示数据库需要重建。
gs_s_starting="Starting"                     # 表示数据库正在启动。
gs_s_waiting="Waiting"                       # 表示正在等待主机降备（备机才有此状态）。
gs_s_demoting="Demoting"                     # 表示主机正在进行降备过程（主机才有此状态）。
gs_s_promoting="Promoting"                   # 表示备机正在进行升主过程（备机才有此状态）。
gs_s_rebuilding="Rebuilding"                 # 表示备机正在进行重建过程（备机才有此状态）。
# 从数据库中查出的数据库角色值
gs_r_primary="Primary"                       # 表示本端数据库作为主机使用。
gs_r_standby="Standby"                       # 表示本端数据库作为备机使用。
gs_r_cstandby="CascadeStandby"               # 表示本端数据库作为级联备机使用。
gs_r_pending="Pending"                       # 表示本端数据库处于等待状态，此时可等待通知命令使其成为主机或备机。
gs_r_normal="Normal"                         # 表示本端数据库作为单机使用。
gs_r_unknown="UNKNOWN"                       # 表示本端数据库以未知的方式使用。
# 从数据库中查出的详细重建原因
gs_d_walremoved="WALsegmentremoved"          # 表示日志段已删除
gs_d_sysIDnotmatched="Systemidnotmatched"    # 表示数据不是同源的，即双机间的数据目录不是同一个数据库初始化创建的
r_success=0
r_failure=1
TIMES_TO_REBUILD=3
# 启动数据库时，指定角色的值
gs_c_pending="pending"                       #
gs_c_primary="primary"                       #
gs_c_standby="standby"                       #
gs_c_normal="normal"                         #
# 返回码
db_normal=0           #   正常运行
db_abnormal=1         #   运行异常
db_stopped=2          #   停止
db_starting=4         #   正在启动
db_primary=6          #   主正常运行
db_standby=7          #   备正常运行
db_activating=8       #   正在升主
db_deactivating=9     #   正在降备
db_notsupported=10    #   动作不存在
db_repairing=11       #   正在修复
######################################################################
#   FUNCTION   : isDBRunning
#   DESCRIPTION: 数据库是否正在运行
######################################################################
function isDBRunning()
{
    # 获取数据运行状态，判断数据库是否正在运行
    dbinfo=$(gs_ctl status)
    # 数据库正在运行
    echo "${dbinfo}" | grep "server is running" ; retVal=$?
    if [[ ${retVal} -eq ${r_success} ]] ; then
        return ${r_success}
    fi
    # 数据库不在运行中
    return ${r_failure}
}
######################################################################
#   FUNCTION   : restartToPrimary
#   DESCRIPTION: 将数据库重启为主的操作
######################################################################
function restartToPrimary()
{
    restartToState "${gs_r_primary}" "${gs_c_primary}" ; retVal=$?
    recordStartResult ${retVal}
    return ${retVal}
}
######################################################################
#   FUNCTION   : recordStartResult
#   DESCRIPTION: 记录启动结果，主要是对失败的结果进行记录
#                对于备数据库，一直没能启动成功，需要考虑重建
#####################################################################
function recordStartResult()
{
    failCount=0
    # 启动失败，则需要记录失败的次数
    touch ${START_FAIL_RECORD}
    if [[ "$1" -ne ${r_success} ]];then
        failCount=$(cat ${START_FAIL_RECORD})
        failCount=$((failCount+1))
        log_error "DB start failure for ${failCount} times."
    fi
    echo "${failCount}" > ${START_FAIL_RECORD}
    return ${r_success}
}
######################################################################
#   FUNCTION   : OMM_EXIT
#   DESCRIPTION: 退出函数，用于退出脚本，并输出日志
######################################################################
function checkLocalRole()
{
    # 获取数据库状态
    dbinfo=$(gs_ctl query)
    if ! echo "${dbinfo}" | grep "local_role" | grep -E "$1"; then
        log_error "[checkLocalRole] get role[$1] failure. [${dbinfo}]"
        return ${r_failure}
    fi
    return ${r_success}
}
######################################################################
#   FUNCTION   : restartToState
#   DESCRIPTION: 重启数据库到目标状态的操作
#   需要输入重启后的检查状态，和重启后的状态，
######################################################################
function restartToState()
{
    retVal=0
    log_info "start to restart db for $2, and check state is $1."
    # 停止前，先检查当前数据库是否已经停止
    if isDBRunning; then
        # 停止数据库，停止失败，则强制停止
        gs_ctl stop -m fast || gs_ctl stop -m immediate
    fi
    # 启动数据库为目标状态
    if [[ "$2" == "${gs_c_normal}" ]]; then
        db_pid=$(pgrep /opt/huawei/dj/bin/gaussdb/app/bin/gs_ctl.*start)
        if [[ -z "${db_pid}" ]];then
            gs_ctl start -w -D "${PGDATA}"; retVal=$?
        else
            log_info "[restartToState] A start task is running."
        fi
    else
        db_pid=$(pgrep /opt/huawei/dj/bin/gaussdb/app/bin/gs_ctl.*start)
        if [[ -z "${db_pid}" ]];then
            gs_ctl start -w -M "$2" -D "${PGDATA}"; retVal=$?
        else
            log_info "[restartToState] A start task is running."
        fi
    fi
    # 数据库未启动
    if [[ ${retVal} -ne ${r_success} ]]; then
        log_error "[restartToState] call (gs_ctl start [-M $2]) failure[${retVal}]."
        return ${r_failure}
    fi
    # 启动后，需要检查当前数据库是否已经启动
    if ! checkLocalRole "$1" ; then
        log_error "[restartToState] call (gs_ctl start [-M $2]) success, but db still not [$1]."
        return ${r_failure}
    fi
    log_info "[restartToState] success to restart db for $2, and check state is $1."
    return ${r_success}
}
######################################################################
#   FUNCTION   : notifyToPrimary
#   DESCRIPTION: 通知数据库角色转换为主的操作
######################################################################
function notifyToPrimary()
{
    # 通知数据库，将状态切换为主用
    gs_ctl notify -w -M ${gs_c_primary} -D "${PGDATA}";retVal=$?
    # 通知失败
    if [[ ${retVal} -ne ${r_success} ]]; then
        log_error "[notifyToPrimary] call [gs_ctl notify -M ${gs_c_primary}] failure[${retVal}]."
        return ${r_failure}
    fi
    return ${r_success}
}
######################################################################
#   FUNCTION   : get DB state
#   DESCRIPTION: 获取数据库状态，从数据库的查询结果中分离出各项信息
#                local_role, db_state, detail_information, peer_role
######################################################################
function getDBState()
{
    dbinfo="$1"
    # 目前比较简单的做法，直接从结果中转换，没有再做其他处理
    local_role=$(echo "${dbinfo}" | grep "local_role" | awk -F':' '{print $NF}' | sed 's/ //g' | sed -n '1p')
    db_state=$(echo "${dbinfo}" | grep "db_state" | awk -F':' '{print $NF}' | sed 's/ //g' | sed -n '1p')
    detail_information=$(echo "${dbinfo}" | grep "detail_information" | awk -F':' '{print $NF}' | sed 's/ //g' | sed -n '1p')
    peer_role=$(echo "${dbinfo}" | grep "peer_role" | awk -F':' '{print $NF}' | sed 's/ //g' | sed -n '1p')
    return ${r_success}
}
######################################################################
#   FUNCTION   : StaToPri
#   DESCRIPTION: 备升主的操作， switchover，失败后尝试强制升主
######################################################################
function StaToPri()
{
    # 备升主，尝试正常切换 switchover
    gs_ctl switchover -t 90 -D "${PGDATA}"; retVal=$?
    # 切换失败，命令执行失败或是状态没有切换为主用
    if [[ ${retVal} -ne ${r_success} ]] || ! checkLocalRole "${gs_r_primary}"; then
        log_error "[notifyToStandby] call [gs_ctl switchover -t 90] failure[${retVal}], try failover."
        # 是否需要尝试多几次后，再试呢？
        StaForceToPri; retVal=$?
        return ${retVal}
    fi
    # 切换成功
    return ${r_success}
}
######################################################################
#   FUNCTION   : StaForceToPri
#   DESCRIPTION: 备强制升主的操作， failover
######################################################################
function StaForceToPri()
{
    # 备升主，强制切换 failover
    gs_ctl failover -D "${PGDATA}"; retVal=$?
    # 切换失败，命令执行失败或是状态没有切换为主用
    if [[ ${retVal} -ne ${r_success} ]] || ! checkLocalRole "${gs_r_primary}"; then
        log_error "[notifyToStandby] call [gs_ctl failover] failure[${retVal}], It can't be helped."
        return ${r_failure}
    fi
    return ${r_success}
}
######################################################################
#   FUNCTION   : restartToStandby
#   DESCRIPTION: 将数据库重启为备的操作
######################################################################
function restartToStandby()
{
    restartToState "${gs_r_standby}" "${gs_c_standby}" ; retVal=$?
    recordStartResult ${retVal}
    return ${retVal}
}
######################################################################
#   FUNCTION   : notifyToStandby
#   DESCRIPTION: 通知数据库角色转换为备的操作
######################################################################
function notifyToStandby()
{
    # 通知数据库，将状态切换为备
    gs_ctl notify -w -M ${gs_c_standby} -D "${PGDATA}"; retVal=$?
    # 通知失败
    if [[ ${retVal} -ne ${r_success} ]]; then
        log_error "[notifyToStandby] call [gs_ctl notify -M ${gs_c_standby}] failure[${retVal}]."
        return ${r_failure}
    fi
    return ${r_success}
}
######################################################################
#   FUNCTION   : restartToPending
#   DESCRIPTION: 将数据库重启为Pending角色的操作
######################################################################
function restartToPending()
{
    restartToState "${gs_r_pending}" "${gs_c_pending}" ; retVal=$?
    recordStartResult ${retVal}
    return ${retVal}
}
######################################################################
#   FUNCTION   : isNeedRebuild
#   DESCRIPTION: 判断能否进行重建
#####################################################################
function isNeedRebuild()
{
    buildStr=$(gs_ctl querybuild -D "${PGDATA}")
    # 如果当前正在进行重建中，则还不需要处理
    if echo "${buildStr}" | grep "Building" > /dev/null; then
        log_warn "[isNeedRebuild] DB is rebuilding now. [${buildStr}]"
        return ${db_repairing}
    fi
    # 当前没有在重建，则判断数据库启动失败是否达到次数
    touch ${START_FAIL_RECORD}
    failCount=$(cat ${START_FAIL_RECORD})
    if [[ "${failCount}" -gt "${TIMES_TO_REBUILD}" ]]; then
        log_error "[isNeedRebuild] DB start failure for ${failCount} times, need repair. [${buildStr}]"
        return ${db_abnormal}
    fi
    return ${db_normal}
}
######################################################################
#   FUNCTION   : PriToSta
#   DESCRIPTION: 主降备的操作，实际就是重启为备
######################################################################
function PriToSta()
{
    restartToStandby; retVal=$?
    return ${retVal}
}
######################################################################
#   FUNCTION   : getStatus
#   DESCRIPTION: 数据库状态查询函数
#   脚本返回码 :
######################################################################
function getStatus()
{
    outResult=${r_success}                    # 返回的结果，用于在主备用情况下，暂存结果。
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_error "[getStatus] db no running now."
        # 如HA期待的数据库角色为主,插件认DB正确就应为主,此时启动为主
        if [[ "${runState}" == "active" ]]; then
            log_warn "[getStatus] HA wanted primary,DB no running,restart to primary!!!"
            restartToPrimary;retVal=$?
            return "${db_stopped}"
        fi
        # 数据库没有运行，有可能是启动不了，检查是否需要重建
        isNeedRebuild; retVal=$?
        [[ ${retVal} -eq ${db_normal} ]] || return ${retVal}
        return "${db_stopped}"
    fi
    # 获取数据库状态
    dbinfo=$(gs_ctl query -D "${PGDATA}")
    # 从数据库的查询结果中分离出各项信息: local_role, db_state, detail_information, peer_role. 前面四个变量会在getDBState赋值。
    getDBState "${dbinfo}"; retVal=$?
    # 根据数据库的角色进行判断，正常情况下，只会出现 Pending, Primary, Standby 和空（即获取失败）
    case "${local_role}" in
        # Pending状态下的数据库，可以直接状态变更，所以返回期望状态的反状态
        "${gs_r_pending}")
            log_warn "[getStatus] db current role[${local_role}], current ha role[${runState}]."
            return "${db_normal}"
            ;;
        # 以下两种状态，需要先判断是否正在重建
        "${gs_r_primary}")
            outResult=${db_primary}
            log_info "[getStatus] db current role[${local_role}], need further check."
            ;;
        "${gs_r_standby}")
            outResult=${db_standby}
            log_info "[getStatus] db current role[${local_role}], need further check."
            ;;
        # 以下角色不是双机会出现的角色，直接重启即可
        "${gs_r_normal}"|"${gs_r_cstandby}"|"${gs_r_unknown}"|*)
            log_error "[getStatus] db current role[${local_role}] error! [${dbinfo}]"
            return "${db_stopped}"
            ;;
    esac
    # 当前为HA的状态为未知时，数据库主备需要返回为正常
    [ "${runState}" == "unknown" ] && outResult=${db_normal}
    # 以下判断是在本端数据库角色为主、备、级联备的情况的判断，需要判断数据库的状态
    case "${db_state}" in
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "${gs_s_rebuilding}")
            log_warn "[getStatus] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${db_abnormal}" #HA advice when standby is repairing, return exception to it.
            ;;
        "${gs_s_demoting}")
            log_warn "[getStatus] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${db_deactivating}"
            ;;
        "${gs_s_waiting}"|"${gs_s_promoting}")
            log_warn "[getStatus] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${db_activating}"
            ;;
        "${gs_s_starting}")
            log_warn "[getStatus] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${db_starting}"
            ;;
        # 数据库需要重建，则需要考虑对端情况
        "${gs_s_needrepair}")
            # 重建状态不一定都是需要修复的
            # 高斯DB，能够进行重建的原因
            if echo "${detail_information}" | grep -E "${gs_d_walremoved}|${gs_d_sysIDnotmatched}"; then
                log_warn "[getStatus] db state[${db_state}] need repair now"
                log_info "current role[${local_role}]. [${dbinfo}]"
                # 如果数据库角色为主，则不先不进行其他处理
                if [ "${local_role}" == "Primary" ];then 
                    return "${db_activating}"
                else
                # 否则，返回需要修复
                    return "${db_abnormal}"
                fi
            else
                log_warn "[getStatus] db state[${db_state}] need repair now, but no need to repair"
                log_info "detail_information[${detail_information}]. [${dbinfo}]"
                return "${outResult}"
            fi
            ;;
        # 其他状态，直接重启跳转到需要的状态
        "${gs_s_normal}"|"${gs_s_unknown}"|*)
            log_info "[getStatus] db state[${db_state}] stable now, can goto want role."
            return "${outResult}"
            ;;
    esac
    # 数据库正常运行，返回当前状态
    return "${db_normal}"
}
######################################################################
#   FUNCTION   : doStart
#   DESCRIPTION: 双机数据库启动函数
######################################################################
function doStart()
{
    # 启动前，先检查当前数据库是否已经启动为Primary|standby|Pending
    if checkLocalRole "Primary|standby|Pending"; then
        log_info "[doStart] db had been start already."
        return "${r_success}"
    fi
    # 启动数据库
    restartToPending; retVal=$?
    return ${retVal}
}
######################################################################
#   FUNCTION   : doStop
#   DESCRIPTION: 数据库停止函数
######################################################################
function doStop()
{
    # 停止前，先检查当前数据库是否已经停止
    if ! isDBRunning; then
        log_info "[doStop]db is not running now."
        return "${r_success}"
    fi
    # 停止数据库
    gs_ctl stop -m fast;retVal=$?
    return "${retVal}"
}
######################################################################
#   FUNCTION   : force_do_stop
#   DESCRIPTION: 数据库强制停止函数
######################################################################
function force_do_stop()
{
    # 停止前，先检查当前数据库是否已经停止
    if ! isDBRunning; then
        log_info "[force_do_stop]db is not running now."
        return "${r_success}"
    fi
    # 强制停止数据库
    gs_ctl stop -m immediate;retVal=$?
    return "${retVal}"
}
######################################################################
#   FUNCTION   : doRepair
#   DESCRIPTION: 数据修复函数
######################################################################
function doRepair()
{
    # 修复只做rebuild操作。数据库角色不正确，通过激活和去激活恢复
    # 修复操作只有数据库角色为备或级联备端，同时数据库状态为 NeedRepair
    # 而且数据库连接状态不为连接中，或是连接断开
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_warn "[doRepair] db no running now."
        # 数据库没有运行，有可能是启动不了，检查是否需要重建
        isNeedRebuild; retVal=$?
        if [[ ${retVal} -eq ${db_abnormal} ]] || [[ ${runState} == "force_repair" ]]; then
            # 调用命令进行重建
            log_warn "[doRepair] start to repair by [gs_ctl build]. "
            gs_ctl build -D "${PGDATA}" -b full -M standby; retVal=$?
            # 重建完成，需要清空一下记录
            recordStartResult ${retVal}
            log_warn "[doRepair] repair finish[${retVal}]."
            return ${retVal}
        fi
        return "${db_stopped}"
    fi
    # 获取数据库状态
    dbinfo=$(gs_ctl query -D "${PGDATA}")
    # 从数据库的查询结果中分离出各项信息: local_role, db_state, detail_information, peer_role. 前面四个变量会在getDBState赋值。
    getDBState "${dbinfo}"; retVal=$?
    # 不为备状态，或是级联备，不处理
    if [[ "${local_role}" != "${gs_r_standby}" ]] && [[ "${local_role}" != "${gs_r_cstandby}" ]]; then
        log_warn "[doRepair] local_role[${local_role}] error. [${dbinfo}]"
        return "${r_success}"
    fi
    # 不是待修复状态，不处理
    if [[ "${db_state}" != "${gs_s_needrepair}" ]]; then
        log_warn "[doRepair] current db_state[${db_state}] no need to repair. [${dbinfo}]"
        return "${r_success}"
    fi
    # 详细信息不需要重建，则不重建
    if ! echo "${detail_information}" | grep -E "${gs_d_walremoved}|${gs_d_sysIDnotmatched}"; then
        log_warn "[doRepair] current detail_information[${detail_information}] error"
        log_warn "it can not be repair. [${dbinfo}]"
        return "${r_success}"
    fi
    # 调用命令进行重建
    log_warn "[doRepair] start exc:[gs_ctl build] to repair. [${dbinfo}]"
    gs_ctl build -D "${PGDATA}" -b full -M standby; retVal=$?
    log_warn "[doRepair] Ended to repair, GaussDB return: [${retVal}]."
    # 重建完成，需要清空一下记录
    recordStartResult ${retVal}
    log_warn "[doRepair] repair finish[${retVal}]."
    return ${retVal}
}
######################################################################
#   FUNCTION   : doDeactivate
#   DESCRIPTION: 数据库去激活函数
######################################################################
function doDeactivate()
{
    # 本端没有启动，尝试启动为备
    if ! isDBRunning; then
        log_warn "[doDeactivate] db no running now."
        # 启动数据库
        restartToStandby; retVal=$?
        return ${retVal}
    fi
    # 获取数据库状态
    dbinfo=$(gs_ctl query -D "${PGDATA}")
    # 从数据库的查询结果中分离出各项信息: local_role, db_state, detail_information, peer_role. 前面四个变量会在getDBState赋值。
    getDBState "${dbinfo}"; retVal=$?
    case "${local_role}" in
        # 本端pending时，直接降备
        "${gs_r_pending}")
            log_warn "[doDeactivate] db current role[${local_role}]"
            log_info "current ha role[${runState}], need notify to standby. [${dbinfo}]"
            notifyToStandby; retVal=$?
            return ${retVal}
            ;;
        # 已经是备用角色，不需要处理
        "${gs_r_standby}")
            log_info "[doDeactivate] db current role[${local_role}] no need to change. [${dbinfo}]"
            return "${r_success}"
            ;;
        # 主角色下，需要检查
        "${gs_r_primary}")
            log_info "[doDeactivate] db current role[${local_role}], need further check. [${dbinfo}]"
            ;;
        # 以下角色不是双机会出现的角色，直接重启即可
        "${gs_r_normal}"|"${gs_r_cstandby}"|"${gs_r_unknown}"|*)
            log_error "[doDeactivate] db current role[${local_role}] error! [${dbinfo}]"
            restartToStandby; retVal=$?
            return ${retVal}
            ;;
    esac
    # 本端角色不正确，需要判断对端，以及重建的情况
    case "${db_state}" in
        # 对端在位，且状态正常，不能直接降备，需要等待对端进行switchover或是双主时本端降备
        "${gs_s_normal}")
            log_warn "[doDeactivate] db state[${db_state}] stable now, start change to standby. [${dbinfo}]"
            ;;
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "${gs_s_rebuilding}"|"${gs_s_starting}"|"${gs_s_demoting}"|"${gs_s_promoting}"|"${gs_s_waiting}")
            log_warn "[doDeactivate] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${r_success}"
            ;;
        # 数据库需要重建，待修复时处理
        "${gs_s_needrepair}")
            # 重建状态不一定都是需要修复的，这个需要高斯DB后续修改
            if echo "${detail_information}" | grep -E "${gs_d_walremoved}|${gs_d_sysIDnotmatched}"; then
                log_warn "[doDeactivate] db state[${db_state}] need repair now"
                log_info "can not start to standby. [${dbinfo}]"
                return "${r_success}"
            fi
            # 不需要修复的情况，直接跳转到需要的状态。
            log_warn "[doDeactivate] db state[${db_state}], detail_information[${detail_information}]"
            log_warn "can not repair, start change to standby. [${dbinfo}]"
            ;;
        # 其他状态，直接重启跳转到需要的状态
        "${gs_s_unknown}"|*)
            log_info "[doDeactivate] db state[${db_state}] stable now, can goto want role. [${dbinfo}]"
            restartToStandby; retVal=$?
            return ${retVal}
            ;;
    esac
    # 主降备，在操作前，需要看对端的状态
    if [[ "" == "${peer_role}" ]]; then
        PriToSta; retVal=$?
        return ${retVal}
    fi            
    log_warn "[doDeactivate] db state[${db_state}] stable now, and peer role[${peer_role}]"
    log_warn "need peer to primary first. [${dbinfo}]"
    return "${r_success}"
}
######################################################################
#   FUNCTION   : doActivate
#   DESCRIPTION: 数据库激活函数
######################################################################
function doActivate()
{
    # 本端没有启动，尝试启动为主
    if ! isDBRunning; then
        log_warn "[doActivate] db no running now."
        # 启动数据库
        restartToPrimary; retVal=$?
        return ${retVal}
    fi
    # 获取数据库状态
    dbinfo=$(gs_ctl query -D "${PGDATA}")
    # 从数据库的查询结果中分离出各项信息: local_role, db_state, detail_information, peer_role. 前面四个变量会在getDBState赋值。
    getDBState "${dbinfo}"; retVal=$?
    case "${local_role}" in
    # 本端pending时，直接升主
        "${gs_r_pending}")
            log_warn "[doActivate] db current role[${local_role}]"
            log_warn "current ha role[${runState}], need restart. [${dbinfo}]"
            # pending 数据库BUG 升主之前需要翻转数据库配置
            notifyToPrimary; retVal=$?
            return ${retVal}
            ;;
        # 已经是主用状态，不需要处理
        "${gs_r_primary}")
            log_info "[doActivate] db current role[${local_role}] no need to change. [${dbinfo}]"
            return "${r_success}"
            ;;
        "${gs_r_standby}")
            log_info "[doActivate] db current role[${local_role}], need further check. [${dbinfo}]"
            ;;
        # 以下角色不是双机会出现的角色，直接重启即可
        "${gs_r_normal}"|"${gs_r_cstandby}"|"${gs_r_unknown}"|*)
            log_error "[doActivate] db current role[${local_role}] error! [${dbinfo}]"
            restartToPrimary; retVal=$?
            return ${retVal}
            ;;
    esac
    # 本端角色不正确，需要判断对端，以及重建的情况
    case "${db_state}" in
        # 对端在位，且状态正常，先尝试switchover，再尝试failover
        "${gs_s_normal}")
            log_info "[doActivate] db state[${db_state}] stable now, start change to primary. [${dbinfo}]"
            StaToPri; retVal=$?
            return ${retVal}
            ;;
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "${gs_s_rebuilding}"|"${gs_s_starting}"|"${gs_s_demoting}"|"${gs_s_promoting}"|"${gs_s_waiting}")
            log_warn "[doActivate] db state[${db_state}] insatiable now, need wait. [${dbinfo}]"
            return "${r_success}"
            ;;
        # 数据库需要重建，待修复时处理
        "${gs_s_needrepair}")
            # 重建状态不一定都是需要修复的，这个需要高斯DB后续修改
            if echo "${detail_information}" | grep -E "${gs_d_walremoved}|${gs_d_sysIDnotmatched}"; then
                log_warn "[doActivate] db state[${db_state}] need repair now, can not start to primary. [${dbinfo}]"
                return "${r_success}"
            fi
            # 不需要修复的情况，直接跳转到需要的状态。
            log_warn "[doActivate] db state[${db_state}], detail_information[${detail_information}]"
            log_warn " can not repair, start change to primary. [${dbinfo}]"
            restartToPrimary; retVal=$?
            return ${retVal}
            ;;
        # 其他状态，直接重启跳转到需要的状态
        "${gs_s_unknown}"|*)
            log_info "[doActivate] db state[${db_state}] stable now, can goto want role. [${dbinfo}]"
            restartToPrimary; retVal=$?
            return ${retVal}
            ;;
    esac
    return "${r_success}"
    
}

function backup_db()
{
    rm -rf ${BACKUP_PATH} && mkdir -p ${BACKUP_PATH}
    cd "$(dirname "${BACKUP_PATH}")" || exit 1
    GAUSSUSER=$(get_info gaussdb_admin_user)
    GAUSSPASSWORD=$(decrypt_pwd "$(get_info gaussdb_admin_pwd)")
    DB_FLOATIP=$(get_info gaussdb_float_ip)
    while read -r database_name;do
        [[ -z "${database_name}" ]] && continue
        [[ "${database_name}" =~ ^template[0-9]+$ ]] && continue
        gs_dump ${database_name} -U "${GAUSSUSER}" -W "${GAUSSPASSWORD}" -h "${DB_FLOATIP}" -Fc --clean --create --quote-all-identifiers -f ${BACKUP_PATH}/${database_name}.bak_file
        CHECK_RESULT $? "gs_dump ${database_name} failed."
    done <<< $(gsql postgres -U "${GAUSSUSER}" -W "${GAUSSPASSWORD}" -h "${DB_FLOATIP}" -lxt | grep Name | awk -F'|' '{print $NF}')
    tar -czf ${BACKUP_PATH}.tar.gz "$(basename "${BACKUP_PATH}")">/dev/null 2>&1
    CHECK_RESULT $? "tar czf ${BACKUP_PATH}.tar.gz failed."
    rm -rf ${BACKUP_PATH}
    return 0
}


function restore_db()
{
    if [[ ! -d ${BACKUP_PATH} ]];then
       CHECK_RESULT 1  "${BACKUP_PATH} not exist."
    fi
    GAUSSUSER=$(get_info gaussdb_admin_user)
    GAUSSPASSWORD=$(decrypt_pwd "$(get_info gaussdb_admin_pwd)")
    DB_FLOATIP=$(get_info gaussdb_float_ip)
    while read -r bak_file;do
        [[ -z "${bak_file}" ]] && continue
        database_name=$(basename ${bak_file} .bak_file)
        gs_restore -d ${database_name} -U "${GAUSSUSER}" -W "${GAUSSPASSWORD}" -h "${DB_FLOATIP}" --clean ${bak_file}
        CHECK_RESULT $? "gs_restore ${database_name} failed."
    done <<< $(find ${BACKUP_PATH} -type f -name "*.bak_file")
    rm -rf ${BACKUP_PATH}
    return 0
}

function main()
{
    case "${optCommand}" in
        status)
            # 查询资源状态，返回码为上面列出的返回码
            getStatus; retVal=$?
            return ${retVal}
            ;;
        start)
            # 启动资源，返回码 0表示成功 1表示失败
            doStart; retVal=$?
            return ${retVal}
            ;;
        stop)
            # 停止资源，返回码 0表示成功 1表示失败
            doStop; retVal=$?
            return ${retVal}
            ;;
        active)
            # 激活资源，返回码 0表示成功 1表示失败
            doActivate; retVal=$?
            return ${retVal}
            ;;
        deactive)
            # 去激活资源，返回码 0表示成功 1表示失败
            doDeactivate; retVal=$?
            return ${retVal}
            ;;
        force-stop)
            # 强制停止资源，返回码 0表示成功 1表示失败
            force_do_stop; retVal=$?
            return ${retVal}
            ;;
        repair)
            # 如果是停止失败，会进行force-stop，不需要修复
            # 在start、active、deactive失败时，会调用此接口
            # 修复资源，返回码 0表示成功 1表示失败
            doRepair; retVal=$?
            return ${retVal}
            ;;
        backup)
            # 执行数据库备份
            backup_db; retVal=$?
            return ${retVal}
            ;;
        restore)
            # 执行数据库恢复
            restore_db; retVal=$?
            return ${retVal}
            ;;
        query)
            gs_ctl query -D "${PGDATA}"; retVal=$?
            return ${retVal}
            ;;
        version)
            gs_ctl -V; retVal=$?
            return ${retVal}
            ;;
        *)
            #echo "Usage: $0 { stop|status|notify|force-stop|..}"
            # 返回动作不存在
            log_info "Unknown cmd[${optCommand}]."
            return "${db_notsupported}"
            ;;
     esac
}

# 获取脚本入参
declare optCommand="$1"; shift   # 当前运行的命令
declare runState="$1";   shift   # 当前ha的运行状态
main ; scriptRetVal=$?
exit ${scriptRetVal}
