#!/bin/bash
if [[ ${UID} -eq 0 ]];then
    sudo -u openstack bash "$(readlink -f "$0")" "$@";exit $?
elif [[ ${UID} -ne 51001 ]];then
    echo "Please switch to user(openstack) for run." && exit 1
fi
#check rabbitmq service cluster status
source /opt/huawei/dj/inst/utils.sh
export LOG_TAG="rabbitmqControl"
RABBITMQCTL_PATH="/opt/huawei/dj/bin/rabbitmq/sbin/rabbitmqctl"
RABBITMQ_CFG_PATH="/opt/huawei/dj/bin/rabbitmq/etc/rabbitmq/rabbitmq.config"
RABBITMQ_WATCHER="/opt/huawei/dj/tools/rabbitmq/rabbitmq_watcher.py"

function init_rabbit_user()
{
    # ensure vhosts is OK
    ${RABBITMQCTL_PATH} add_vhost "/"
    CHECK_RESULT "$?" "ERROR | add rabbitmq vhosts failed."
    log_info "INFO | The rabbitmq vhosts \"/\" is ok."

    # ensure user is OK
    ${RABBITMQCTL_PATH} add_user rabbit ''
    ret=$? && [[ ${ret} -eq 70 ]] && ret=0
    CHECK_RESULT "${ret}" "ERROR | add rabbitmq user failed."
    log_info "INFO | The user rabbit is OK."

    # ensure user administrator is OK
    ${RABBITMQCTL_PATH} set_user_tags rabbit administrator
    CHECK_RESULT "$?" "ERROR | add rabbit's administrator failed."
    log_info "INFO | The user rabbit's administrator is OK."

    # ensure rabbit permissions is OK
    ${RABBITMQCTL_PATH} set_permissions rabbit ".*" ".*" ".*"
    CHECK_RESULT "$?" "ERROR | add rabbitmq rabbit permissions failed."
    log_info "INFO | The rabbitmq rabbit permissions \".*\" is OK."

    # ensure user password is OK
    cmd_retry "csbs_python ${RABBITMQ_WATCHER} rabbitmq.pwd" 60 5
    CHECK_RESULT "$?" "ERROR | reset rabbitmq pwd rabbit failed."
    log_info "INFO | reset rabbitmq pwd rabbit success."
    return 0
}

function init_server_params()
{
    mem_total=$(grep MemTotal /proc/meminfo |awk '{print $2}')
    mem_total_MB=$((mem_total/1024))
    frac=$(awk 'BEGIN{x=8192/'"${mem_total_MB}"'*'1.3'; if (x < 1) {printf "%.4f", x} else {printf "0.4"}}')

    ${RABBITMQCTL_PATH} set_vm_memory_high_watermark "${frac}"
    CHECK_RESULT "$?" "ERROR | rabbitmqctl set_vm_memory_high_watermark failed."
    log_info "INFO | set_vm_memory_high_watermark success."

    ${RABBITMQCTL_PATH} set_policy ha-allqueue "^"  '{"ha-mode":"all","ha-sync-mode":"automatic"}' --priority 1 --apply-to exchanges
    CHECK_RESULT "$?" "ERROR | rabbitmqctl set_policy ha-allqueue failed."
    log_info "INFO | set_policy ha-allqueue success."

    ${RABBITMQCTL_PATH} set_policy ha_length_ttl_meter "^metering.sample$" '{"ha-mode":"all","ha-sync-mode":"automatic", "max-length":720000, "message-ttl": 86400000}' --priority 1 --apply-to queues
    CHECK_RESULT "$?" "ERROR | rabbitmqctl set_policy ha_length_ttl_meter failed."
    log_info "INFO | set_policy ha_length_ttl_meter success."

    ${RABBITMQCTL_PATH} set_policy ha_length_ttl '^(?!metering\.sample).+' '{"ha-mode":"all","ha-sync-mode":"automatic", "max-length":59600, "message-ttl": 86400000}' --priority 1 --apply-to queues
    CHECK_RESULT "$?" "ERROR | rabbitmqctl set_policy ha_length_ttl failed."
    log_info "INFO | set_policy ha_length_ttl success."
    return 0
}

function join_cluster()
{
    # ensure rabbit cluster is running
    ${RABBITMQCTL_PATH} -n "$1" status
    CHECK_RESULT "$?" "ERROR | $1 is not running."
    log_info "INFO | $1 is running."

    # ensure rabbit app is stop
    ${RABBITMQCTL_PATH} stop_app
    CHECK_RESULT "$?" "ERROR | stop rabbit app failed."
    log_info "INFO | stop rabbit app success."

    # ensure rabbit cluster_status is ok
    ${RABBITMQCTL_PATH} reset
    CHECK_RESULT "$?" "ERROR | reset rabbit app failed."
    log_info "INFO | reset rabbit app success."

    # ensure rabbit join_cluster is ok
    ${RABBITMQCTL_PATH} join_cluster "$1"
    CHECK_RESULT "$?" "ERROR | join_cluster $1 failed."
    log_info "INFO | join_cluster $1 app success."

    # ensure rabbit app is ok
    ${RABBITMQCTL_PATH} start_app
    CHECK_RESULT "$?" "ERROR | start rabbit app failed."
    log_info "INFO | start rabbit app success."
    log_info "join_cluster rabbitmq success."
    return 0
}

function waitServerStarted()
{
    for i in $(seq 30); do
        if ${RABBITMQCTL_PATH} status >/dev/null 2>&1; then
            return 0
        fi
        log_warn "service not running. waitServerStarted ${i}"
        sleep 10
    done
    CHECK_RESULT "1" "ERROR | rabbitmq is not running."
}

function restart_rabbitmq()
{
    log_info "Try to start the server, even if it is already started"
    csbs_python ${RABBITMQ_WATCHER} generate_config
    CHECK_RESULT $? "ERROR | Generate rabbitmq config failed."

    bash /opt/huawei/dj/bin/rabbitmq/sbin/rabbitmq-plugins disable rabbitmq_management >> /dev/null 2>&1
    bash /opt/huawei/dj/bin/rabbitmq/sbin/rabbitmq-server -detached >> /dev/null 2>&1

    log_info "Wait rabbitmq running."
    waitServerStarted

    log_info "Init rabbitmq user info."
    sleep 10
    init_rabbit_user

    log_info "Init rabbitmq server params."
    sleep 10
    init_server_params
    log_info "Restart rabbitmq success."
    return 0
}

# 检查当前节点是否正在运行，未运行则进行重启
queryCmd=$(pgrep -U 51001 -f /opt/huawei/dj/bin/erlang.*rabbit)
if [[ -z "${queryCmd}" ]]; then
    log_error "Current node rabbit is stop, need to start it."
    restart_rabbitmq
fi

# rabbitmq集群配置为默认值，无需添加集群
if grep "rabbit@rabbitmqNode0" "${RABBITMQ_CFG_PATH}";then
    log_info "One node no need to join cluster."
    exit 0
fi

# rabbitmq配置缺失当前节点集群信息，无法添加集群
cur_node_name="rabbit@$(get_info hostname)"
cluster_nodes=$(grep "${cur_node_name}" "${RABBITMQ_CFG_PATH}" | awk  -F '['  '{print $2}'| awk  -F ']'  '{print $1}')
if [[ "${cluster_nodes}" == "" ]];then
    log_error "Cannot get CLUSTER_INFO from conf."
    exit 1
fi

# 查询集群信息
cluster_info="/opt/huawei/dj/bin/rabbitmq/cluster_info.txt"
${RABBITMQCTL_PATH} cluster_status > ${cluster_info}

# 当前节点为集群首节点，无需添加集群
IFS="," read -r -a clusters <<< "${cluster_nodes}"
if [[ "${cur_node_name}" == "${clusters[0]}" ]];then
    log_info "This current node is cluster name, not need join cluster."
    exit 0
fi

# 集群节点中仅包含当前节点则当前节点尚未加入集群，反之，当前节点已经存在集群，无需添加集群
running_nodes_line_number=$(grep -n "Running Nodes" ${cluster_info} | cut -d : -f 1)
version_line_number=$(grep -n "Versions" ${cluster_info} | cut -d : -f 1)
nodes=$(awk -v s="${running_nodes_line_number}" -v e="${version_line_number}" 'NR>s&&NR<e' ${cluster_info} | grep -v "^$")
if [[ "${nodes}" != "${cur_node_name}" ]];then
    log_info "The current node is in the cluster."
    exit 0
fi

# 加入首节点所在的集群
if ! join_cluster "${clusters[0]}";then
   log_info "Join cluster failed."
   exit 1
fi
log_info "Join cluster success."
exit 0