# coding: utf-8
import random
from datetime import datetime
from functools import partial

import six
from eventlet import sleep
from oslo_config import cfg
from oslo_log import log
from oslo_serialization import jsonutils

import kangaroo.data_models.backup
from kangaroo.client import iam
from kangaroo.client.openstack_client_factory import ClientFactory
from kangaroo.common import cbs_constants
from kangaroo.common.utils import cbs_utils
from kangaroo.common.utils import volume_utils
from kangaroo.common.utils.cbs_utils import get_expired_at
from kangaroo.db import api
from kangaroo.plugin import base_operation
from kangaroo.plugin.utils import delete_utils, backup_utils
from kangaroo.plugin.utils import utils as plugin_utils
from kangaroo.protection.flows import utils as flow_utils
from kangaroo.protection.queue_kits import SNAPSHOT_QUEUE_SWITCH

LOG = log.getLogger(__name__)
CONF = cfg.CONF

POLL_INTERVAL = 90
# csbs未上时，vbs产生的快照名前缀
OLD_SNAP_NAME_PREFIX = 'autobk_snapshot_'
# csbs服务上线后，vbs产生的快照名前缀
SNAP_NAME_PREFIX = 'autobk_snapshot_vbs'
SNAP_TAG_SERVER = 'csbs'

PROTECTING = cbs_constants.CHECKPOINT_ITEM_STATUS_PROTECTING


class BackupOperation(base_operation.BaseOperation):
    RESOURCE_TYPE = cbs_constants.RESOURCE_TYPE.VOLUME_RESOURCE_TYPE
    OPERATION_TYPE = 'protect'

    def __init__(self):
        super(BackupOperation, self).__init__()
        self.image_type = cbs_constants.CHECKPOINT_ITEM_IMAGE_TYPE_BACKUP
        self.operation_log = None

    @plugin_utils.log_with_unexpected_error
    @plugin_utils.eagerly_finish_task
    def on_main(self, checkpoint, resource, context, parameters, **kwargs):
        self.operation_log = kwargs.get('operation_log')
        volume_id = resource.id
        checkpoint_item = checkpoint.get_item(volume_id)
        auto_trigger = parameters.get(cbs_constants.AUTO_TRIGGER, True)
        LOG.info(
            f"Backup start for volume {resource.id} with checkpoint_item: {checkpoint_item}, parameters: {parameters}.")
        backup, snap = None, None
        new_item_id = checkpoint_item.id
        extend_info = kangaroo.data_models.backup.ExtendInfo(
            auto_trigger=auto_trigger, resource_name=resource.name, image_type=cbs_constants.OPERATE_TYPE.BACKUP,
            copy_status=cbs_constants.CHECKPOINT_ITEM_COPY_STATUS_NA)
        volume_backup = kangaroo.data_models.backup.VolumeBackup(image_type=cbs_constants.OPERATE_TYPE.BACKUP,
                                                                 source_volume_id=volume_id,
                                                                 status=cbs_constants.CHECKPOINT_ITEM_STATUS_PROTECTING)
        extend_info.volume_backups = [volume_backup]
        self.operation_log.extra_info['backup'] = {
            'checkpoint_item_id': new_item_id,
            'checkpoint_item_name': checkpoint_item.name,
            'auto_trigger': auto_trigger
        }
        self.operation_log.save()

        try:
            cinder_client = ClientFactory.create_client("cinder", context)
            volume = cinder_client.volumes.get(volume_id)
            self._init_extend_info(context, parameters, resource, volume, extend_info)

            create_param, snap = self._build_create_params(parameters, extend_info, checkpoint, checkpoint_item,
                                                           cinder_client, volume_id, volume.availability_zone)
            if context.pre_run:
                LOG.info(f"Backup for volume {volume_id} pause due to backup queue is full.")
                context.need_pause = True
                return

            backup, new_item_id = self._create_backup(context, cinder_client, create_param, checkpoint,
                                                      checkpoint_item, extend_info)
            self.wait_backup(context, cinder_client, checkpoint, volume_id, resource, parameters, self.operation_log)
        except Exception as err:
            LOG.exception(f"Backup volume {resource.id} failed.")
            if parameters.get("reset_to_full_backup", False):
                # 增备转全备，失败后更新resource数据库
                plugin_utils.update_full_backuped_fix_tag(context, resource, "resource_fix_failed")
            extend_info.fail_reason = six.text_type(err)
            extend_info.volume_backups[0].status = cbs_constants.CHECKPOINT_ITEM_STATUS_ERROR
            extend_info.fail_op = 'backup'
            checkpoint.update_item(new_item_id, {
                'status': cbs_constants.CHECKPOINT_ITEM_STATUS_ERROR,
                'extend_info': str(extend_info)
            })
            self._cleanup_resource(context, getattr(backup, "id", None), getattr(snap, "id", None),
                                   extend_info.resource_az)
            plugin_utils.update_protection_summary(context, resource.id, 'backup')
            flow_utils.update_op_log_fail(self.operation_log, error_code='BackupService.1000',
                                          error_msg=six.text_type(err))
            backup_utils.send_retry_failed_alarm(context, self.operation_log)

    def _create_backup(self, context, cinder_client, create_param, checkpoint, checkpoint_item, extend_info):
        LOG.info(f"Create backups with {create_param}.")
        backup = cinder_client.backups.create(**create_param)
        extend_info.volume_backups[0].id = backup.id
        extend_info.volume_backups[0].name = backup.name
        extend_info.volume_backups[0].status = backup.status
        new_item = checkpoint.update_item(checkpoint_item.id, {'id': backup.id, 'extend_info': str(extend_info)})
        self.create_tag_in_cinder(context, backup.id)
        self.operation_log.extra_info['backup']['checkpoint_item_id'] = backup.id
        self.operation_log.checkpoint_item_id = backup.id
        self.operation_log.save()
        LOG.info(f"Change item id from {checkpoint_item.id} to {new_item.id}.")
        new_item_id = new_item.id
        return backup, new_item_id

    @staticmethod
    def _init_extend_info(context, parameters, resource, volume, extend_info):
        if plugin_utils.check_to_full_backup(context, parameters, resource):
            parameters["incremental"] = False
            parameters["reset_to_full_backup"] = True
        attach_server_cpu_vendor, attach_server_arch = volume_utils.get_vender_arch(context, volume)
        extend_info.volume_backups[0].source_volume_size = volume.size
        extend_info.volume_backups[0].bootable = volume.bootable
        extend_info.volume_backups[0].source_volume_attach_device = _get_volume_attach_device(volume)
        extend_info.volume_backups[0].source_volume_name = volume.name
        # 通过CMKID判断是否是加密卷，更新加密字段
        encryption_info = getattr(volume, 'encryption_info', {})
        extend_info.volume_backups[0].encrypted = True if encryption_info.get("cmk_id") else False
        extend_info.resource_az = volume.availability_zone
        extend_info.incremental = parameters.get("incremental", True)
        extend_info.architecture = attach_server_arch
        extend_info.cpu_vendor = attach_server_cpu_vendor

    def _build_create_params(self, parameters, extend_info, checkpoint,
                             checkpoint_item, cinder_client, volume_id,
                             resource_az):
        create_param = dict()
        snap = None
        for _ in range(5):
            try:
                snap = self._create_snapshot(
                    cinder_client, extend_info, checkpoint_item,
                    checkpoint, volume_id, resource_az)
                create_param["snapshot_id"] = snap.id
                break
            except Exception:
                LOG.exception("Failed to create snapshot for volume: "
                              "{volume_id}, try again after 120s."
                              "".format(volume_id=volume_id))
                sleep(120)

        create_param["container"] = parameters.get(
            "container") or create_param.get("snapshot_id")
        create_param["force"] = parameters.get("force") if parameters.get(
            "force") else True
        create_param["volume_id"] = volume_id
        create_param["name"] = checkpoint_item.name
        if extend_info.incremental:
            description = {
                "INC": 1,
                "DESC": checkpoint_item.description
            }
        else:
            description = {
                "INC": 0,
                "DESC": checkpoint_item.description
            }
        create_param["description"] = jsonutils.dumps(description,
                                                      ensure_ascii=False)
        return create_param, snap

    @staticmethod
    def _create_snapshot(cinder_client, extend_info, checkpoint_item, checkpoint, volume_id, resource_az):
        def __create_snapshot():
            snap_name = SNAP_NAME_PREFIX + cbs_utils.build_time_suffix()
            snap = cinder_client.volume_snapshots.create(volume_id=volume_id, force=True, name=snap_name)
            extend_info.volume_backups[0].snapshot_id = snap.id
            checkpoint.update_item(
                checkpoint_item.id, {
                    'status': PROTECTING,
                    'extend_info': str(extend_info),
                    'protected_at': snap.created_at
                })
            is_snap_success = plugin_utils.progress_poll(
                partial(plugin_utils.get_resource, cinder_client.volume_snapshots,
                        snap.id, 'snap'),
                None, interval=POLL_INTERVAL,
                success_statuses={'available'}, failure_statuses={'error'},
                ignore_statuses={'creating'})
            if is_snap_success:
                SNAPSHOT_QUEUE_SWITCH.notify_result(f"create_snapshot_{resource_az}", True)
                return snap
            else:
                SNAPSHOT_QUEUE_SWITCH.notify_result(f"create_snapshot_{resource_az}", False)
                raise Exception("Create snapshot failed.")
        lock = SNAPSHOT_QUEUE_SWITCH.get_queue_lock(f"create_snapshot_{resource_az}")
        min_get_lock_randint_timeout_s = 1800
        max_get_lock_randint_timeout_s = 5400
        timeout = random.randint(min_get_lock_randint_timeout_s, max_get_lock_randint_timeout_s)
        try:
            if not lock.acquire(timeout=timeout):
                LOG.warn(f"Failed to get create_snapshot_{resource_az} lock, the wait time is {timeout} seconds.")
            return __create_snapshot()
        finally:
            lock.release()

    def wait_backup(self, context, cinder_client, checkpoint, volume_id,
                    resource, parameters, operation_log):
        def update_progress(cinder_backup):
            progress = plugin_utils.extract_progress(cinder_backup)
            extend_info.progress = progress
            checkpoint.update_item(checkpoint_item.id, {'extend_info': str(extend_info)})
            operation_log.extra_info['common']['progress'] = progress
            operation_log.save()

        checkpoint_item = checkpoint.get_item(volume_id)
        extend_info = kangaroo.data_models.backup.ExtendInfo.from_str(checkpoint_item.extend_info)
        backup_id = extend_info.volume_backups[0].id
        if extend_info.auto_trigger:
            interval = POLL_INTERVAL * 6
        else:
            interval = POLL_INTERVAL * 2

        is_backup_success = plugin_utils.progress_poll(
            partial(plugin_utils.get_resource, cinder_client.backups, backup_id, 'backup'),
            update_progress,
            interval=interval,
            success_statuses={'available', 'restoring'},
            failure_statuses={'error'},
            ignore_statuses={'creating'},
        )
        backup = cinder_client.backups.get(backup_id)
        if is_backup_success:
            checkpoint_item = checkpoint.get_item(volume_id)
            extend_info = kangaroo.data_models.backup.ExtendInfo.from_str(checkpoint_item.extend_info)
            plugin_utils.rebuild_extend_info_from_backup(extend_info, backup)
            if parameters.get("reset_to_full_backup", False):
                # 增备转全备，备份完成后数据库中的数据改为增备
                plugin_utils.update_full_backuped_fix_tag(context, resource, CONF.features.full_backup_tag)
                extend_info.incremental = True
                extend_info.is_incremental_backup = True
            elif CONF.features.full_backup_switch and not parameters.get("incremental", True):
                plugin_utils.update_full_backuped_fix_tag(context, resource, CONF.features.full_backup_tag)
            finished_at = datetime.utcnow().isoformat()
            extend_info.finished_at = finished_at
            expired_at = get_expired_at(jsonutils.loads(checkpoint_item.extend_info).get('retention_duration', -1))
            update_info = {
                'status': cbs_constants.CHECKPOINT_ITEM_STATUS_AVAILABLE,
                "expired_at": expired_at,
                'extend_info': str(extend_info)
            }
            checkpoint_item = checkpoint.update_item(checkpoint_item.id, update_info)
            LOG.info(f"The volume of {checkpoint_item.resource_name} backuped successfully, "
                     f"volume id: {checkpoint_item.resource_id}, checkpoint_item_id: {checkpoint_item.id}, "
                     f"protected_at: {checkpoint_item.protected_at}, finished_at: {finished_at}, "
                     f"expired_at: {expired_at}.")
            plugin_utils.update_protection_summary(context, resource.id, 'backup')
            self._clean_snapshot(context, cinder_client, volume_id, parameters, extend_info.resource_az)
            # 更新副本中保留最后一个副本的标志状态
            plugin_utils.update_reserve_the_latest(context, checkpoint_item)
            flow_utils.update_op_log_success(operation_log)
        else:
            # 抛异常让外层处理
            raise Exception(backup.fail_reason)
        return backup

    @staticmethod
    def create_tag_in_cinder(ctx, backup_id):
        """Sync vbs tag to cinder."""
        tags = api.backup_tags_get(ctx, backup_id)
        if not tags:
            return

        tags_dict = [{'key': tag['key'], 'value': tag['value']}
                     for tag in tags]
        new_cc = ClientFactory.create_client("cinder", ctx, retries=3)
        try:
            new_tag_body = {'action': 'create', 'tags': tags_dict}
            new_url = '/os-vendor-backups/{backup_id}/tags/action'. \
                format(backup_id=backup_id)
            new_cc.client.post(new_url, body=new_tag_body)
            LOG.info("Sync tags %s to backup %s with new API version"
                     "successfully." % (new_tag_body, backup_id))
            return
        except Exception:
            LOG.exception("Sync tags with new API version failed.")

        try:
            old_tag_body = {'tags': tags_dict}
            old_url = '/os-vendor-tags/backups/{backup_id}'. \
                format(backup_id=backup_id)
            new_cc.client.post(old_url, body=old_tag_body)
            LOG.info("Sync tags %s to backup %s with old API version"
                     "successfully." % (old_tag_body, backup_id))
        except Exception:
            LOG.exception("Sync tags with old API version failed.")

    def _clean_snapshot(self, context, cinder_client, volume_id, parameters,
                        resource_az):
        # 查询所有快照，默认按时间倒序
        try:
            snapshots = cinder_client.volume_snapshots.list(
                detailed=False, search_opts={"volume_id": volume_id},
                sort="created_at:asc")
            cbs_snapshots = self._filter_vbs_snapshots(snapshots)
            # 保留最新的一个快照
            admin_ctx = iam.get_privilege_context(context, retries=CONF.client_retries + 1)
            admin_client = ClientFactory.create_client("cinder", admin_ctx)

            if CONF.features.clean_cur_time_snap and \
                    plugin_utils.is_delete_current_time_snap(
                        context,
                        parameters.get("plan_id"),
                        parameters.get("auto_trigger", True)
                    ) is True:
                LOG.info("The next time is a full backup, "
                         "so delete this snapshot of volume(%s)" % volume_id)
                delete_snapshots = cbs_snapshots
            else:
                delete_snapshots = cbs_snapshots[0:len(cbs_snapshots) - 1]

            for snap in delete_snapshots:
                try:
                    # 强制删除
                    delete_utils.delete_and_wait_volume_snapshots(
                        context, admin_client, [snap.id, ], resource_az)
                except Exception as err:
                    LOG.exception("Clean up old snapshot %s failed: %s." % (
                        snap.id, six.text_type(err)))
        except Exception as err:
            LOG.exception("Clean up snapshot failed, error: %s"
                          % six.text_type(err))

    @staticmethod
    def _filter_vbs_snapshots(snapshots):
        vbs_snapshots = []
        for snapshot in snapshots:
            # 通过旧的快照名获取的快照需要排除掉整机产生的
            prefix_match = snapshot.name.startswith(OLD_SNAP_NAME_PREFIX)
            name_match = snapshot.name.find(SNAP_TAG_SERVER) == -1
            is_old_name_match = prefix_match and name_match
            is_name_match = snapshot.name.startswith(SNAP_NAME_PREFIX)
            is_status_match = snapshot.status in [
                cbs_constants.SNAPSHOT_STATUS_ERROR,
                cbs_constants.SNAPSHOT_STATUS_AVAILABLE]
            if (is_old_name_match or is_name_match) and is_status_match:
                vbs_snapshots.append(snapshot)
        return vbs_snapshots

    @staticmethod
    def _cleanup_resource(context, backup_id, snap_id, resource_az):
        """备份失败清理资源"""
        try:
            admin_ctx = iam.get_privilege_context(context)
            admin_client = ClientFactory.create_client("cinder", admin_ctx)
            if snap_id:
                delete_utils.delete_and_wait_volume_snapshots(
                    context, admin_client, [snap_id, ], resource_az)
            if backup_id:
                # 备份不支持用force删除
                LOG.info("Clean up backup %s" % backup_id)
                admin_client.backups.delete(backup=backup_id)
        except Exception as err:
            LOG.exception("Clean up resource failed: %s" % six.text_type(err))

    @plugin_utils.log_with_unexpected_error
    def on_continue(self, checkpoint, resource, context, parameters, **kwargs):
        """备份任务续做"""
        context.update_store()
        volume_id = resource.id
        self.operation_log = kwargs.get('operation_log')
        if plugin_utils.is_operation_log_finish(self.operation_log):
            LOG.info("Volume %s's backup has finished with operation log "
                     "%s." % (resource.id, self.operation_log.id))
            return

        checkpoint_item = checkpoint.get_item(volume_id)
        LOG.info("On continue backup for volume %s, "
                 "checkpoint_item: %s, parameters: %s" % (
                     resource.id, checkpoint_item, parameters))

        extend_info = kangaroo.data_models.backup.ExtendInfo.from_str(
            checkpoint_item.extend_info)
        if extend_info.volume_backups and extend_info.volume_backups[0].id:
            # 任务已经下发，直接等待结果
            LOG.info("Backup task has send to cinder, so direct wait result")
            try:
                cinder_client = ClientFactory.create_client("cinder", context)
                if plugin_utils.check_to_full_backup(context, parameters, resource):
                    parameters["incremental"] = False
                    parameters["reset_to_full_backup"] = True
                self.wait_backup(context, cinder_client, checkpoint, volume_id,
                                 resource, parameters, self.operation_log)
            except Exception as err:
                LOG.exception("On continue backup failed")
                if parameters.get("reset_to_full_backup", False):
                    # 增备转全备，失败后更新resource数据库
                    plugin_utils.update_full_backuped_fix_tag(
                        context, resource, "resource_fix_failed")
                extend_info.volume_backups[0].status = \
                    cbs_constants.CHECKPOINT_ITEM_STATUS_ERROR
                extend_info.fail_reason = six.text_type(err)
                extend_info.fail_op = 'backupcontinue'
                checkpoint.update_item(checkpoint_item.id, {
                    'status': cbs_constants.CHECKPOINT_ITEM_STATUS_ERROR,
                    'extend_info': str(extend_info)
                })
                flow_utils.update_op_log_fail(
                    self.operation_log, error_code='BackupService.1000',
                    error_msg=six.text_type(err))
                backup_utils.send_retry_failed_alarm(context, self.operation_log)
                self._cleanup_resource(
                    context, extend_info.volume_backups[0].id,
                    extend_info.volume_backups[0].snapshot_id,
                    extend_info.resource_az)
                plugin_utils.update_protection_summary(context, resource.id, 'backup')
        else:
            LOG.info("Backup task has not send to cinder, so start resend")
            # 任务没有下发，重走整个流程
            self.on_main(checkpoint, resource, context, parameters, **kwargs)


def _get_volume_attach_device(volume):
    """获取磁盘挂载点

    跟evs确认，目前系统盘的挂载点一定是满足如下两点：
    1、volume详情中attachments的第一个元素
    2、元素中的device一定为"/dev/sda"或者为"/dev/vda"

    注: karbor只是把device写入item，由前台结合bootable综合判断是否是系统盘
    :param volume:
    :return:
    """
    attachments = getattr(volume, "attachments", [])
    if not attachments:
        return None
    return attachments[0].get("device")
