# -*- coding: utf-8 -*-
import random
import time
from collections import OrderedDict

import six
from cinderclient import exceptions as cinder_exc
from cinderclient.apiclient.base import Resource
from eventlet import sleep
from novaclient import exceptions as nova_exc
from oslo_config import cfg
from oslo_db.exception import DBDuplicateEntry
from oslo_log import log
from oslo_serialization import jsonutils

from kangaroo.client import nova
from kangaroo.client import proxy
from kangaroo.client.https import workflow_client
from kangaroo.common.errorcode.server import error_const
from kangaroo.common import kangaroo_alarms
from kangaroo.db import api as db_api
from kangaroo.plugin.utils import utils as p_utils
from kangaroo.protection.queue_kits import SNAPSHOT_QUEUE_SWITCH

CONF = cfg.CONF
LOG = log.getLogger(__name__)

RESTORE_LISTEN_TIME = 60 * 5
BACKUP_AUTO_LISTEN_TIME = 60 * 5
BACKUP_MANUAL_LISTEN_TIME = 90
SNAPSHOTS_LISTEN_TIME = 15
FAIL_WHEN_APP_BACKUP_FAIL = 1
SNAPSHOTS_CONSTANT_LISTEN_TIME = 20
DO_NORMAL_BACKUP_WHEN_APP_BACKUP_FAIL = 2
NOT_COMPRESSION_DEDUPLICATION = 0
DEDUPLICATION = 1
COMPRESSION = 2
COMPRESSION_DEDUPLICATION = 3
AZ_COMPLETE = 20
PROGRESS_PERCENTAGE = 0.7

AUTO_SNAPSHOT_PREFIX = "autobk_snapshot_csbs_"
MANUAL_SNAPSHOT_PREFIX = "manualbk_snapshot_csbs_"
AUTO_BACKUP_PREFIX = "autobk_"
MANUAL_BACKUP_PREFIX = "manualbk_"

NORMAL_BACKUP = 0
APP_BACKUP_FAIL = 1
APP_BACKUP_SUCC = 2
APP_BACKUP_FAIL_BY_OLD_AGENT_VERSION = 11
APP_BACKUP_SUCC_BY_OLD_AGENT_VERSION = 12


class BackupException(Exception):
    def __init__(self, code='', message='', body=None):
        super(BackupException, self).__init__()
        self.code = code
        self.message = message
        self.body = body or {}


def _list_volumes_by_spec_ids(volumes_id, cinder_client):
    times = 0
    volumes = []
    while True:
        ids = volumes_id[times * 60:(times + 1) * 60]
        if not ids:
            break
        url = "/os-vendor-volumes/detail?ids=%s" % ids
        _, body = cinder_client.client.get(url)
        volumes.extend(body.get("volumes", []))
        times += 1
    return volumes


def get_volumes_metadata(volumes_id, cinder_client):
    """Query volume metadata from cinder.

    :param volumes_id:
    :param cinder_client:
    :return:{
                volume_id1:{"volume":{"id":xxx, "status":xx}},
                volume_id2:{"volume":{"id":xxx, "status":xx}},
                ...
            }
    """
    volumes_metadata = dict()
    try:
        volumes = _list_volumes_by_spec_ids(volumes_id, cinder_client)
    except Exception as err:
        error_msg = getattr(err, 'message', 'Unknown error')
        error_code = getattr(err, 'code', 'unknown')
        msg = "Get volume(%s) metadata from cinder error, err: %s." \
              % (volumes_id, err)
        LOG.exception(msg)
        raise BackupException(code=error_code, message=error_msg)

    for volume in volumes:
        if volume["id"] in volumes_id:
            volumes_metadata[volume["id"]] = {"volume": volume}
    expect_count = len(volumes_id)
    actual_count = len(volumes_metadata)
    _check_result(expect_count, actual_count, desc='get volume meta failed')
    return volumes_metadata


def _get_ext_server(server_id, nova_ext_client):
    search_opts = {'server_id': server_id}
    servers = nova_ext_client.servers.list(
        search_opts=search_opts, detailed=False)
    if servers:
        return servers[0]
    else:
        raise nova_exc.NotFound(code=404)


def get_vm_metadata(context, resource, nova_client, nova_ext_client, neutron_client):
    """Query server metadata from nova and neutron.

    :param nova_ext_client:
    :param resource: resource obj
    :param nova_client:
    :param neutron_client:
    :return: {
                "vmMetaInfo":{"server":{"name":xxx}},
                "subnetsMetaInfo":{"subnet": {"host_routes": [],"name": "x",}},
                "interfaceMetaInfo":{"interfaceAttachments":{}},
                "floatipMetaInfo":[{"floatingips" : []}],
                "flavorMetaInfo":{"flavor":{"name":xxx}},
                "NetworkInfo":{port_id_1:{},port_id_2:{}},
                "HypervisorInfo":{"id":1402,"status": "enabled"...},
                "Architectrue":""
              }
    """
    vm_metadata = dict()
    try:
        server = nova.get_ext_server(context, resource.id, nova_ext_client)
        vm_metadata["vmMetaInfo"] = {"server": server.to_dict()}
        hypervisor_detail = _get_vm_hypervisor(server, nova_client)
        vm_metadata["HypervisorInfo"] = hypervisor_detail.to_dict()
        interfaces = nova_client.servers.interface_list(resource.id)
        vm_metadata["interfaceMetaInfo"] = {
            "interfaceAttachments": [interface.to_dict() for interface in interfaces]
        }
        flavor = nova_client.flavors.get(server.flavor.get("id"))
        vm_metadata["flavorMetaInfo"] = {"flavor": flavor.to_dict()}
        _, architecture = nova.get_arch_vendor(nova_ext_client, resource.id)
        vm_metadata["Architecture"] = architecture
    except Exception as err:
        error_msg = getattr(err, 'message', 'Unknown error')
        error_code = 'karbor.nova.%s' % getattr(err, 'code', 'unknown')
        msg = "Get vm(%s) metadata from nova error, error message: %s" % (
            resource.id, error_msg)
        LOG.exception(msg)
        raise BackupException(code=error_code, message=msg)

    try:
        # 查询网络信息
        _get_vm_network_info(vm_metadata, interfaces, neutron_client)
    except Exception as err:
        error_msg = getattr(err, 'message', 'Unknown error')
        error_code = 'karbor.neutron.%s' % getattr(err, 'code', 'unknown')
        msg = "Get vm(%s) metadata from neutron error, error message: %s" % (
            resource.id, error_msg)
        LOG.exception(msg)
        raise BackupException(code=error_code, message=msg)

    return vm_metadata


def _get_vm_hypervisor(server, nova_client):
    # 查询hypervisor
    property_name = "OS-EXT-SRV-ATTR:hypervisor_hostname"
    hypervisor_hostname = getattr(server, property_name, None)
    hypervisors = nova_client.hypervisors.search(hypervisor_hostname)
    hypervisor = hypervisors[0]
    hypervisor_detail = nova_client.hypervisors.get(hypervisor.id)
    return hypervisor_detail


def _get_vm_network_info(vm_metadata, interfaces, neutron_client):
    # 查询网络详情
    net_work_detail = dict()
    try:
        net_work_detail = _get_vm_network_detail(interfaces, neutron_client)
    except Exception as err:
        LOG.exception("Get vm network detail failed, err: %s" % err)
    vm_metadata["NetworkInfo"] = net_work_detail

    # 获取浮动IP
    floatip_meta_info = []
    subnet_ids = []
    for interface in interfaces:
        floatingips = []
        port_id = interface.port_id
        fixed_ips = interface.fixed_ips
        for fixed_ip in fixed_ips:
            subnet_id = fixed_ip.get("subnet_id")
            subnet_ids.append(subnet_id)
        if not port_id or not fixed_ips:
            continue
        floating_ips = neutron_client.list_floatingips(port_id=port_id)
        for floating_ip in floating_ips.get("floatingips", []):
            # save floatingIPMetaInfo to SNAP ,where ACTIVE status。
            if floating_ip.get("status", None) == "ACTIVE":
                floatingips.append(floating_ip)
        floatip_meta_info.append({"floatingips": floatingips})
    vm_metadata["floatipMetaInfo"] = floatip_meta_info

    # 查询subnets(contain cidr)
    subnet = neutron_client.list_subnets(name='external_relay_network')
    vm_metadata["subnetsMetaInfo"] = subnet


def _get_vm_network_detail(interfaces, neutron_client):
    net_work_detail = dict()
    for interface in interfaces:
        port_id = interface.port_id
        if not port_id:
            LOG.info("os-interface port_id is empty")
            continue

        net_work_detail[port_id] = {}
        port = {}
        try:
            port = neutron_client.show_port(port_id)
        except Exception as err:
            # Ignore all exception.
            LOG.warn('Get neutron port failed, msg: {}'.format(err))

        net_work_detail[port_id]["PortDetail"] = port

        network = _get_vm_network(interface, neutron_client)
        net_work_detail[port_id]["NetworkDetail"] = network

        fixed_ips = interface.fixed_ips
        net_work_detail[port_id]["SubnetDetail"] = []
        for fixed_ip in fixed_ips:
            subnet_id = fixed_ip.get("subnet_id", None)
            if subnet_id:
                # 根据subnet_id获取subnet详情
                try:
                    subnet = neutron_client.show_subnet(subnet_id)
                    net_work_detail[port_id]["SubnetDetail"].append(subnet)
                except Exception as err:
                    # Ignore all exception.
                    LOG.warn('Get subnet info failed, msg: {}'.format(err))

    return net_work_detail


def _get_vm_network(interface, neutron_client):
    net_id = interface.net_id
    network = {}
    if net_id:
        try:
            network = neutron_client.show_network(net_id)
        except Exception as err:
            LOG.error("Get network({id}) info failed, "
                      "msg: {msg}".format(id=net_id, msg=err))
    return network


def build_snap_and_backup_name(parameters, created_at):
    # 约定的前缀 + 时间戳；时间戳使用item的创建时间
    auto_trigger = parameters.get("auto_trigger", False)
    timestamp = str(int(time.mktime(created_at.timetuple()))) + str(created_at.microsecond)
    if auto_trigger:
        snapshot_prefix = AUTO_SNAPSHOT_PREFIX
        backup_prefix = AUTO_BACKUP_PREFIX
    else:
        snapshot_prefix = MANUAL_SNAPSHOT_PREFIX
        backup_prefix = MANUAL_BACKUP_PREFIX
    snapshot_name = snapshot_prefix + timestamp
    backup_name = backup_prefix + timestamp
    return snapshot_name, backup_name


def initial_app_consistency_info(parameters):
    return {
        "app_consistency": parameters.get('app_consistency', 0),
        "app_consistency_status": 0,
        "app_consistency_error_code": 0,
        "app_consistency_error_message": ''
    }


def get_app_consistency_info(context, vm_metadata, app_info):
    app_consistency = app_info.get('app_consistency')
    if app_consistency:
        server = vm_metadata["vmMetaInfo"].get("server", {})
        floating_ips = p_utils.find_float_ips(server)
        agent_ports = CONF.workflow.agent_ports

        if not floating_ips and app_consistency == 1:
            app_info.update({
                "app_consistency_error_code":
                    error_const.EC_WF_CBS_FLOATING_IP_NOT_FOUND,
                "app_consistency_error_message": "Get floating ips failed",
                "app_consistency_status": 1,
            })
            error_code = "karbor.nova.200"
            error_msg = "Get floating ips failed, app_info:%s" % app_info
            LOG.error(error_msg)
            raise BackupException(code=error_code, message=error_msg)

        app_info.update({"floating_ips": floating_ips})

        if not agent_ports and app_consistency == 1:
            app_info.update({
                "app_consistency_error_code":
                    error_const.EC_COM_INTERNAL_ERROR,
                "app_consistency_error_message": "Get agent ports failed",
                "app_consistency_status": 1,
            })
            error_code = "karbor.karbor.404"
            error_msg = "Get agent ports failed, app_info:%s" % app_info
            LOG.error(error_msg)
            raise BackupException(code=error_code, message=error_msg)

        try:
            resp = p_utils.query_agent_status(context, server)
            app_info.update({"is_old": resp.get("IsOld")})
        except workflow_client.WorkflowException as err:
            error_code = "karbor.proxy.%s" % getattr(err, 'code', 'unknown')
            error_msg = "Get agent status failed"
            app_info.update({
                "app_consistency_error_code": err.code,
                "app_consistency_error_message": error_msg,
                "app_consistency_status": 1,
            })
            if app_consistency == 1:
                LOG.exception(error_msg)
                raise BackupException(code=error_code, message=error_msg)
            LOG.info("Get agent status failed, app_consistency_mode:%s"
                     % app_consistency)


def get_existed_snapshots(volume_ids, name, cinder_client):
    """Query and wait all snapshots status change to stable.

    :param volume_ids:
    :param name:
    :param cinder_client:
    :return: available_snapshots = {
                volume_id_1:snapshot_1_dict,
                volume_id_2:snapshot_2_dict,
                volume_id_3:snapshot_3_dict,
             }
             provider_auth = []
    """
    url = "/os-hw_snapshots/detail"
    params = {"name": name}
    while True:
        available_snapshots = dict()
        provider_auth = []
        creating_num = 0
        try:
            _, body = cinder_client.client.get(url, params=params)
        except Exception as err:
            error_msg = getattr(err, 'message', 'Unknown error')
            error_code = getattr(err, 'code', 'unknown')
            msg = "Get snapshots(%s) from cinder error, error message: %s." % (
                name, error_msg)
            LOG.exception(msg)
            raise BackupException(code=error_code, message=msg)
        snapshots = body.get("snapshots", [])
        for snapshot in snapshots:
            if snapshot["volume_id"] not in volume_ids:
                continue

            if snapshot["status"] == "creating":
                creating_num += 1
                continue
            if snapshot["status"] in ["available", "backing-up"]:
                available_snapshots[snapshot["volume_id"]] = snapshot
                provider_auth.append(
                    {"snapshot_provider_location": snapshot["provider_auth"]})
            if snapshot["status"] == "error":
                delete_snapshot(snapshot, cinder_client)
        if creating_num == 0:
            break
        sleep(SNAPSHOTS_LISTEN_TIME)
    return available_snapshots, provider_auth


def create_unactive_snapshots(volume_ids, snapshot_name, cinder_client,
                              resource_az=None):
    """Create unactive snapshots with parameter force=True"""
    if not volume_ids:
        return

    snapshots = dict()
    try:
        metadata = {"__system__enableActive": "false"}
        for volume_id in volume_ids:
            snapshot = cinder_client.volume_snapshots.create(
                volume_id=volume_id, name=snapshot_name, force=True,
                metadata=metadata)
            snapshots[volume_id] = snapshot.id
    except Exception as err:
        error_msg = getattr(err, 'message', 'Unknown error')
        error_code = getattr(err, 'code', 'unknown')
        msg = "Create snapshots(%s) from cinder error, error message: %s." % (
            snapshot_name, error_msg)
        LOG.exception(msg)
        raise BackupException(code=error_code, message=msg)
    LOG.info("Creating snapshots(volume_id:snapshot_id): %s" % snapshots)


def wait_snapshots_finished(volumes_id, snapshot_name, cinder_client,
                            resource_az=None):
    """List snapshots by name.

    :param volumes_id:
    :param snapshot_name:
    :param cinder_client:
    :param resource_az: az of snapshot, same as server or volume.
    :return: available_snapshots={volume_id:snapshot_info_dict}
             snapshots_provider_auth = [{"snapshot_provider_location": "xx"),]
    """
    if not volumes_id:
        return {}, []
    # before query，wait 20s
    sleep(SNAPSHOTS_CONSTANT_LISTEN_TIME)
    url = "/os-hw_snapshots/detail"
    params = {"name": snapshot_name}
    while True:
        creating_num = 0
        snapshots_provider_auth = []
        available_snapshots = dict()
        try:
            _, body = cinder_client.client.get(url, params=params)
        except Exception as err:
            error_msg = getattr(err, 'message', 'Unknown error')
            error_code = getattr(err, 'code', 'unknown')
            msg = "Get snapshots(%s) from cinder error, err_msg: %s" % (
                snapshot_name, error_msg)
            LOG.exception(msg)
            raise BackupException(code=error_code, message=msg)
        snapshots = body.get("snapshots", [])
        for snapshot in snapshots:
            if snapshot["volume_id"] not in volumes_id:
                continue

            if snapshot["status"] == "creating":
                creating_num += 1
                continue
            if snapshot["status"] == "available":
                available_snapshots[snapshot["volume_id"]] = snapshot
                snapshots_provider_auth.append(
                    {"snapshot_provider_location": snapshot["provider_auth"]})
                SNAPSHOT_QUEUE_SWITCH.notify_result(f"create_snapshot_{resource_az}", True)
            if snapshot["status"] == "error":
                delete_snapshot(snapshot, cinder_client)
                SNAPSHOT_QUEUE_SWITCH.notify_result(f"create_snapshot_{resource_az}", False)

        if creating_num == 0:
            break
        sleep(SNAPSHOTS_LISTEN_TIME)

    expect_count = len(volumes_id)
    actual_count = len(available_snapshots)
    _check_result(expect_count, actual_count)
    return available_snapshots, snapshots_provider_auth


def delete_snapshot(snapshot, cinder_client):
    """Delete snapshot.

    :param snapshot: 1) type can be dict or snapshot_obj
                     2) data maybe come from ext_cinder or not
    :param cinder_client:
    :return:
    """
    if isinstance(snapshot, Resource):
        snapshot = snapshot.to_dict()
    try:
        LOG.info("Put error snapshot(%s) into leftovers" % snapshot['id'])
        if 'os-hw_extended-snapshot-attributes:project_id' in snapshot:
            project_id = snapshot[
                'os-hw_extended-snapshot-attributes:project_id']
        else:
            project_id = snapshot['os-extended-snapshot-attributes:project_id']
        db_api.leftovers_create([{
            "project_id": project_id,
            'resource_id': snapshot['volume_id'],
            'clean_type': 'snapshot',
            'id': snapshot['id'],
        }])
    except DBDuplicateEntry:
        pass

    try:
        cinder_client.volume_snapshots.delete(snapshot['id'])
    except Exception:
        LOG.error("Delete snapshot(%s) failed" % snapshot['id'])


def delete_snapshots_by_name(snapshot_name, volumes_id, cinder_client):
    search_opts = {'name': snapshot_name}
    snapshots = cinder_client.volume_snapshots.list(search_opts=search_opts)
    for snapshot in snapshots:
        if snapshot.volume_id not in volumes_id:
            continue
        delete_snapshot(snapshot, cinder_client)


def active_snapshots(provider_auth, snapshots, volume_id, cinder_client):
    snap_ids = []
    metadata_updated = 0
    for snapshot in snapshots.values():
        if snapshot["metadata"].get("__system__enableActive", None) == "true":
            metadata_updated += 1
        else:
            snap_ids.append(snapshot["id"])
    if metadata_updated == len(snapshots):
        LOG.info("All snapshots have been active")
        return

    if metadata_updated == 0:
        # 理论上讲，只要有metadata更新了，就表明已经激活成功了
        body = {
            "snapshots": {
                "volume_snapshots": provider_auth,
                "volume_id": volume_id
            }
        }
        url = "/os-hw_snapshots/active_snapshots"
        try:
            cinder_client.client.post(url, body=body)
        except Exception as err:
            error_msg = getattr(err, 'message', 'Unknown error')
            error_code = getattr(err, 'code', 'unknown')
            LOG.exception("Active snapshots from cinder error")
            raise BackupException(code=error_code, message=error_msg)

    metadata = {"__system__enableActive": "true"}
    for snap_id in snap_ids:
        try:
            cinder_client.volume_snapshots.update_all_metadata(snap_id,
                                                               metadata)
        except Exception as err:
            LOG.exception("Update metadata failed, err:%s" % err)
    LOG.info("Active snapshots(%s) with success" % snap_ids)


def wait_backup(backup_name, volumes_id, cinder_client, operation_log,
                call_back=None):
    """Wait backup finished

    :param call_back:
    :param operation_log:
    :param backup_name:
    :param volumes_id: [volume_id1,volume_id2,volume_id3]
    :param cinder_client:
    :return: {
                volume_id1:backup1,
                volume_id2:backup2,
             }
    """
    ori_progress = operation_log.progress
    listen_interval = _get_backup_listen_interval(backup_name)
    LOG.info("Start to listen backups(%s), listen_interval: %s" %
             (backup_name, listen_interval))
    code = 'karbor.cinder.200'
    desc = None
    ebk_task_id = None
    while True:
        sum_progress = 0
        creating_num = 0
        available_backups = {}
        backups = _list_volume_backups(backup_name, cinder_client)
        for backup in backups:
            if backup.volume_id not in volumes_id:
                continue
            if backup.status == "creating":
                creating_num += 1
            if backup.status in ["available", "restoring"]:
                available_backups[backup.volume_id] = backup
            if backup.status == "error":
                # only keep one backup's fail_reason
                code, desc = _parse_fail_reason(backup)
                ebk_task_id = jsonutils.loads(
                    getattr(backup, "service_metadata", "{}") or "{}"
                ).get("ebk_T_I")
                delete_volume_backup(backup, cinder_client)
            sum_progress += p_utils.extract_progress(backup)
        now_progress = int(sum_progress / len(volumes_id) *
                           PROGRESS_PERCENTAGE) + AZ_COMPLETE
        ori_progress = _update_progress(now_progress, ori_progress,
                                        operation_log, call_back)

        if creating_num == 0:
            break
        sleep(listen_interval)

    operation_log.extra_info['common']['task_id'] = ebk_task_id
    operation_log.save()

    expect_count = len(volumes_id)
    actual_count = len(available_backups)
    _check_result(expect_count, actual_count, code, desc)
    return available_backups


def _update_progress(now_progress, ori_progress, operation_log, call_back):
    if now_progress > ori_progress:
        if call_back:
            call_back(progress=now_progress)
        operation_log.progress = now_progress
        operation_log.save()
        ori_progress = now_progress
    return ori_progress


def _check_result(expect_count, actual_count, code=None, desc=None):
    if actual_count != expect_count:
        error_msg = "Should get %s items. but only get %s. desc: %s" % (
            expect_count, actual_count, desc)
        LOG.error(error_msg)
        if not code:
            code = 'karbor.cinder.200'
        raise BackupException(code=code, message=error_msg)


def _list_volume_backups(backup_name, cinder_client):
    try:
        search_opts = {"name": backup_name}
        backups = cinder_client.backups.list(search_opts=search_opts)
    except Exception as err:
        error_msg = getattr(err, 'message', 'Unknown error')
        error_code = getattr(err, 'code', 'unknown')
        msg = "List backups(%s) from cinder error, err_msg: %s" % (
            backup_name, error_msg)
        LOG.exception(msg)
        raise BackupException(code=error_code, message=msg)
    return backups


def _parse_fail_reason(volume_backup):
    try:
        fail_reason = jsonutils.loads(volume_backup.fail_reason)
        code = fail_reason.get("code")
        desc = fail_reason.get("desc")
    except Exception:
        code = 'karbor.cinder.200'
        desc = volume_backup.fail_reason
    return code, desc


def get_existed_backups(volume_ids, backup_name, cinder_client):
    """如果有创建中的副本需等待其变稳态

    :param volume_ids:
    :param backup_name:
    :param cinder_client:
    :return: {
                volume_id_1:backup_1_obj,
                volume_id_2:backup_2_obj,
                volume_id_3:backup_2_obj,
             }
    """
    listen_interval = _get_backup_listen_interval(backup_name)
    while True:
        available_backups = {}
        creating_num = 0
        backups = _list_volume_backups(backup_name, cinder_client)
        for backup in backups:
            if backup.volume_id not in volume_ids:
                continue
            if backup.status in ["available", "restoring"]:
                available_backups[backup.volume_id] = backup
            if backup.status == "creating":
                creating_num += 1
                continue
            if backup.status == "error":
                delete_volume_backup(backup, cinder_client)

        if creating_num == 0:
            break
        sleep(listen_interval)

    return available_backups


def _get_backup_listen_interval(backup_name):
    """Volume backup name start with 'autobk_' or 'manualbk_'"""
    if backup_name.startswith("autobk_"):
        return BACKUP_AUTO_LISTEN_TIME

    return BACKUP_MANUAL_LISTEN_TIME


def delete_volume_backup(backup, cinder_client):
    try:
        db_api.leftovers_create([{
            'project_id': getattr(backup, "os-bak-tenant-attr:tenant_id"),
            'resource_id': backup.volume_id,
            'clean_type': 'backup',
            'id': backup.id,
        }])
    except DBDuplicateEntry:
        pass

    try:
        cinder_client.backups.delete(backup)
    except Exception:
        LOG.info("Delete error backup %s failed" % backup.id)


def delete_volume_backups_by_name(backup_name, volumes_id, cinder_client):
    search_opts = {'name': backup_name}
    backups = cinder_client.backups.list(search_opts=search_opts)
    res = {}
    for backup in backups:
        if backup.volume_id not in volumes_id:
            continue
        res.update({backup.volume_id: backup})
        delete_volume_backup(backup, cinder_client)
    return res


def release_snapshots(context, backup_id, snapshot_ids, cinder_client):
    """Delete snapshots

    :param context:
    :param backup_id: server backup id
    :param snapshot_ids:
    :param cinder_client:
    :return:
    """
    LOG.info("Will release snapshots %s" % snapshot_ids)
    need_check = []
    for snapshot_id in snapshot_ids:
        try:
            cinder_client.volume_snapshots.delete(snapshot_id)
            need_check.append(snapshot_id)
        except cinder_exc.NotFound:
            pass
        except Exception:
            LOG.exception("Release snapshot %s failed." % snapshot_id)
            _save_snapshot_to_leftover(context, backup_id, snapshot_id)
    _wait_snapshots_delete(context, backup_id, need_check, cinder_client)


def _wait_snapshots_delete(context, backup_id, snapshot_ids, cinder_client):
    success_delete = []
    error_delete = []
    times = 4
    while times >= 0:
        times -= 1
        sleep(30)
        for snapshot_id in snapshot_ids:
            if snapshot_id in success_delete + error_delete:
                continue
            try:
                snapshot = cinder_client.volume_snapshots.get(snapshot_id)
                if snapshot.status in ['available', 'error', 'error_deleting',
                                       'creating']:
                    error_delete.append(snapshot_id)
                    _save_snapshot_to_leftover(context, backup_id, snapshot_id)
            except cinder_exc.NotFound:
                success_delete.append(snapshot_id)
            except Exception as err:
                error_msg = "Get snapshot from cinder error, snapshot " \
                            "id: %s, error info: %s, put it into " \
                            "leftovers." % (snapshot_id, str(err))
                LOG.info(error_msg)
                error_delete.append(snapshot_id)
                _save_snapshot_to_leftover(context, backup_id, snapshot_id)

            if times == 0:
                _save_snapshot_to_leftover(context, backup_id, snapshot_id)

        if len(success_delete) + len(error_delete) == len(snapshot_ids):
            break


def _save_snapshot_to_leftover(context, backup_id, snapshot_id):
    try:
        db_api.leftovers_create([{
            "project_id": context.project_id,
            'resource_id': backup_id,
            'clean_type': 'snapshot',
            'id': snapshot_id,
        }])
    except DBDuplicateEntry:
        pass


def get_unactive_snapshot_info(cinder_client, snapshot_name, volumes,
                               resource_az=None):
    def _get_unactive_snapshot_info():
        # 兼容续作场景，先查询已有快照，再创建缺少的快照
        unactive_snapshots, provider_auth = get_existed_snapshots(
            volumes, snapshot_name, cinder_client)
        available_volumes = [volume_id for volume_id in unactive_snapshots]
        rest_volumes = list(set(volumes).difference(set(available_volumes)))
        create_unactive_snapshots(rest_volumes, snapshot_name, cinder_client,
                                  resource_az=resource_az)
        snapshots, snapshots_provider_auth = wait_snapshots_finished(
            rest_volumes, snapshot_name, cinder_client, resource_az=resource_az)
        provider_auth.extend(snapshots_provider_auth)
        unactive_snapshots.update(snapshots)
        return unactive_snapshots, provider_auth
    lock = SNAPSHOT_QUEUE_SWITCH.get_queue_lock(f"create_snapshot_{resource_az}")
    min_get_lock_randint_timeout_s = 1800
    max_get_lock_randint_timeout_s = 5400
    timeout = random.randint(min_get_lock_randint_timeout_s, max_get_lock_randint_timeout_s)
    try:
        if not lock.acquire(timeout=timeout):
            LOG.warn(f"Failed to get create_snapshot_{resource_az} lock, the wait time is {timeout} seconds.")
        return _get_unactive_snapshot_info()
    finally:
        lock.release()


def active_snapshot_for_server(context, snapshots, cinder_client, app_info,
                               provider_auth, volumes):
    # csbs整机备份快照激活
    app_consistency = app_info["app_consistency"]
    if app_consistency:
        _unfreeze_vm_io(context, app_info)
        _active_app_snapshots(context, provider_auth, snapshots, volumes,
                              app_info, cinder_client)
    else:
        active_snapshots(provider_auth, snapshots, volumes[0], cinder_client)


def _active_app_snapshots(context, provider_auth, available_snapshots, volumes,
                          app_info, cinder_client):
    _freeze_vm_io(context, app_info)
    try:
        active_snapshots(provider_auth, available_snapshots, volumes[0],
                         cinder_client)
        _set_success_app_info(app_info)
    except BackupException as err:
        app_consistency = app_info["app_consistency"]
        if app_consistency == 1:
            _set_fail_app_info(app_info, err.message,
                               error_const.EC_WF_CBS_ACTIVE_SNAPSHOT_FAILED)
        _unfreeze_vm_io(context, app_info)
        raise
    _unfreeze_vm_io(context, app_info)


def _set_success_app_info(app_info):
    if app_info.get('fail_freeze'):
        LOG.info("app_info:%s" % app_info)
        return

    if "is_old" not in app_info:
        return

    is_old = app_info.get("is_old", False)
    if is_old:
        app_consistency_status = APP_BACKUP_SUCC_BY_OLD_AGENT_VERSION
    else:
        app_consistency_status = APP_BACKUP_SUCC
    app_info.update({
        "app_consistency_status": app_consistency_status,
        "app_consistency_error_code": 0,
    })
    LOG.info("app_info:%s" % app_info)


def _set_fail_app_info(app_info, message, code):
    is_old = app_info.get("is_old", False)
    if is_old:
        app_consistency_status = APP_BACKUP_FAIL_BY_OLD_AGENT_VERSION
    else:
        app_consistency_status = APP_BACKUP_FAIL
    app_info.update({
        "app_consistency_status": app_consistency_status,
        "app_consistency_error_code": code,
        "app_consistency_error_message": message
    })


def _freeze_vm_io(context, app_info):
    if 'is_old' not in app_info:
        return
    floating_ips = app_info["floating_ips"]
    try:
        proxy.freeze_vm_io(context, floating_ips)
    except workflow_client.WorkflowException as err:
        app_consistency = app_info["app_consistency"]
        msg = "Freeze cloud machine IO failed"
        if app_consistency == 1:
            _set_fail_app_info(app_info, msg, err.code)
            code = "karbor.proxy.%s" % err.code
            raise BackupException(code=code, message=msg)
        app_info.update({
            "app_consistency_status": 1,
            "app_consistency_error_code": err.code,
            "app_consistency_error_message": msg,
            "fail_freeze": True
        })


def _unfreeze_vm_io(context, app_info):
    # 没有这个key，说明查询agent信息就失败了，没有必要再下发
    # 或者是已经az complete
    if 'is_old' not in app_info:
        return
    floating_ips = app_info["floating_ips"]
    try:
        proxy.unfreeze_vm_io(context, floating_ips)
    except Exception:
        LOG.error("Unfreeze cloud machine IO failed")


def create_volume_backups(snapshots, backup_name, description, cinder_client):
    """Create volume backups by cinder.

    :param snapshots:
    :param backup_name:
    :param description:
    :param cinder_client:
    :return: {
                volume_id_1:backup_1_obj,
                volume_id_2:backup_2_obj,
                volume_id_3:backup_2_obj,
             }
    """
    volumes = [vol_id for vol_id in snapshots]
    available_backups = get_existed_backups(volumes, backup_name,
                                            cinder_client)
    available_volumes = [volume_id for volume_id in available_backups]
    rest_volumes = list(set(volumes).difference(set(available_volumes)))

    all_backups = {}
    for volume_id in rest_volumes:
        create_param = {
            "description": jsonutils.dumps(description),
            "volume_id": volume_id,
            "force": True,
            "name": backup_name,
            "snapshot_id": snapshots[volume_id].get("id")
        }
        try:
            backup = cinder_client.backups.create(**create_param)
        except Exception as err:
            error_msg = getattr(err, 'message', 'Unknown error')
            error_code = getattr(err, 'code', 'unknown')
            msg = "Create backups(%s) from cinder error, err_msg: %s" % (
                backup_name, error_msg)
            LOG.exception(msg)
            raise BackupException(code=error_code, message=msg)
        all_backups[volume_id] = backup
        LOG.info("Creating volume backup: %s, request_id: %s" %
                 (backup.id, backup.request_ids))
    all_backups.update(available_backups)
    return all_backups


def build_app_consistency(parameters, app_backup_info):
    app_consistency = {
        'app_consistency': parameters.get('app_consistency', 0),
        'app_consistency_status':
            app_backup_info.get('app_consistency_status', 0),
        'app_consistency_error_code': app_backup_info.get(
            'app_consistency_error_code', 0),
        'app_consistency_error_message': app_backup_info.get(
            'app_consistency_error_message', '')
    }
    return app_consistency


def build_backup_data(vm_metadata):
    metadata = vm_metadata["vmMetaInfo"]["server"].get("metadata", {})
    flavor = vm_metadata["flavorMetaInfo"].get("flavor", {})
    backup_data = {
        '__openstack_region_name': metadata.get('__openstack_region_name', ''),
        'vcpus': flavor.get('vcpus'),
        'ram': flavor.get('ram'),
        'disk': flavor.get('disk'),
        'imagetype': metadata.get("metering.imagetype"),
        'cloudservicetype': metadata.get("metering.cloudServiceType"),
        'eip': '',
        'private_ip': ''
    }
    return six.binary_type(jsonutils.dumps(backup_data).encode('utf-8'))


def get_space_saving_ratio(volume_backup):
    service_metadata = getattr(volume_backup, 'service_metadata', '{}')
    if service_metadata:
        service_metadata = jsonutils.loads(service_metadata)
        return service_metadata.get('SS', 0)
    return 0


def get_average_speed(volume_backup):
    service_metadata = getattr(volume_backup, 'service_metadata', '{}')
    if service_metadata:
        service_metadata = jsonutils.loads(service_metadata)
        return service_metadata.get('AT', 0)
    return 0


def get_cs(volume_backup):
    service_metadata = getattr(volume_backup, 'service_metadata', '{}')
    if service_metadata:
        service_metadata = jsonutils.loads(service_metadata)
        return service_metadata.get('CS', 0)
    return 0


def set_incremental_info(backups):
    """If 'Type' in service_metadata is 1, it is a incremental backup"""
    for volume_id in backups:
        backup = backups[volume_id]
        service_metadata = getattr(backup, 'service_metadata', '{}')
        if service_metadata:
            service_metadata = jsonutils.loads(service_metadata)
            if service_metadata.get('Type') == 1:
                backup.is_incremental = True
    return backups


def is_incremental(backups):
    """If one volume backup is incremental, server backup is incremental"""
    return any(backup.is_incremental for backup in backups.values())


def get_dl(backup):
    try:
        service_metadata = jsonutils.loads(backup.service_metadata)
        return service_metadata.get("DL")
    except Exception:
        return 0


def is_support_lld(backups):
    for volume_id in backups:
        backup = backups[volume_id]
        service_metadata = jsonutils.loads(backup.service_metadata)
        dl = service_metadata.get("DL")
        if dl is None:
            LOG.info("unable to get DL,return False")
            return False

        if dl == NOT_COMPRESSION_DEDUPLICATION:
            compression = False
            de_duplication = False
        elif dl == COMPRESSION:
            compression = True
            de_duplication = False
        elif dl == DEDUPLICATION:
            compression = False
            de_duplication = True
        elif dl == COMPRESSION_DEDUPLICATION:
            compression = True
            de_duplication = True
        else:
            LOG.info("Unexpected dl value, dl=%s" % dl)
            return False
        backup_id = backup.id
        data_layout = {
            "compression": compression,
            "de_duplication": de_duplication
        }
        if compression or de_duplication:
            LOG.info("The data layout of backup %s is %s, does not "
                     "support lld." % (backup_id, data_layout))
            return False
    return True


def get_backup_description(context, parameters, resource, app_info):
    app_consistency_status = app_info["app_consistency_status"]
    if app_consistency_status in [1, 11]:
        app = 1
    elif app_consistency_status in [2, 12]:
        app = 2
    else:
        app = 0
    incremental = parameters.get("incremental", True)
    description = OrderedDict()
    description["ST"] = 1
    description["INC"] = 1 if incremental else 0
    description["VMID"] = resource.id
    description["DEC"] = 0
    description["APP"] = app
    description["CBR"] = 0
    LOG.info("description: %s" % jsonutils.dumps(description))
    return description


def is_servers_restoring(context, volumes_metadata):
    times = 20
    server_ids = _get_all_servers(volumes_metadata)
    if len(server_ids) == 1:
        return

    LOG.info("Check if servers(%s) are restoring" % server_ids)
    filters = [
        {'opt': '=', 'value': "running", 'key': 'status'},
        {'opt': 'in', 'value': server_ids, 'key': 'restore_id'}
    ]
    while times >= 0:
        times -= 1
        count = db_api.operation_log_count(context, filters)
        if count == 0:
            LOG.info("Servers(%s) are not restoring." % server_ids)
            return
        sleep(RESTORE_LISTEN_TIME)
    code = "karbor.karbor.500"
    msg = "Code:%s, err_msg: Time out, when check servers" % code
    raise BackupException(code=code, message=msg)


def _get_all_servers(volumes_metadata):
    server_ids = []
    for volume_id in volumes_metadata:
        vol_metadata = volumes_metadata[volume_id]
        attachments = vol_metadata["volume"].get("attachments", [])
        for attachment in attachments:
            server_id = attachment.get("server_id")
            if server_id not in server_ids:
                server_ids.append(server_id)
    return server_ids


def is_encrypted(volume_backup):
    try:
        if not volume_backup.service_metadata:
            return False

        service_metadata = jsonutils.loads(volume_backup.service_metadata)
        cmk_id = service_metadata.get('CMKID', '')
        if cmk_id:
            return True
    except Exception as err:
        LOG.error("get encrypted info failed, err:%s" % err)

    return False


def get_az(ck_item):
    extend_info = jsonutils.loads(ck_item.extend_info)
    return ck_item.get("resource_az") or extend_info.get("resource_az")


def send_retry_failed_alarm(context, operation_log):
    job = db_api.job_get(context, operation_log.job_id)
    param = jsonutils.loads(job.parameters) if job.parameters else {}
    retry_times = param.get('retry_param', {}).get('retry_times', 0)
    auto_trigger = param.get("auto_trigger")
    # 在备份job重试结束，任务状态置为失败的同时发送告警，保证任务失败时间与告警发送时间一致。
    # 任务失败后进入CompleteProtectTask中，retry_times加1，然后判断若retry_times <= CONF.backup_retry_limit + 1，则
    # 进入下一次重试，再次重试时_check_retry_times判断retry_times > CONF.backup_retry_limit，则停止重试。
    LOG.info(f"Check retry_times {retry_times} for op_log_id {operation_log.id} of job_id {operation_log.job_id}")
    if auto_trigger and retry_times >= CONF.backup_retry_limit:
        error_code = operation_log.error_info.get('code', 'CSBS.9999')
        kangaroo_alarms.send_backup_failed_alarm(operation_log.id, context, error_code)
