import copy
import datetime
import random

from cinderclient import exceptions as cinder_exception
from eventlet import sleep
from oslo_db.exception import DBDuplicateEntry
from oslo_log import log as logging
from oslo_serialization import jsonutils

from kangaroo.db import api as db_api
from kangaroo.plugin.utils import utils as p_utils
from kangaroo.protection.queue_kits import SNAPSHOT_QUEUE_SWITCH

LOG = logging.getLogger(__name__)
BACKEND_CACHE = dict()
WAIT_SNAPSHOT_INTERVAL = 20
WAIT_SNAPSHOT_LIMIT = 900
WAIT_VOLUME_BACKUP_INTERVAL = 120


class DeleteException(Exception):
    def __init__(self, code='', description='', body=None):
        super(DeleteException, self).__init__()
        self.code = code
        self.description = description
        self.body = body or {}


def delete_volume_snapshots(context, cinder_client, snapshot_ids,
                            resource_az=None):
    LOG.info("Begin delete volume snapshots %s" % snapshot_ids)
    deleted_snapshots = set()
    for snapshot_id in snapshot_ids:
        # get snapshot, and check its status
        try:
            snapshot = p_utils.get_snapshot_by_snapshot_id(
                cinder_client, snapshot_id)
        except Exception as err:
            LOG.info("Get snapshot %s from cinder failed, error info: %s." %
                     (snapshot_id, str(err)))
            _save_snapshot_to_leftovers(context, snapshot_id)
            continue
        if not snapshot:
            deleted_snapshots.add(snapshot_id)
            continue
        if snapshot.status in ["deleting"]:
            continue
        if snapshot.status in ["error_deleting", "backing-up"]:
            try:
                _reset_volume_snapshot_status(
                    cinder_client, snapshot_id, 'error')
            except Exception as err:
                error_msg = "Reset snapshot status from cinder error, " \
                            "snapshot id: %s, error info: %s." % (
                                snapshot_id, str(err))
                LOG.info(error_msg)
                _save_snapshot_to_leftovers(context, snapshot_id)
                continue

        # delete snapshot
        try:
            cinder_client.volume_snapshots.delete(snapshot_id)
        except cinder_exception.NotFound:
            LOG.info("The snapshot is not exist in cinder, snapshot id: "
                     "%s" % snapshot_id)
            deleted_snapshots.add(snapshot_id)
        except Exception as err:
            error_msg = "Delete snapshot from cinder error, snapshot id: " \
                        "%s, error info: %s, put it into leftovers." % \
                        (snapshot_id, str(err))
            LOG.info(error_msg)
            _save_snapshot_to_leftovers(context, snapshot_id)
    return deleted_snapshots


def _reset_volume_snapshot_status(cinder_client, snapshot_id, status):
    try:
        cinder_client.volume_snapshots.reset_state(
            snapshot_id, status)
    except cinder_exception.NotFound:
        LOG.info("The snapshot is not exist in cinder, snapshot id: "
                 "%s" % snapshot_id)
        return None
    except Exception as err:
        error_msg = "Reset snapshot status from cinder error, " \
                    "snapshot id: %s, error info: %s." % (
                        snapshot_id, str(err))
        LOG.exception(error_msg)
        raise


def wait_delete_volume_snapshots(context, cinder_client, snapshot_ids, resource_az=None):
    if not snapshot_ids:
        return
    start_at = datetime.datetime.utcnow()
    snapshot_ids = copy.deepcopy(snapshot_ids)
    left_snapshot_ids = copy.deepcopy(snapshot_ids)
    while True:
        for snapshot_id in snapshot_ids:
            try:
                snapshot = p_utils.get_snapshot_by_snapshot_id(
                    cinder_client, snapshot_id)
                if not snapshot:
                    SNAPSHOT_QUEUE_SWITCH.notify_result(f"delete_snapshot_{resource_az}", True)
                    left_snapshot_ids.remove(snapshot_id)
                    continue
                if snapshot.status in ['available', 'error', 'error_deleting', 'creating']:
                    SNAPSHOT_QUEUE_SWITCH.notify_result(f"delete_snapshot_{resource_az}", False)
                    _save_snapshot_to_leftovers(context, snapshot_id)
                    left_snapshot_ids.remove(snapshot_id)
            except Exception as err:
                error_msg = "Get snapshot from cinder error, snapshot " \
                            "id: %s, error info: %s, put it into " \
                            "leftovers." % (snapshot_id, str(err))
                LOG.info(error_msg)
                _save_snapshot_to_leftovers(context, snapshot_id)
                left_snapshot_ids.remove(snapshot_id)

        end_at = datetime.datetime.utcnow()
        if (end_at - start_at).seconds > WAIT_SNAPSHOT_LIMIT:
            for snapshot_id in left_snapshot_ids:
                _save_snapshot_to_leftovers(context, snapshot_id)
            break
        if not left_snapshot_ids:
            break

        snapshot_ids = copy.deepcopy(left_snapshot_ids)
        sleep(WAIT_SNAPSHOT_INTERVAL)


def delete_volume_backups(cinder_client, vol_backup_ids):
    LOG.info("Begin delete volume backup %s" % vol_backup_ids)

    for vol_backup_id in vol_backup_ids:
        # get volume backup, and check its status
        try:
            c_vol_backup = p_utils.get_volume_backup_by_backup_id(
                cinder_client, vol_backup_id)
        except Exception as err:
            error_code = getattr(err, 'code', 'Unknown')
            error_msg = getattr(err, 'message', 'Unknown error')
            msg = "Get backup %s from cinder error, error message: %s." % (
                vol_backup_id, error_msg)
            LOG.info(msg)
            raise DeleteException(code=error_code, description=msg)
        if not c_vol_backup:
            continue
        if c_vol_backup.status in ["deleting"]:
            continue
        if c_vol_backup.status in ["error_deleting"]:
            error_code, error_des = _build_error_info_from_failreason(
                c_vol_backup.fail_reason)
            raise DeleteException(code=error_code, description=error_des)

        # delete volume backup
        try:
            cinder_client.backups.delete(vol_backup_id)
        except cinder_exception.NotFound:
            LOG.info("Backup is not exist in cinder, backup id: %s" %
                     vol_backup_id)
        except Exception as err:
            error_code = getattr(err, 'code', 'Unknown')
            error_msg = getattr(err, 'message', 'Unknown error')
            msg = "Volume backup delete error from cinder, backup id: %s, " \
                  "error info: %s" % (vol_backup_id, str(error_msg))
            LOG.info(msg)
            raise DeleteException(code=error_code, description=msg)


def wait_delete_volume_backups(cinder_client, vol_backup_ids, operation_log,
                               callback=None, success_vol_backup_ids=None):
    ori_progress = operation_log.progress
    success_vol_backup_ids = _get_success_vol_backup_ids(
        success_vol_backup_ids)
    while True:
        sum_progress = 0
        failed = 0
        running = 0
        for vol_backup_id in vol_backup_ids:
            c_vol_backup = _get_cinder_volume_backup(
                cinder_client, vol_backup_id, success_vol_backup_ids)
            if c_vol_backup is None:
                if vol_backup_id not in success_vol_backup_ids:
                    success_vol_backup_ids.append(vol_backup_id)
                sum_progress += 100
            elif c_vol_backup.status in ['error', 'error_deleting']:
                error_code, error_des = _build_error_info_from_failreason(
                    c_vol_backup.fail_reason)
                failed += 1
            else:
                sum_progress += p_utils.extract_progress(c_vol_backup)
                running += 1
        now_progress = sum_progress / len(vol_backup_ids)
        ori_progress = _update_progress(
            now_progress - 1, ori_progress, operation_log, callback)

        if running == 0 and failed == 0:
            return
        elif running == 0 and failed != 0:
            raise DeleteException(code=error_code, description=error_des)
        sleep(WAIT_VOLUME_BACKUP_INTERVAL)


def _get_success_vol_backup_ids(success_vol_backup_ids):
    if success_vol_backup_ids is None:
        success_vol_backup_ids = list()
    return success_vol_backup_ids


def _update_progress(now_progress, ori_progress, operation_log, callback):
    if now_progress > ori_progress:
        if callback:
            callback(progress=now_progress)
        operation_log.progress = now_progress
        operation_log.save()
        ori_progress = now_progress
    return ori_progress


def _get_cinder_volume_backup(cinder_client, vol_backup_id,
                              success_vol_backup_ids):
    if vol_backup_id in success_vol_backup_ids:
        return None

    try:
        c_vol_backup = p_utils.get_volume_backup_by_backup_id(
            cinder_client, vol_backup_id)
    except Exception as err:
        error_code = getattr(err, 'code', 'Unknown')
        error_msg = getattr(err, 'message', 'Unknown error')
        msg = "Get backup %s from cinder error, error message: %s." % (
            vol_backup_id, error_msg)
        LOG.info(msg)
        raise DeleteException(code=error_code, description=msg)
    return c_vol_backup


def _build_error_info_from_failreason(fail_reason):
    try:
        fail_reason = jsonutils.loads(fail_reason)
        error_code = fail_reason.get('code')
        error_des = fail_reason.get('desc')
    except Exception:
        error_code = 'Unknown'
        error_des = fail_reason
    return error_code, error_des


def _save_snapshot_to_leftovers(context, snapshot_id):
    try:
        db_api.leftovers_create([{
            'project_id': context.project_id,
            'resource_id': snapshot_id,
            'clean_type': 'snapshot',
            'id': snapshot_id,
        }])
    except DBDuplicateEntry:
        pass


def get_resource_az_for_rep_item(context, item_id):
    filters = [{'value': item_id, 'key': 'checkpoint_item_id', 'opt': '='}]
    records = db_api.replication_records_get_all(context, filters=filters)
    if not records:
        return None
    global BACKEND_CACHE
    source_region_id = records[0].source_region
    if source_region_id in BACKEND_CACHE:
        return BACKEND_CACHE[source_region_id]
    backend_cache = db_api.backend_cache_get(
        {"source_region_id": source_region_id})
    BACKEND_CACHE[source_region_id] = backend_cache.az_name
    return backend_cache.az_name


def delete_and_wait_volume_snapshots(context, cinder_client, snapshots, resource_az):
    def _delete_and_wait_volume_snapshots():
        deleted_snapshots = delete_volume_snapshots(
            context, cinder_client, snapshots, resource_az=resource_az)
        wait_delete_volume_snapshots(
            context, cinder_client, set(snapshots) - deleted_snapshots,
            resource_az=resource_az)
    lock = SNAPSHOT_QUEUE_SWITCH.get_queue_lock(f"delete_snapshot_{resource_az}")
    min_get_lock_randint_timeout_s = 1800
    max_get_lock_randint_timeout_s = 5400
    timeout = random.randint(min_get_lock_randint_timeout_s, max_get_lock_randint_timeout_s)
    try:
        if not lock.acquire(timeout=timeout):
            LOG.warn(f"Failed to get delete_snapshot_{resource_az} lock, the wait time is {timeout} seconds.")
        return _delete_and_wait_volume_snapshots()
    finally:
        lock.release()
