#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
from functools import lru_cache

from oslo_config import cfg
from oslo_log import log

from kangaroo.client.iam import get_op_service_token
from kangaroo.client.manageone.sc_client import SCClient
from kangaroo.dfxextend.common import dfx_utils

LOG = log.getLogger(__name__)
CONF = cfg.CONF


class ProjectData(object):
    def __init__(self, project_info, region_id):
        self.tenant_name = project_info.get("domain_name")
        self.tenant_id = project_info.get("domain_id")
        self.vdc_name = project_info.get("tenant_name")
        self.vdc_id = project_info.get("tenant_id")
        self.project_name = project_info.get("name")
        self.project_id = project_info.get("id")
        self.regions = project_info.get("regions", [])
        for region in self.regions:
            if region.get("region_id") == region_id:
                self.region_id = region_id
                self.region_status = region.get("region_status")
                self.region_name = region.get("region_name")
                azs = region.get("cloud_infras")[0].get("azs")
                self.az_dict = {}
                for az_info in azs:
                    self.az_dict[az_info.get("az_id")] = az_info
                break
        else:
            self.region_id = region_id
            self.region_status = None
            self.region_name = None
            self.az_dict = {}
            LOG.warning("This region is not configured for the current resource space, "
                        f"project_id:{self.project_id}, region_id:{region_id}.")


class ReportContext(object):
    def __init__(self):
        self.region_id = self.get_region_id()
        self.region_name = self.get_region_name()
        self.sc_client = SCClient(CONF.sc_client.sc_endpoint, CONF.features.sc_price_rate_switch)
        self.projects_dict = {}
        self.az_dict = {}
        self._token = None

    @staticmethod
    def get_region_id():
        return dfx_utils.get_info("region_id")

    @staticmethod
    def get_region_name():
        return dfx_utils.get_info("region_name")

    def get_azs_info(self):
        """

        :return: 返回类型：dict， 数据结构示例：
        {
          "az0.dc0": {
            "az_id": "az0.dc0",
            "region_id": "sa-fb-1",
            "cloud_infra_id": "FUSION_CLOUD_sa-fb-1",
            "name": "北京",
            "type": "KVM",
            "status": "normal",
            "extend_param": "{\"cloud_location\":\"localcloud\"}"
          }
        }
        """
        if not self._token:
            self._token = get_op_service_token()
        query_azs = self.sc_client.query_azs(self._token)
        az_dict = {}
        for az_info in query_azs:
            az_dict[az_info.get("az_id")] = az_info
        return az_dict

    def get_project_data(self, project_id):
        project_info = self.projects_dict.get(project_id)
        if project_info:
            return self.projects_dict
        try:
            project_info = self.sc_client.get_project_detail_by_id(project_id)
        except Exception as err:
            LOG.warning("Failed to obtain the resource space information, this resource space may have been deleted, "
                        f"project_id:{project_id}, err_msg:{err}.")
        if project_info:
            domain_id = project_info.get("domain_id")
            domain_info = self.sc_client.get_tenant_info(domain_id)
            project_info["domain_name"] = domain_info.get("name") if domain_info else None
            self.projects_dict[project_id] = ProjectData(project_info, self.region_id)
        else:
            self.projects_dict[project_id] = ProjectData({"id": project_id}, self.region_id)
        return self.projects_dict

    def get_az_info_by_az_id(self, az_id):
        if self.az_dict.get(az_id):
            return self.az_dict.get(az_id)
        self.az_dict = self.get_azs_info()
        if not self.az_dict.get(az_id):
            raise Exception(f"Failed to obtain AZ information, az id:{az_id}.")
        return self.az_dict.get(az_id)

    @lru_cache
    def get_az_id_by_az_name(self, az_name):
        if not self._token:
            self._token = get_op_service_token()
        query_azs = self.sc_client.query_azs(self._token, self.region_id)
        for az in query_azs:
            if az.get("name") == az_name:
                return az.get("az_id")
        raise Exception(f"Failed to obtain the az_id, az_name:{az_name}")
