#!/bin/bash
set +x
CURRENT_PATH=$(dirname "${BASH_SOURCE[0]}")
LINE_ROOT=$(dirname "$(dirname "${CURRENT_PATH}")")
source "${LINE_ROOT}/bin/utils.sh"
source /opt/huawei/dj/etc/zookeeper/java.env

DB_FLOATIP=$(get_info gaussdb_float_ip)
THIRD_PARTY_PATH="${LINE_ROOT}/repo/third_party/"
######################################################################
#   FUNCTION   : generate_ssl_cert
#   DESCRIPTION: 生成ssl证书，用户DataSync工具连接数据库
######################################################################
function generate_ssl_cert()
{
  cert_type=$1
  password="$(get_safe_password "$(decrypt_pwd "$2")")"
  cert_dir="${LINE_ROOT}/gaussdb_cert/${cert_type}"
  rm -rf "${cert_dir}" && mkdir -p "${cert_dir}"
  gaussdb_cert="/opt/huawei/dj/bin/gaussdb/cert"
  ca_cert="${gaussdb_cert}/ca-cert.pem"
  cd "$cert_dir" || exit 1

  log_info "Generate cacert.crt.der."
  openssl x509 -in ${ca_cert} -out cacert.crt.der -outform der
  CHECK_RESULT $? "Generate cacert.crt.der failed!"
  log_info "Generate gs_truststore."
  /usr/bin/expect<<-EOF
    set timeout 60
    spawn keytool -keystore gs_truststore -alias cacert -import -file cacert.crt.der
    expect {
        "Enter keystore password:" {send "${password}\r"}
    }
    expect {
        "Re-enter new password:" {send "${password}\r"}
    }
    expect {
        "Trust this certificate?" {send "yes\r"}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
  CHECK_RESULT $? "Generate gs_truststore failed."

  log_info "Generate client.pkcs12."
  /usr/bin/expect<<-EOF
    set timeout 60
    spawn openssl pkcs12 -export -out client.pkcs12 -in "${gaussdb_cert}/client.crt" -inkey "${gaussdb_cert}/client.key"
    expect {
        "Enter pass phrase for" {send "${password}\r"}
    }
    expect {
        "Enter Export Password:" {send "${password}\r"}
    }
    expect {
        "Verifying - Enter Export Password:" {send "${password}\r"}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
  CHECK_RESULT $? "Generate client.pkcs12 failed."

  /usr/bin/expect<<-EOF
    set timeout 60
    spawn keytool -importkeystore -destkeystore client.jks -srckeystore client.pkcs12 -srcstoretype PKCS12 -alias 1
    expect {
        "Enter destination keystore password:" {send "${password}\r"}
    }
    expect {
        "Re-enter new password:" {send "${password}\r"}
    }
    expect {
        "Enter source keystore password" {send "${password}\r"}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
  CHECK_RESULT $? "Generate client.jks failed."
  sed -i "s|.*${cert_type}_trust_store.*|\"trust_store\":\"${cert_dir}/gs_truststore\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
  sed -i "s|.*${cert_type}_key_store.*|\"key_store\":\"${cert_dir}/client.jks\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
  cd - || exit 1
}

######################################################################
#   FUNCTION   : generate_database_password
#   DESCRIPTION: 数据库密码加密，并将密文写入配置文件
######################################################################
function generate_database_password()
{
    cd "${LINE_ROOT}/DataSync" || exit 1
    pwd_type="$1"
    encrypted_password="$2"
    password=$(get_safe_password "$(decrypt_pwd "${encrypted_password}")")
    /usr/bin/expect<<-EOF
    set timeout 60
    spawn java -jar DSS.jar -p config/cfg.ini -pwd ${pwd_type}
    expect {
        "Please enter the password to be encrypted and press Enter to confirm!" {send "${password}\r"}
    }
    expect {
        "config/cfg.ini succeed" {exit 1}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
  cd - || exit 1
}

######################################################################
#   FUNCTION   : prepare_data_sync_tool
#   DESCRIPTION: 准备DataSync工具
######################################################################
function prepare_data_sync_tool()
{
  backup_dir="${LINE_ROOT}/gaussdb_backup"
  rm -rf "${backup_dir}" && mkdir -p "${backup_dir}"
  tar -zxf "${THIRD_PARTY_PATH}/DataSync.tar.gz" --no-same-owner -C "${LINE_ROOT}"
  sed -i "s|.*source_database_ip.*|\"ip\":\"${DB_FLOATIP}\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
  sed -i "s|.*target_database_ip.*|\"ip\":\"${DB_FLOATIP}\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
  sed -i "s|.*export_local_path.*|\"export_local_path\":\"${backup_dir}\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
  sed -i "s|.*import_local_path.*|\"import_local_path\":\"${backup_dir}\",|" "${LINE_ROOT}/DataSync/config/cfg.ini"
}