#!/bin/bash
CURR_PATH=$(dirname "$(readlink -f "$0")")
LINE_ROOT=$(dirname "$(dirname "${CURR_PATH}")")
source "${LINE_ROOT}/bin/utils.sh"
source "${LINE_ROOT}/services/gaussdb/utils.sh"
source /opt/huawei/dj/etc/zookeeper/java.env
export LOG_TAG="csbs_install"

db_primary=6          #   主正常运行
db_standby=7          #   备正常运行
IMPORT_DATABASE_PWD=2
IMPORT_DATABASE_TRUSTSTORE_PWD=8
IMPORT_DATABASE_KEYSTORE_PWD=9

#######################################
# gaussdb升级后处理过程
# 1. gaussdb升级完成后，数据已经完成恢复到初始状态，只有karbor一个账户
# 2. gaussdb升级完成后，需要完成以下几个步骤：
# Primary节点:
# 1) 修改pg_hba，开启openstack免密登录
# 2) 使用openstack账户将karbor用户初始密码修改为当前密码
# 3) 在主端Primary节点创建数据库用户、创建数据库
# 4) 使用DataSync在主端Primary节点完成数据恢复
#######################################
######################################################################
# 功能说明  : 修改数据库用户和密码
######################################################################
function modify_usr_password()
{
    export CSBS_PYTHON_DATA1="$1"
    export CSBS_PYTHON_DATA2="$2"
    (
        echo "import os"
        echo "from basesdk import db_client"
        echo "from kmc import kmc"
        echo "try:"
        echo "    db_user = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    db_pass = kmc.API().decrypt(0, os.getenv('CSBS_PYTHON_DATA2'))"
        echo "    exit(db_client.modify_usr_password(db_user, db_pass))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}

######################################################################
#   FUNCTION   : get_cms_from_db
#   DESCRIPTION: 直接从cms数据库获取cms info
######################################################################
function get_cms_info_from_db()
{
  export CSBS_PYTHON_DATA1="$1"
  (
        echo "import os"
        echo "from basesdk import monkey_patches"
        echo "monkey_patches.do_patch_sqlalchemy()"
        echo "monkey_patches.do_patch_oslo_config_ssl()"
        echo "from cms.restSrvCtl import CONF, FAKE_ARGV"
        echo "CONF(FAKE_ARGV, project='cms', prog='cms')"
        echo "from cms.db import api as db_api"
        echo "try:"
        echo "    cms_key = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    print(db_api.get_item_value(cms_key))"
        echo "except Exception as e:"
        echo "    exit(1)"
    ) | csbs_python
}

######################################################################
#   FUNCTION   : change_karbor_pwd
#   DESCRIPTION: 修改karbor数据库用户密码
######################################################################
function change_karbor_pwd()
{
  # 修改karbor用户密码
  \cp -f "${CURR_PATH}/config_db_user.sh" /home/openstack
  chown openstack:openstack /home/openstack/config_db_user.sh
  bash /home/openstack/config_db_user.sh; result=$?
  rm -rf /home/openstack/config_db_user.sh
  CHECK_RESULT $result "Change db password failed!"
}

######################################################################
#   FUNCTION   : change_db_user_passwd
#   DESCRIPTION: 根据cms中的内容修改数据库用户密码
######################################################################
function change_db_user_passwd()
{
  services="karbor alarm haproxy resource_manager"
  for service in ${services};do
    log_info "Get ${service} db user info from cms."
    user_info=$(get_cms_info_from_db "${service}.db_user_pass")
    user_info=$(get_dict_info "${user_info}" "item_value")
    db_user=$(get_dict_info "${user_info}" "${service}_db_user")
    if [[ "${db_user}" == "" ]];then
      continue
    fi
    db_pass=$(get_dict_info "${user_info}" "${service}_db_pass")
    log_info "Begin modify ${db_user} db pass."
    modify_usr_password "${db_user}" "${db_pass}"
    CHECK_RESULT $?  "Change db user password failed."
  done
}

######################################################################
#   FUNCTION   : create_role_and_db
#   DESCRIPTION: 创建数据库用户、创建数据库
######################################################################
function create_role_and_db()
{
  change_karbor_pwd
  DEFAULT_DB_PWD=$(get_info "gaussdb_admin_pwd")
  users="resource_manager resource_manager_a resource_manager_b alarm_a alarm_b cms_a cms_b haproxy_a haproxy_b karbor_a karbor_b"
  cms_connection_info=$(awk '/connection/' /opt/huawei/dj/etc/cms/cms.conf  | awk '{print $NF}')
  cms_db_user=$(echo "${cms_connection_info}" | awk -F '//' '{print $2}' | awk -F ':' '{print $1}')
  for user in ${users};do
    log_info "Start create ${user} user"
    if [[ "${user}" == "${cms_db_user}" ]]; then
      cms_db_pwd=$(echo "${cms_connection_info}" | awk -F '//' '{print $2}' | awk -F ':' '{print $2}' | awk -F '@' '{print $1}')
      create_user_in_role "${user}" "${cms_db_pwd}"
    else
      create_user_in_role "${user}" "${DEFAULT_DB_PWD}"
    fi
    CHECK_RESULT $?  "Create db user failed."
  done
  databases="karbor alarm cms haproxy resource_manager"
  for database in ${databases};do
    log_info "Start create ${database} database"
    create_db_for_owner "${database}"
    CHECK_RESULT $? "Create db ${database} failed"
  done
  create_enum_type
  CHECK_RESULT $? "Create enum type failed"
}

function gaussdb_restore()
{
  # Primary节点 进行数据恢复
  log_info "begin gaussdb restore action."
  privkey_file="/opt/huawei/dj/DJSecurity/privkey/privkey.conf"
  privkey=$(awk '{print $2}' ${privkey_file})
  generate_ssl_cert target "$privkey"
  GAUSSPASSWORD=$(get_info gaussdb_admin_pwd)
  generate_database_password ${IMPORT_DATABASE_PWD} "${GAUSSPASSWORD}"
  generate_database_password ${IMPORT_DATABASE_TRUSTSTORE_PWD} "${privkey}"
  generate_database_password ${IMPORT_DATABASE_KEYSTORE_PWD} "${privkey}"
  sed -i "s/.*flow_type.*/\"flow_type\":2,/" "${LINE_ROOT}/DataSync/config/cfg.ini"
  cd "${LINE_ROOT}/DataSync" || exit 1
  java -jar DSS.jar -p config/cfg.ini; result=$?
  return $result
}

function post_upgrade_gaussdb()
{
  # 等待数据库状态正常，local_role peer_role 需要有一个主
  db_state=""
  local retry=0
  while [[ "${db_state}" != "${db_primary}" ]] && [[ "${db_state}" != "${db_standby}" ]] && [[ ${retry} -lt 10 ]]; do
    retry=$((retry+1))
    bash /opt/huawei/dj/tools/gaussdb/gaussdb_helper.sh status; db_state=$?
    log_info "Current db state is ${db_state}."
    sleep 10
  done
  if [[ "${db_state}" != "${db_primary}" ]] && [[ "${db_state}" != "${db_standby}" ]];then
    log_error "Database is not normal."
    exit 1
  fi
  if [[ "${db_state}" == "${db_primary}" ]];then
      create_role_and_db
      CHECK_RESULT $? "Failed to create role and database."
      cmd_retry gaussdb_restore 120 10
      CHECK_RESULT $? "Failed to restore gaussdb."
      change_db_user_passwd
      CHECK_RESULT $? "Failed to change db user passwd."
  fi
}

log_info "begin gaussdb Post-Upgrade."
post_upgrade_gaussdb
touch "${CURR_PATH}/post-gaussdb_upgrade_finish"
log_info "Post-Upgrade gaussdb success."