#!/bin/bash
###############################################################
# 功能说明: 调整系统环境变量
##############################################################
export PATH=/opt/huawei/dj/sbin:${PATH}
export PATH=/opt/huawei/dj/bin/erlang/bin:${PATH}
if [ "${LD_LIBRARY_PATH}" == "" ]; then
    export LD_LIBRARY_PATH=/opt/huawei/dj/bin/python/lib
else
    export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:/opt/huawei/dj/bin/python/lib
fi
export PYTHONDONTWRITEBYTECODE=1
umask 077
cd ~ || exit 1
###############################################################
# 功能说明: 定义常用变量
##############################################################
LOG_CONFIG=$(dirname "$(dirname "${BASH_SOURCE[0]}")")/cfg/rsyslog_csbs.conf
MAX_UNCOMPRESSED_SIZE=$(( 1 * 1024 * 1024 * 1024 * 20 ))
MAX_UNCOMPRESSED_FILE_COUNT=$(( 1 * 1000 * 100))
###############################################################
# 功能说明: 公共方法
# 约束: conf配置为[section]下key=value
##############################################################
function read_ini_file()
{
    awk -F '=' '/\['"$2"'\]/{a=1}a==1&&$1~/'"$1"'/{print $2;exit}' "$3" | awk '$1=$1'
}
###############################################################
# 功能说明: 日志公共方法
# 约束: 通过source 导入该工具文件时，需要在其脚本中定义 LOG_TAG 变量
# 约束值: LOG_TAG; 日志标记
##############################################################
function LOG()
{
    log_file=$(read_ini_file "${LOG_TAG}" "LOG_FILE" "${LOG_CONFIG}")
    [[ "$3" -ne 0 ]] && echo "[$1] ${*:4}"
    mkdir -p "$(dirname "${log_file}")"
    echo "$(date "+%Y-%m-%d %H:%M:%S.%N" | cut -b 1-23) $1 ${LOG_TAG} [pid:$$] [$(basename "$0"):$2] ${*:4}" >> "${log_file}"
}
shopt -s expand_aliases
alias log_error='LOG ERROR ${LINENO} 0'
alias log_info='LOG INFO ${LINENO} 0'
alias log_warn='LOG WARNING ${LINENO} 0'
alias log_debug='LOG DEBUG ${LINENO} 0'
alias echo_error='LOG ERROR ${LINENO} 1'
alias echo_info='LOG INFO ${LINENO} 1'
alias echo_warn='LOG WARNING ${LINENO} 1'
alias echo_debug='LOG DEBUG ${LINENO} 1'
###############################################################
# 功能说明: 日志公共方法-异常打印日志退出
# 约束: 通过source 导入该工具文件时，需要在其脚本中定义 LOG_TAG 变量
# 约束值: LOG_TAG; 日志标记
##############################################################
function CHECK_RESULT()
{
    [[ "$1" -ne 0 ]] && log_error "$2" && exit "$1"
    return 0
}

###############################################################
# 功能说明: 日志公共方法-异常输出回显退出
# 约束: 通过source 导入该工具文件时，需要在其脚本中定义 LOG_TAG 变量
# 约束值: LOG_TAG; 日志标记
##############################################################
function ECHO_RETURN()
{
    [[ "$1" -ne 0 ]] && echo_error "$2" && exit "$1"
    return 0
}
###############################################################
# 功能说明: 读取配置文件key=value
# 约束: conf配置为key=value
##############################################################
function get_pwd_with_python()
{
    export CSBS_PYTHON_DATA_SECTION="$1"
    export CSBS_PYTHON_DATA_OPTION="$2"
    export CSBS_PYTHON_DATA_CONF_PATH="$3"
    (
        echo "import os"
        echo "import configparser"
        echo "try:"
        echo "    section = os.getenv('CSBS_PYTHON_DATA_SECTION')"
        echo "    option = os.getenv('CSBS_PYTHON_DATA_OPTION')"
        echo "    conf_path = os.getenv('CSBS_PYTHON_DATA_CONF_PATH')"
        echo "    config = configparser.ConfigParser()"
        echo "    config.read(conf_path)"
        echo "    value = config.get(section, option)"
        echo "    print(value)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
###############################################################
# 功能说明: 读取配置文件key=value
# 约束: conf配置为key=value
##############################################################
function get_pwd()
{
    grep "^$1[[:space:]]*=" "$2" | cut -d= -f2 | awk '$1=$1'
}
###############################################################
# 功能说明: 公共方法-->结束指定进程
# 入    参: $1 进程的进程路径，支持正则通配
##############################################################
function kill_process()
{
    while read -r process_pid;do
        [[ -n "${process_pid}" ]] && kill -9 "${process_pid}"
    done <<< "$(pgrep -U 51001 -f "$1")"
}
###############################################################
# 功能说明: 公共方法-->命令重试执行
# 入    参: $1 cmd
#          $2 重试次数
#          $3 重试间隔
##############################################################
function cmd_retry()
{
    if [[ $# -ne 3 ]]; then
        return 2
    fi
    cmd=$1
    retry_count=$2
    retry_interval=$3
    for i in $(seq "${retry_count}")
    do
        if ${cmd}; then
            return 0
        fi
        if [ "$i" -ne "${retry_count}" ]; then
            sleep "${retry_interval}"
        fi
    done
    return 1
}
######################################################################
# 功能说明  : 获取当前节点数据库角色
######################################################################
function get_db_role()
{
    sync_monitor="/opt/huawei/dj/tools/gaussdb/sync_monitor.sh"
    if [[ -f ${sync_monitor} ]]; then
        bash ${sync_monitor} get_status | grep "DB last online role" | awk -F':' '{print $NF}' | awk '$1=$1'
    else
        echo "NotExist"
    fi
}
######################################################################
# 功能说明  : 获取当前数据库主节点IP
######################################################################
function get_primary_db_node()
{
    host_ip=$(get_info manage_ip)
    if [[ "$(get_db_role)" == "Primary" ]];then
        echo "${host_ip}"
        return 0
    fi
    sync_monitor="/opt/huawei/dj/tools/gaussdb/sync_monitor.sh"
    IFS="," read -r -a node_ip_list <<< "$(get_info gaussdb_nodes)"
    for node_ip in "${node_ip_list[@]}";do
        if [[ "${node_ip}" == "${host_ip}" ]];then
            continue
        fi
        role=$(cmd_manager --node_ip "${node_ip}" --cmd "${sync_monitor}" --parameters "get_status" \
        | grep "DB last online role" | awk -F':' '{print $NF}' | awk '$1=$1')
        if [[ "${role}" == "Primary" ]];then
            echo "${node_ip}"
            return 0
        fi
    done
    return 1
}
######################################################################
# 功能说明  : 生成密码，可选入参为 password_length 密码长度，默认 9； 可选入参 special_count 特殊字符个数，默认 1
######################################################################
function mkpassword(){
    if [[ $# -eq 0 ]];then
        export PASSWORD_LENGTH=9
        export SPECIAL_COUNT=1
    elif [[ $# -eq 1 ]];then
        export PASSWORD_LENGTH="$1"
        export SPECIAL_COUNT=1
    elif [[ $# -eq 2 ]];then
        export PASSWORD_LENGTH="$1"
        export SPECIAL_COUNT="$2"
    fi
    (
        echo "import os"
        echo "import string"
        echo "import secrets"
        echo "try:"
        echo "    length = int(os.getenv('PASSWORD_LENGTH'))"
        echo "    special_count = int(os.getenv('SPECIAL_COUNT', 1))"
        echo "    if special_count > length:"
        echo "        raise Exception(f'special_count {special_count} should not bigger than length {length}')"
        echo "    special_chars = '~!@#%^&*()_+=-'"
        echo "    alphabet = string.ascii_letters + string.digits + special_chars"
        echo "    while True:"
        echo "        password = ''.join(secrets.choice(alphabet) for i in range(length))"
        echo "        current_special_count = 0"
        echo "        for char in password:"
        echo "            if char in special_chars:"
        echo "                current_special_count += 1"
        echo "        if current_special_count == special_count:"
        echo "            break"
        echo "    print(password)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 加密数据
######################################################################
function encrypt_pwd()
{
    export CSBS_PYTHON_DATA="$*"
    (
        echo "import os"
        echo "from kmc import kmc"
        echo "try:"
        echo "    decrypt_data = os.getenv('CSBS_PYTHON_DATA')"
        echo "    print(kmc.API().encrypt(0, decrypt_data))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 解密密文
######################################################################
function decrypt_pwd()
{
    export CSBS_PYTHON_DATA="$*"
    (
        echo "import os"
        echo "from kmc import kmc"
        echo "try:"
        echo "    encrypt_data = os.getenv('CSBS_PYTHON_DATA')"
        echo "    print(kmc.API().decrypt(0, encrypt_data))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : hash 密文, 入参为 password 返回结果为 "{algorithm}@{iterations}@{salt}@{b64_hash}"
######################################################################
function hash_password()
{
    export CSBS_PYTHON_DATA_PASSWORD="$1"
    (
        echo "import os"
        echo "from basesdk.security_harden import hash_password"
        echo "try:"
        echo "    password = os.getenv('CSBS_PYTHON_DATA_PASSWORD')"
        echo "    result = hash_password(password)"
        echo "    print(result)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 检查密码hash 后是否符合参考值一致, 入参为 password, ref_password_hash 返回结果为 True or False
######################################################################
function verify_password()
{
    export CSBS_PYTHON_DATA_PASSWORD="$1"
    export CSBS_PYTHON_DATA_REF_PASSWORD_HASH="$2"
    (
        echo "import os"
        echo "from basesdk.security_harden import verify_password"
        echo "try:"
        echo "    password = os.getenv('CSBS_PYTHON_DATA_PASSWORD')"
        echo "    ref_password_hash = os.getenv('CSBS_PYTHON_DATA_REF_PASSWORD_HASH')"
        echo "    result = verify_password(password, ref_password_hash)"
        echo "    print(result)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 获取cms_info
######################################################################
function get_cms_info()
{
    export CSBS_PYTHON_DATA="$*"
    (
        echo "import os"
        echo "from basesdk import cms_info"
        echo "try:"
        echo "    key = os.getenv('CSBS_PYTHON_DATA')"
        echo "    print(cms_info.get_cms_info(key))"
        echo "except Exception:"
        echo "    print('')"
    ) | csbs_python
}
######################################################################
# 功能说明  : 更新cms_info
######################################################################
function put_cms_info()
{
    export CSBS_PYTHON_DATA1="$1"
    export CSBS_PYTHON_DATA2="$2"
    (
        echo "import os"
        echo "from basesdk import cms_info"
        echo "try:"
        echo "    key = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    value = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    exit(cms_info.put_cms_info(key, value))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 获取字典数据
######################################################################
function get_dict_info()
{
    export CSBS_PYTHON_DATA1="$1"
    export CSBS_PYTHON_DATA2="$2"
    (
        echo "import ast"
        echo "import os"
        echo "import json"
        echo "try:"
        echo "    convert_data = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    key_list = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    try:"
        echo "        convert_data = ast.literal_eval(convert_data)"
        echo "    except Exception:"
        echo "        convert_data = json.loads(convert_data)"
        echo "    for key in key_list.split('.'):"
        echo "        convert_data = convert_data[key]"
        echo "    print(convert_data)"
        echo "except Exception:"
        echo "    print('')"
    ) | csbs_python
}
######################################################################
# 功能说明  : 获取转义后安全的密码
######################################################################
function get_safe_password()
{
    export CSBS_PYTHON_DATA="$1"
    (
        echo "from basesdk.security_harden import get_safe_password"
        echo "import os"
        echo "try:"
        echo "    value = os.getenv('CSBS_PYTHON_DATA')"
        echo "    value = get_safe_password(value)"
        echo "    print(value)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 发送告警
######################################################################
function send_alarm()
{
    export CSBS_PYTHON_DATA1="$1"
    export CSBS_PYTHON_DATA2="$2"
    (
        echo "import os"
        echo "import json"
        echo "from basesdk import alarm_manager"
        echo "try:"
        echo "    alarm_id = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    alarm_data = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    parameters = json.loads(alarm_data) if alarm_data else ''"
        echo "    exit(alarm_manager.send_alarm(alarm_id, parameters))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}
######################################################################
# 功能说明  : 清除告警
######################################################################
function clear_alarm()
{
    export CSBS_PYTHON_DATA1="$1"
    export CSBS_PYTHON_DATA2="$2"
    (
        echo "import os"
        echo "import json"
        echo "from basesdk import alarm_manager"
        echo "try:"
        echo "    alarm_id = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    alarm_data = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    parameters = json.loads(alarm_data) if alarm_data else ''"
        echo "    exit(alarm_manager.clear_alarm(alarm_id, parameters))"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}

######################################################################
# 功能说明  : 校验 tar 包加压后大小和文件个数，防止压缩文件炸弹
#            绝对路径场景下压缩包中文件路径是否在白名单中
######################################################################
function check_tar_file_before_unzip()
{
    file_path="$1";shift
    uncompressed_size=0
    while read -r file_size;do
        uncompressed_size=$((uncompressed_size+file_size))
    done <<< "$(tar -tvf "${file_path}" | awk '{print $3}')"
    if [[ ${uncompressed_size} -gt MAX_UNCOMPRESSED_SIZE ]];then
        return 1
    fi
    uncompressed_file_count=$(tar -tf "${file_path}" | wc -l)
    if [[ ${uncompressed_file_count} -gt MAX_UNCOMPRESSED_FILE_COUNT ]];then
        return 2
    fi
    if (tar -P -tf "${file_path}" | xargs echo -n | grep -E "\.\./|\*|\~|\?" > /dev/null 2>&1);then
        return 3
    fi
    if [[ "$#" -eq 0 ]];then
        return 0
    fi
    path_trust_list="$*"
    if (tar -P -tf "${file_path}" | grep -Ev "${path_trust_list// /\|}" > /dev/null 2>&1);then
        return 4
    fi
}
