#!/bin/bash
set +x
CURRENT_PATH=$(dirname "$(readlink -f "$0")")
LINE_ROOT=$(dirname "${CURRENT_PATH}")
source "${LINE_ROOT}/bin/utils.sh"
source "${LINE_ROOT}/services/gaussdb/utils.sh"
export LOG_TAG="csbs_install"

ARB_SOURCE_CA_PATH="/home/djmanager/ca.crt"
ARB_DEST_CA_PATH="/home/openstack/ca.crt"
EXPORT_DATABASE_PWD=1
EXPORT_DATABASE_TRUSTSTORE_PWD=6
EXPORT_DATABASE_KEYSTORE_PWD=7


######################################################################
#   FUNCTION   : gaussdb_backup
#   DESCRIPTION: 在Karbor1上执行gaussdb备份，数据备份路径为/home/djmanager/gaussdb_data
######################################################################
function gaussdb_backup()
{
  # karbor1 进行数据备份
  log_info "begin gaussdb backup action."
  prepare_data_sync_tool
  CHECK_RESULT $? "failed to prepare tool."
  privkey_file="/opt/huawei/dj/DJSecurity/privkey/privkey.conf"
  privkey=$(awk '{print $2}' ${privkey_file})
  generate_ssl_cert source "$privkey"
  CHECK_RESULT $? "failed to generate ssl files."
  GAUSSPASSWORD=$(get_info gaussdb_admin_pwd)
  generate_database_password ${EXPORT_DATABASE_PWD} "${GAUSSPASSWORD}"
  generate_database_password ${EXPORT_DATABASE_TRUSTSTORE_PWD} "${privkey}"
  generate_database_password ${EXPORT_DATABASE_KEYSTORE_PWD} "${privkey}"
  sed -i "s/.*flow_type.*/\"flow_type\":1,/" "${LINE_ROOT}/DataSync/config/cfg.ini"
  cd "${LINE_ROOT}/DataSync" || exit 1
  java -jar DSS.jar -p config/cfg.ini; result=$?
  return $result
}

function start_upgrade()
{
    log_info "begin upgrade backup service."
    # 为了解决docker版本升级到去docker版本root帐户锁定问题，升级前增加指定脚本提权
    if [[ -f "/opt/huawei/root_tools/base/exec_docker_action.sh" ]];then
        echo "openstack ALL = (root) NOPASSWD:/opt/huawei/root_tools/base/exec_docker_action.sh" >> ${LINE_ROOT}/services/base/conf/sudo_base_conf
    fi
    cps-monitorControl -S STOP
    # karbor3不执行数据库备份
    node_index=$(get_info node_index)
    if [[ ! -d "${LINE_ROOT}/gaussdb_backup/dataMigration" ]] && [[ $node_index != 3 ]]; then
      # 添加重试
      cmd_retry gaussdb_backup 120 10
      CHECK_RESULT $? "Failed to backup gaussdb."
    fi
    DJserviceControl stop
    if [[ ! -f "${LINE_ROOT}/old_version_data.tar.gz" ]];then
        tar -P -czf "${LINE_ROOT}/old_version_data.tar.gz" /opt/huawei/dj /opt/huawei/root_tools /etc/sudoers.d  /etc/cron.d
    fi
    if [[ ! -f "${LINE_ROOT}/upgrade_files.tar.gz" ]];then
        run_tar_upgrade_files_cmd="tar -P -czf ${LINE_ROOT}/upgrade_files.tar.gz $(cat "${LINE_ROOT}/cfg/upgrade_files.conf")"
        ${run_tar_upgrade_files_cmd}
    fi
    return 0
}

function stop_upgrade()
{
    docker ps -a | grep -v "CONTAINER ID" | awk '{print $1}' | xargs docker rm -f
    DJserviceControl restart
    while read -r upgrade_file;do
        [[ -n "${upgrade_file}" ]] && rm -rf "${upgrade_file}"
    done < "${LINE_ROOT}/cfg/upgrade_files.conf"

    # 同步仲裁证书
    if [[ -f "${ARB_SOURCE_CA_PATH}" ]];then
        log_info "Begin to sync arb cert."
        cp -f ${ARB_SOURCE_CA_PATH} ${ARB_DEST_CA_PATH}
        chown -h openstack:openstack "${ARB_DEST_CA_PATH}"
        sync_karbor_certs --arb ${ARB_DEST_CA_PATH}
        CHECK_RESULT $? "Failed to sync arb ca cert."
    fi
    # 为了解决docker版本升级到去docker版本root帐户锁定问题，升级后移除docker相关脚本
    if [[ -f "/opt/huawei/root_tools/base/exec_docker_action.sh" ]];then
        sed -i "/exec_docker_action/d" /etc/sudoers.d/sudo_base_conf
        rm -f /opt/huawei/root_tools/base/exec_docker_action.sh
    fi
    log_info "end upgrade backup service."
}

function start_rollback()
{
    log_info "begin rollback backup service."
    if [[ -f /opt/huawei/dj/tools/base/DJserviceControl.sh ]]; then
        DJserviceControl stop
        CHECK_RESULT $? "Stop DJserviceControl failed before rollback."
    fi
}

function stop_rollback()
{
    rm -rf /opt/huawei/dj /opt/huawei/root_tools
    old_version_data_file="${LINE_ROOT}/old_version_data.tar.gz"
    upgrade_files="${LINE_ROOT}/upgrade_files.tar.gz"
    check_tar_file_before_unzip "${old_version_data_file}"
    CHECK_RESULT $? "Check old_version_data.tar.gz failed."
    check_tar_file_before_unzip "${upgrade_files}"
    CHECK_RESULT $? "Check upgrade_files.tar.gz failed."
    tar -P -xzf "${old_version_data_file}" >/dev/null 2>&1
    tar -P -xzf "${upgrade_files}" >/dev/null 2>&1
    amend_conf
    DJserviceControl start
    CHECK_RESULT $? "Start DJserviceControl failed after rollback."
    log_info "end rollback backup service."
}

function amend_conf() {
    local db_sync_conf_path=/opt/huawei/dj/etc/gaussdb/db_sync.conf
    if [ -f "${db_sync_conf_path}" ] && ! (grep -q "^allow_sync_progress" "${db_sync_conf_path}"); then
      /usr/bin/cp -f "${LINE_ROOT}"/services/gaussdb/conf/db_sync.conf  "${db_sync_conf_path}"
    fi
    local rabbitmq_enabled_plugins_conf_path=/opt/huawei/dj/bin/rabbitmq/etc/rabbitmq/enabled_plugins
    echo "[]." > "${rabbitmq_enabled_plugins_conf_path}"
    set_modify_permission "${rabbitmq_enabled_plugins_conf_path}"
    rm -rf /opt/huawei/dj/bin/rabbitmq/rabbit@${HOSTNAME}
}

if [[ "$1" == "begin_upgrade" ]];then
    start_upgrade
elif [[ "$1" == "end_upgrade" ]];then
    stop_upgrade
elif [[ "$1" == "begin_rollback" ]];then
    start_rollback
elif [[ "$1" == "end_rollback" ]];then
    stop_rollback
fi