# -*- encoding:utf-8 -*-
import os
import shutil
import stat
import sys

import six
from six.moves import configparser

from basesdk import security_harden
from basesdk import utils
from kmc import kmc

log = utils.get_logger("cli-client")
SYS_CIPHER_TYPE = {
    "config_file": '/opt/huawei/dj/cfg/sys.ini',
    "config_sections": [{
        "section_name": "SYSTEM",
        "config_items": [{"name": "cipher_type"}]
    }]
}

SYS_INI = {
    "config_file": '/opt/huawei/dj/cfg/sys.ini',
    "config_sections": [{
        "section_name": "SYSTEM",
        "config_items": [{"name": "svc_account_password"},
                         {"name": "default_op_service_pwd"},
                         {"name": "gaussdb_admin_pwd"}]
    }]
}


class ConfManager(object):
    def __init__(self, o_cipher_type, new_cipher_type):
        self.o_cipher_type = o_cipher_type
        self.new_cipher_type = new_cipher_type
        self.kmc_api = kmc.API()

    def _backup(self, config_file):
        """Backup files
        :param config_file:
        :return:
        """
        file_bak = '{}{}'.format(config_file, '_backup')
        self._delete_file(file_bak)
        shutil.copy2(config_file, file_bak)

    def _revert_change(self, config_file):
        """Revert change when failed"""
        file_bak = '{}{}'.format(config_file, '_backup')
        os.remove(config_file)
        shutil.copy2(file_bak, config_file)
        self._delete_file(file_bak)

    def _delete_backups(self, config_file):
        """Delete backup files when success
        :param config_file:
        :return:
        """
        file_bak = '{}{}'.format(config_file, '_backup')
        self._delete_file(file_bak)

    @staticmethod
    def _delete_file(file):
        """Delete file
        :param file:
        :return:
        """
        if os.path.exists(file):
            os.remove(file)

    def do_changes(self, config_file, config_sections, force_add=False):
        """Do changes
        :param config_file:
        :param config_sections:
        :param force_add: if set true, then the section will to add to conf
        even it's not exist.
        :return:
        """
        try:
            self._backup(config_file)
            self._do_changes(config_file, config_sections, force_add=force_add)
            self._delete_backups(config_file)
        except Exception as e:
            self._revert_change(config_file)
            raise e

    @staticmethod
    def _do_changes(conf_file, config_sections, force_add=False):
        cf = configparser.ConfigParser()
        cf.read(conf_file)
        for config_section in config_sections:
            config_items = config_section['config_items']
            for config_item in config_items:
                if force_add or cf.has_option(config_section['section_name'], config_item['name']):
                    cf.set(config_section['section_name'], config_item['name'], six.text_type(config_item['value']))
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(conf_file, flags, modes), 'w') as file:
            cf.write(file)

    def update_conf_cipher(self):
        for config_context in [SYS_INI]:
            config_file = config_context['config_file']
            cf = configparser.ConfigParser()
            cf.read(config_file)
            config_sections = config_context['config_sections']
            for section in config_sections:
                items = section['config_items']
                for item in items:
                    value = cf.get(section['section_name'], item['name'])
                    dep_value = self.kmc_api.decrypt(0, value, cipher_type=self.o_cipher_type)
                    item['value'] = self.kmc_api.encrypt(0, dep_value, cipher_type=self.new_cipher_type)
            self.do_changes(config_file, config_sections)

    def update_private_conf(self):
        privkey_conf = '/opt/huawei/dj/DJSecurity/privkey/privkey.conf'
        with open(privkey_conf, 'r') as f:
            txt = f.read()
        dep_value = self.kmc_api.decrypt(0, txt.split()[1], cipher_type=self.o_cipher_type)
        en_value = self.kmc_api.encrypt(0, dep_value, cipher_type=self.new_cipher_type)
        new_txt = ' '.join(['encrypt_password', en_value])

        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(privkey_conf, flags, modes), 'w') as fd:
            fd.write(new_txt)

    def update_cms_conf(self):
        cms_conf = '/opt/huawei/dj/etc/cms/cms.conf'
        cms_configuration = configparser.RawConfigParser()
        cms_configuration.read(cms_conf)
        database_connection = cms_configuration.get("database", 'connection')
        if not database_connection or len(database_connection.split('@')) != 2:
            return
        passwd = database_connection.split('@')[0].split(':')[-1]
        dep_value = self.kmc_api.decrypt(0, passwd, cipher_type=self.o_cipher_type)
        en_value = self.kmc_api.encrypt(0, dep_value, cipher_type=self.new_cipher_type)
        database_connection.replace(passwd, en_value)
        cms_configuration.set("database", 'connection', database_connection)
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(cms_conf, flags, modes), 'w') as fd:
            cms_configuration.write(fd)

    def update_sys_cipher(self):
        config_file = SYS_CIPHER_TYPE['config_file']
        config_sections = SYS_CIPHER_TYPE['config_sections']
        items = config_sections[0]['config_items']
        items[0]['value'] = self.new_cipher_type
        self.do_changes(config_file, config_sections, force_add=True)


if __name__ == "__main__":
    try:
        security_harden.switch_to_openstack_user()
        if len(sys.argv) < 3:
            raise Exception('Para invalid, at least 3 params.')
        method = sys.argv[1]
        if method not in ['update_sys_cipher', 'update_encrypt_data']:
            raise Exception('Para method invalid.')
        cipher_type = sys.argv[2]
        support_type = ['generalCipher', 'SMCompatible']
        if cipher_type not in support_type:
            raise Exception('Para cipher_type invalid.')
        support_type.remove(cipher_type)
        server_action = ConfManager(support_type[0], cipher_type)
        if method == 'update_sys_cipher':
            server_action.update_sys_cipher()
        else:
            server_action.update_conf_cipher()
            server_action.update_private_conf()
            server_action.update_cms_conf()
    except Exception as err:
        log.error(f"update_conf_cipher failed {err}")
        sys.exit(1)
