import base64
import json
import os.path
import time
import requests

from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common import constant
from plugins.eBackup.common import model
from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.model import SshInfo, CmdInfo, SshCmdInfo
from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.common.util import Utils, SshTool
from plugins.eBackup.scripts.common.ebackup_util import CommonTools
from plugins.eBackup.scripts.common.ebackup_util import CustomParametersUtil
from plugins.eBackup.scripts.common.ebackup_util import check_compress_file
from plugins.eBackup.scripts.upgrade.util import get_ebackup_driver_nodes

ADMIN_NODE = "0"
PRIMARY_NODE = "2"
STANDBY_NODE = "3"


class EBackupUtil(object):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = constant.EBACKUP_SERVICE_NAME
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ssh_client = Ssh()
        self.rest_api = EbackupRest()
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.manageone = ManageOneUtil2()
        self.custom_pararms = CustomParametersUtil(project_id, pod_id)

    @staticmethod
    def netmask_to_bit_length(netmask):
        if "." in netmask:
            return str(sum([bin(int(i)).count('1') for i in netmask.split('.')]))
        else:
            return netmask

    def get_internal_float_ip(self):
        float_ip = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, 'datamover_internal_float_ip')
        if not float_ip:
            raise Exception("Failed to obtain the datamover_internal_float_ip.")
        return float_ip

    def get_management_float_ip(self):
        if self.param_tool.is_expand_ebackup_server():
            datamover_ip_list = self._get_param_by_param_key('datamover_externalom_iplist').split(',')
            if len(datamover_ip_list) == 1:
                return datamover_ip_list[0]
            return self._get_param_by_param_key('datamover_management_float_ip')
        elif self.is_ex_az_reuse_ebackup():
            datamover_ip_list = self._get_param_by_param_key('existed_datamover_externalom_iplist').split(',')
            if len(datamover_ip_list) == 1:
                return datamover_ip_list[0]
            return self._get_param_by_param_key('existed_datamover_mg_float_ip')
        else:
            datamover_server_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "datamover_server_ip")
            if not datamover_server_ip:
                datamover_server_ip = self._get_param_by_param_key('datamover_externalom_iplist').split(',')[0]
            return self.get_management_float_ip_by_server_ip(datamover_server_ip)

    def get_externalom_iplist(self):
        return self._get_param_by_param_key('datamover_externalom_iplist').split(',')

    def get_node_ip_netmask(self, ips, user, password):
        logger.info(f"Start to get netmask on node {ips}")
        # 只查询第一个的netmask
        for node_ip in ips:
            cmd = '''netmask=$(ip a | grep -w %s | grep inet | awk '{print $2}' | awk -F'/' '{print $2}');
            netcard_name=$(ip a | grep -w %s | grep inet | awk '{print $NF}' );
            gateway=$(route -n | awk '{if ($1=="0.0.0.0"&&$8=="'${netcard_name}'")print $2}');
            echo "ebackup_netmask: ${gateway}/${netmask}"''' % (node_ip, node_ip)
            ssh_client = self.ssh_client.ssh_create_client(node_ip, user, password)
            result = self.ssh_client.ssh_exec_command_return_list(ssh_client, cmd)
            for lines in result:
                if "ebackup_netmask: " in lines and "echo" not in lines:
                    logger.info(f"Get netmask {lines} on node {node_ip}")
                    return lines.split("ebackup_netmask:")[1].strip()
            return ""

    def get_active_stanby_node(self, datamover_ip):
        base_url = self.get_server_url(datamover_ip)
        portal_account = self.get_portal_account(datamover_ip)
        _, headers = self.login_portal(base_url, portal_account)
        rsp = requests.get(url=f"{base_url}/vbackup_server", headers=headers, verify=False)
        rsp_dict = rsp.json()
        error_code = rsp_dict.get("error", dict()).get("code")
        if error_code is None:
            logger.error("errcode is empty.")
            raise Exception("Get active standby node failed. errcode is empty.")
        if error_code != 0:
            error_des = rsp_dict.get("error").get("description")
            raise Exception(f"Get active standby node failed. error description is {error_des}.")

        node_list = set()
        server_list = rsp_dict.get("data", [])
        for server_info in server_list:
            role = server_info.get("ROLE")
            if str(role).strip() in (ADMIN_NODE, PRIMARY_NODE, STANDBY_NODE):
                ip_addr = server_info.get("IPADDR", "")
                node_list.add(ip_addr)
        logger.info(f"Successful get node list:{list(node_list)}")
        return list(node_list)

    def get_management_float_ip_by_server_ip(self, server_ip):
        cmd_base = "grep -w listen /opt/huawei-data-protection/ebackup/" \
                   "microservice/ebk_lb/bin/nginx/conf/nginx.conf|grep -w 8088"
        cmd_awk = "|awk '{print $2}'|awk -F '[' '{print $2}'|awk -F ']' '{print $1}'" \
            if self.param_tool.is_ipv6_scene() else "|awk '{print $2}'|awk -F ':' '{print $1}'"
        cmd = cmd_base + cmd_awk
        node_user = self.get_node_user(server_ip)
        try:
            ssh_client = self.ssh_client.ssh_create_client(server_ip, node_user.user_name, node_user.user_pwd)
            self.ssh_client.ssh_send_command(ssh_client, 'su', 'Password:', 100)
            self.ssh_client.ssh_send_command(ssh_client, node_user.root_pwd, '#', 100)
            self.ssh_client.ssh_send_command(ssh_client, "TMOUT=0", '#', 100)
        except Exception:
            ssh_client = self.ssh_client.ssh_create_client(server_ip, node_user.root_name, node_user.root_pwd)
            self.ssh_client.ssh_send_command(ssh_client, "TMOUT=0", '#', 100)
        result = self.ssh_client.ssh_exec_command_return_list(ssh_client, cmd)
        self.ssh_client.ssh_close(ssh_client)
        return result[0]

    def _get_param_by_param_key(self, param_key):
        param_value = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, param_key)
        if not param_value:
            logger.error(f"Failed to obtain the {param_key}.")
        return param_value

    def is_ex_az_reuse_ebackup(self):
        return self.param_tool.is_expand_az() and \
            self.param_tool.is_installed_csbs() and \
            not self.param_tool.is_csha_storage()

    def get_server_url(self, management_float_ip):
        if self.param_tool.is_ipv6_scene():
            return f"https://[{management_float_ip}]:8088/rest/dev"
        return f"https://{management_float_ip}:8088/rest/dev"

    def get_default_node_user(self):
        os_user_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "hcp_ssh_user")
        if not os_user_name:
            os_user_name = "hcp"
        root_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "eBackup_os_root_username")
        if not root_name:
            root_name = "root"
        os_user_pwd = self._get_param_by_param_key("hcp_ssh_password")
        root_pwd = self._get_param_by_param_key("eBackup_os_root_password")
        return model.NodeUser(os_user_name, os_user_pwd, root_name, root_pwd)

    def get_node_user(self, vm_ip):
        node_user = self.get_default_node_user()
        region_id = self.param_tool.get_region_id()
        try:
            account_info_list = self.param_tool.get_account_info_from_unify_pwd(
                region_id, [(node_user.user_name, 1), (node_user.root_name, 1)], vm_ip=vm_ip)
            for account_info in account_info_list:
                if account_info.get("ip") != vm_ip:
                    continue
                if account_info.get('accountName') == node_user.user_name:
                    node_user.user_pwd = account_info.get('newPasswd')
                if account_info.get('accountName') == node_user.root_name:
                    node_user.root_pwd = account_info.get('newPasswd')
        except Exception as err:
            logger.warning(f"Failed to obtain account info from unify pwd platform, err_mag:{str(err)}.")
        return node_user

    def get_portal_account(self, management_float_ip=None, is_upgrade=False):
        account_name = "admin"
        account_pwd_key = "eBackup_admin_pwd" if is_upgrade else "eBackup_weblogin_password"
        default_pwd = None
        if self.param_tool.is_expand_ebackup_server():
            account_pwd = self._get_param_by_param_key(account_pwd_key)
            if not is_upgrade:
                default_pwd = self._get_param_by_param_key("eBackup_weblogin_default_password")
        else:
            if not management_float_ip:
                management_float_ip = self.get_management_float_ip()
            region_id = self.param_tool.get_region_id()
            account_info_list = self.param_tool.get_account_info_from_unify_pwd(
                region_id, [(account_name, 6)], vm_ip=management_float_ip)
            account_pwd = None
            for account_info in account_info_list:
                if account_info.get("ip") == management_float_ip:
                    account_pwd = account_info.get('newPasswd')
                    break
        if not account_pwd:
            raise Exception(f"Failed to obtain the {account_name}'s passwd.")
        account = model.Account(account_name, account_pwd)
        if default_pwd:
            account.default_pwd = default_pwd
        return account

    def login_portal(self, server_url, portal_account):
        logger.info(f"Start to login ebackup portal, server url:{server_url}.")
        login_url = f"{server_url}/login"
        headers = {'Accept': 'application/json'}
        data = {"scope": 0,
                "username": portal_account.account_name,
                "password": portal_account.account_pwd}
        rsp = self.rest_api.post(login_url, headers, json.dumps(data))
        if rsp.status_code != 200:
            logger.error("Response status_code is not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("Errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            i_base_token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            device_id = rsp.json()["data"].get("deviceid", 'dev')
        else:
            description = rsp.json()["error"]["description"]
            logger.error(f"Login failed, description:{description}.")
            raise HCCIException(653022, description)
        logger.info("Succeed to login ebackup portal.")
        headers = {
            'Accept': 'application/json;version=2.2;charset=UTF-8',
            'iBaseToken': i_base_token,
            'Cookie': f'language=en;{session_content}; DEVICE_ID=dev; sessionIdleTime=60000; '
                      f'MACHINEROLE=0; CSRF_IBASE_TOKEN={i_base_token}'}
        return device_id, headers

    def logout_portal(self, server_url, headers):
        url = f"{server_url}/sessions"
        try:
            self.rest_api.delete(url, headers)
        except Exception as err:
            logger.warning(str(err))
        logger.info("Succeed to logout portal.")

    def add_new_user(self, management_float_ip, user_name, is_upgrade=False):
        base_url = self.get_server_url(management_float_ip)
        portal_account = self.get_portal_account(management_float_ip, is_upgrade)
        device_id, headers = self.login_portal(base_url, portal_account)

        add_user_url = f"https://{management_float_ip}:8088/rest/{device_id}/user"
        if user_name == "EBKUser":
            default_passwd = self._get_param_by_param_key('EBKUser_default_password')
        elif user_name == "NBIUser":
            default_passwd = self._get_param_by_param_key('eBackup_azconfig_apipwd')
        else:
            raise Exception("Unknown ebackup user name.")
        body = {
            "SCOPE": 4,
            "NAME": user_name,
            "PASSWORD": default_passwd,
            "LEVEL": 2,
            "SESSIONMODE": 0,
            "SESSIONNUM": 5
        }
        rsp = self.rest_api.post(add_user_url, headers, json.dumps(body))
        if rsp.status_code != 200:
            logger.error("Response status_code is not 200.")
            self.logout_portal(base_url, headers)
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("Errcode is empty.")
            self.logout_portal(base_url, headers)
            raise HCCIException(653020)
        if errcode == 0:
            logger.info(f"Succeed to add {user_name}.")
            self.logout_portal(base_url, headers)
        elif rsp.json()["error"].get('code') == 1610664452:
            logger.info(f"{user_name} is already exist.")
            self.logout_portal(base_url, headers)
        else:
            description = rsp.json()["error"]["description"]
            logger.error(f"Add EBKUser failed, description:{description}.")
            self.logout_portal(base_url, headers)
            raise HCCIException(650069, description)

    def set_cloud_service_info_of_ebackup(self, param_dict, az_list_str, host_ips=None, is_upgrade=True):
        index_name = 'eBackup'
        if is_upgrade:
            input_version = param_dict.get('eBackup_Version')
        else:
            input_version = self.get_old_version(host_ips, param_dict)
        cloud_info_list = self.cmdb_util.get_cloud_service_info_v3(param_dict.get('current_region_id'), index_name)
        for cloud_service in cloud_info_list:
            deploy_node = self.cmdb_util.get_deploy_node_info(param_dict.get('current_region_id'), index_name)
            if cloud_service.get("name", "").startswith("eBackup_service1"):
                extend_infos = cloud_service.get("extendInfos", [])
                extend_infos_list = [extend_info for extend_info in extend_infos if extend_info.get("key") != "az_list"]
                extend_infos_list.append({"key": "az_list", "value": az_list_str})
                cloud_service.update({"extendInfos": extend_infos_list})

            cloud_service.update({"deploy_node_infos": deploy_node, "version": input_version})
            service_name = cloud_service.get("name", "default")
            if is_upgrade and service_name.startswith("eBackup_service"):
                self.set_service_name(deploy_node, service_name, param_dict)
            logger.info(f"start set cloud info, info {cloud_service}.")
            self.cmdb_util.set_cloud_service_info_v3(param_dict.get('current_region_id'), cloud_service)

    def set_service_name(self, deploy_nodes, service_name, param_dict):
        upgrade_ip_list = param_dict["eBackup_Datamover_nodes"].replace(" ", "").split("|")
        cmdb_ip_list = []
        for deploy_node in deploy_nodes:
            ip_addresses = deploy_node.get('ipAddresses')
            cmdb_ip_list += [ip_address.get("ip", "") for ip_address in ip_addresses]
        for group_ips in upgrade_ip_list:
            node_ips = group_ips.split(";")
            for node_ip in node_ips:
                logger.info(f"Start to check current node:{node_ip} and cmdb_ip:{cmdb_ip_list}")
                if node_ip in cmdb_ip_list:  # cmdb上存在存储平面ip 因此需要判断是否在升级的管理ip列表中
                    CommonTools(self.project_id, self.pod_id).set_service_name_to_node(node_ips, service_name)
                    return  # 找到该IP后代表该ip所在的集群属于service_name所代表的集群，该集群只需调用一次设置方法

    def get_old_version(self, host_ips, param_dict):
        host_ip = host_ips[0]
        hcp_password = param_dict["eBackup_hcp_pwd"]
        root_password = param_dict["eBackup_root_pwd"]
        cmd = "cat /tmp/upgrade/upgrade_pkg/upgrade/versions.conf.old"
        ssh_client = self.ssh_client.ssh_create_client(host_ip, "hcp", hcp_password)
        self.ssh_client.ssh_send_command(ssh_client, 'su', 'Password:', 100)
        self.ssh_client.ssh_send_command(ssh_client, root_password, '#', 100)
        # The response of the function would be a list and "\n" in the string in the list, so take [0][:-1]
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)[0][:-1]
        self.ssh_client.ssh_close(ssh_client)
        return result

    def config_alarm(self):
        management_float_ip = self.get_management_float_ip()
        base_url = self.get_server_url(management_float_ip)
        portal_account = self.get_portal_account(management_float_ip)
        _, headers = self.login_portal(base_url, portal_account)
        try:
            self.config_alarm_report(base_url, headers)
        finally:
            if base_url != "" or headers != "":
                self.logout_portal(base_url, headers)
        return

    def config_alarm_report(self, base_url, headers):
        logger.info("Begin to config alarm report.")

        url = base_url + '/alarm_settings/default'
        alarm_user, alarm_pwd = self.manageone.getThirdpartyInfor(self.pod_id)[:2]
        if not alarm_user:
            logger.error("Alarm user is empty.")
            raise HCCIException(653039, "alarm user is empty")
        if not alarm_pwd:
            logger.error("Alarm pwd is empty.")
            raise HCCIException(653039, "alarm pwd is empty")

        oc_addrss = self.custom_pararms.get_oc_domain_name()
        logger.info("Alarm url:" + oc_addrss)
        data = '{"ISALARMON":1,"ALARMSERVICETYPE":"1","ALARMSERVERURL":"' + oc_addrss + '","ALARMUSERNAME":"' + \
               alarm_user + '","ALARMPASSWORD":"' + alarm_pwd + '"}'

        description = ''
        num = 0
        while num < 3:
            rsp = self.rest_api.put(url, headers, data)
            if rsp.status_code != 200:
                logger.error("Rest req return not 200.")
                raise HCCIException(653020)
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("Errcode or description is empty.")
                raise HCCIException(653019)
            if errcode != 0:
                logger.error(f"Config alarm report failed,description:{description}, retry config.")
                num += 1
                time.sleep(30)
            else:
                logger.info("Config alarm report succeeded.")
                return
        logger.error("config alarm report failed, description:" + description)
        raise HCCIException(653039, description)


class DMKDriverUtil:
    """eBackup Driver DMK工具类"""
    param_map = {}
    driver_host_list = []
    cps_host_list = []

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ssh_client = Ssh()
        self.rest_api = EbackupRest()
        self.custom_pararms = CustomParametersUtil(project_id, pod_id)
        self._db_param_dict = Utils.init_system_params(project_id, region_id)
        self.cps_admin_pwd = ""
        tmp_fsp_pwd = self.param_util.get_param_value(
            self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        self.fsp_pwd = tmp_fsp_pwd if tmp_fsp_pwd else self._db_param_dict.get('openstack_fsp_pwd')
        tmp_root_pwd = self.param_util.get_param_value(
            self.pod_id, "OpenStack", 'root_password', 'openstack_root_password')
        self.fsp_root_pwd = tmp_root_pwd if tmp_root_pwd else self._db_param_dict.get('openstack_root_pwd')
        self.cps_admin_pwd_unicode = ""
        self.is_deploy = True

    def set_dmk_parm(self, val_content, host_content):
        self.param_map.update(val_content)
        self.driver_host_list = host_content.get("eBackup_host", [])
        self.cps_host_list = host_content.get("cpsnode", [])
        self.cps_admin_pwd_unicode = self.param_map.get('ebk_PasswordEnv', "")
        self.cps_admin_pwd = base64.b64encode(self.cps_admin_pwd_unicode.encode("utf-8")).decode(encoding="utf-8")

    def get_driver_file(self):
        file_path, filename = API.find_file("eBackupDriver_", ".tar.gz")
        if filename == "":
            zippath, zipname = API.find_file(
                "OceanStor BCManager", "_eBackup_DMK.zip" if self.is_deploy else "_eBackup_Upgrade_DMK.zip")
            if zipname == "":
                raise HCCIException(653066, "OceanStor BCManager..._DMK.zip")
            file_path = os.path.join(zippath, zipname)
            check_compress_file([file_path])
            API.extract_file(file_path, "/home/pkg/eBackupbin")
            file_path, filename = API.find_file("eBackupDriver_", ".tar.gz")
        logger.info("find eBackupDrive succ.")
        return file_path, filename

    def upload_package_to_hosts(self):
        path, name = self.get_driver_file()
        file_path = os.path.join(path, name)
        for fsp_ip in self.driver_host_list:
            logger.info(f"start to upload driver package to {fsp_ip}.")

            ssh_info = SshInfo(fsp_ip, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
            uncompress_cmd_list = [
                f"rm -rf {constant.FSP_BIN_PATH}",
                f"cd {constant.FSP_HOME_PATH}",
                f"mkdir -p {constant.FSP_BIN_PATH}/bin",
                f"tar xvf \"{name}\" -C {constant.FSP_BIN_PATH}"
            ]
            ssh_client = SshTool.get_ssh_client(ssh_info, True)
            clean_cmd = f"rm -rf {constant.FSP_HOME_PATH}/{name}"
            clean_cmd_info = CmdInfo(ssh_client, cmd_list=[clean_cmd], is_expect=False)
            SshTool.send_ssh_command(clean_cmd_info, 100)

            SshTool.upload_file(ssh_info, file_path, constant.FSP_HOME_PATH)
            cmd_info = CmdInfo(ssh_client, cmd_list=uncompress_cmd_list, is_expect=False)
            SshTool.send_ssh_command(cmd_info, 100)

    def driver_install(self):
        """DMK install.yml 功能迁移"""
        logger.info(f"start to execute install.yml.")

        self.upload_package_to_hosts()
        install_cmd_list = [
            f"chmod +x {constant.FSP_BIN_PATH}/eBackupDriver_{constant.EBACKUP_VERSION}/action/*.py",
            f"export PATH=$PATH:/usr/local/sbin:/usr/sbin",
            f"echo \"{self.cps_admin_pwd}\" | python {constant.FSP_BIN_PATH}/eBackupDriver_"
            f"{constant.EBACKUP_VERSION}/action/actions.py cps_admin install 1>/dev/null",
            "sed -i '/action.py/d' /root/.bash_history"
        ]
        for fsp_ip in self.driver_host_list:
            ssh_info = SshInfo(fsp_ip, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
            ssh_client = SshTool.get_ssh_client(ssh_info, True)
            cmd_info = CmdInfo(ssh_client, cmd_list=install_cmd_list, is_expect=False)
            SshTool.send_ssh_command(cmd_info, 100)
        return True

    def generate_config_param(self, config_node_client):
        param_map = {
            "CpsUsername": "cps_admin",
            "Ipaddr": self.param_map.get("ebk_Ipaddr", ""),
            "Azs": self.param_map.get("ebk_Azs", ""),
            "LoginUser": "NBIUser",
            "Port": "8088",
            "CertFilePath": self.param_map.get("ebk_CertFilePath", ""),
            "PhysicalNetworkType": "",
            "FusionStorageIp": "",
            "FusionStorageAgentIp": "",
            "LazyLoading": "",
            "configMode": self.param_map.get("ebk_configMode", ""),
            "Extend": "",
            "Tenant_list": ""
        }
        clean_cmd = CmdInfo(
            config_node_client,
            [f"> {constant.FSP_BIN_PATH}/bin/eBackupServer.ini"])
        SshTool.send_ssh_command(clean_cmd, 100)
        wirte_cmd_list = []
        for key, value in param_map.items():
            logger.info(f"start to set key {key} {value}.")
            if key == "Ipaddr":
                wirte_cmd_list.append(f"echo \"{key}={value}\" >> {constant.FSP_BIN_PATH}/bin/eBackupServer.ini")
            else:
                wirte_cmd_list.append(f"echo \"{key}:{value}\" >> {constant.FSP_BIN_PATH}/bin/eBackupServer.ini")
        write_cmd = CmdInfo(config_node_client, wirte_cmd_list)
        SshTool.send_ssh_command(write_cmd, 100)

    def driver_config(self):
        """DMK config.yml 功能迁移"""
        logger.info(f"start to execute config.yml.")

        if not self.driver_host_list:
            logger.error(f"driver host is empty.")
            return False

        config_node = self.driver_host_list[0]
        file_path = os.path.realpath(os.path.join(os.path.dirname(__file__), "../shell_tool/configeBackupDriver.sh"))
        ssh_info = SshInfo(config_node, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
        SshTool.upload_file(ssh_info, file_path, constant.FSP_HOME_PATH)
        config_node_client = SshTool.get_ssh_client(ssh_info)
        self.generate_config_param(config_node_client)
        nbi_pwd_unicode = self.param_map.get("ebk_PasswordUsr", "")
        nbi_pwd = base64.b64encode(nbi_pwd_unicode.encode("utf-8")).decode(encoding="utf-8")
        start_cmd = [
            f"export PATH=$PATH:/usr/local/sbin:/usr/sbin;python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py start",
            f"echo '{self.cps_admin_pwd}' | python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py update "
            f"cps_admin ebackup_env_type 2 default",
            f"cd \"{constant.FSP_HOME_PATH}\" && chmod +x configeBackupDriver.sh && "
            f"echo \"{self.cps_admin_pwd} {nbi_pwd}\" | sh configeBackupDriver.sh \"private\" "
            f"\"{constant.DRIVER_INSTALL_PATH}\"",
            f"echo '{self.cps_admin_pwd}' | python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py update "
            f"cps_admin ebackup_nfs_data_layout {self.param_map.get('ebk_NfsDataLayout', '')} default",
            f"echo '{self.cps_admin_pwd}' | python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py update "
            f"cps_admin ebackup_uds_data_layout {self.param_map.get('ebk_S3DataLayout', '')} default",
        ]
        start_cmd_info = CmdInfo(config_node_client, start_cmd)
        SshTool.send_ssh_command(start_cmd_info, 100)

        config_cmd = [
            f"export PATH=$PATH:/usr/local/sbin:/usr/sbin;"
            f"echo '{self.cps_admin_pwd}' | python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py check cps_admin",
            f"echo '{self.cps_admin_pwd}' | python {constant.DRIVER_INSTALL_PATH}/ebackupconfig.py commit cps_admin"
        ]
        config_cmd_info = CmdInfo(config_node_client, config_cmd)
        SshTool.send_ssh_command(config_cmd_info, 100)
        return True

    def driver_precheck_driver_node_list(self):
        """DMK precheck_driver_node_list.yml 功能迁移"""
        logger.info(f"start to execute precheck_driver_node_list.yml.")

        if not self.driver_host_list:
            logger.error(f"driver host is empty.")
            return False
        host_info = get_ebackup_driver_nodes(
            self.driver_host_list[0], self._db_param_dict, self.fsp_pwd, self.fsp_root_pwd, self.cps_admin_pwd_unicode)
        self.driver_host_list = host_info.get("eBackupDriver")
        self.cps_host_list = host_info.get("cpsnode")
        SshTool.verify_turnkey_agent(self.cps_host_list + self.driver_host_list)
        return True

    def driver_rollback(self):
        """DMK rollback.yml 功能迁移"""
        self.upload_package_to_hosts()
        openstack_fsp_pwd = self._db_param_dict.get('openstack_fsp_pwd')
        openstack_root_pwd = self._db_param_dict.get('openstack_root_pwd')
        rollback_cmds = 'export PATH=$PATH:/usr/local/sbin:/usr/sbin;' \
                        f'echo "{self.cps_admin_pwd}" | ' \
                        f'python {constant.FSP_BIN_PATH}/eBackupDriver_{constant.EBACKUP_VERSION}/action/actions.py ' \
                        f'cps_admin rollback'
        for node_ip in self.driver_host_list:
            rollback_cmds_info = SshCmdInfo(node_ip, f"{rollback_cmds}", "fsp", openstack_fsp_pwd,
                                            openstack_root_pwd, "", "")
            SshTool.ssh_cmds(rollback_cmds_info)

    def driver_upgrade(self):
        """DMK upgrade.yml 功能迁移"""
        self.upload_package_to_hosts()
        openstack_fsp_pwd = self._db_param_dict.get('openstack_fsp_pwd')
        openstack_root_pwd = self._db_param_dict.get('openstack_root_pwd')

        upgrade_cmds = 'export PATH=$PATH:/usr/local/sbin:/usr/sbin;' \
                       f'echo "{self.cps_admin_pwd}" | ' \
                       f'python {constant.FSP_BIN_PATH}/eBackupDriver_{constant.EBACKUP_VERSION}/action/actions.py ' \
                       f'cps_admin upgrade'
        for node_ip in self.driver_host_list:
            upgrade_cmds_info = SshCmdInfo(node_ip, f"{upgrade_cmds}", "fsp", openstack_fsp_pwd,
                                           openstack_root_pwd, "", "")
            SshTool.ssh_cmds(upgrade_cmds_info)

    def clean_history(self):
        clean_cmd = [
            "sed -i '/ebackupconfig.py/d' /root/.bash_history",
            "sed -i '/configeBackupDriver.py/d' /root/.bash_history"
        ]
        for node_ip in self.driver_host_list + self.cps_host_list:
            ssh_info = SshInfo(node_ip, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
            node_client = SshTool.get_ssh_client(ssh_info)
            clean_cmd_info = CmdInfo(node_client, clean_cmd)
            SshTool.send_ssh_command(clean_cmd_info, 100)

    def driver_sync_config(self):
        """DMK sync_config.yml 功能迁移"""
        logger.info(f"start to execute sync_config.yml.")

        if not self.driver_host_list:
            logger.error(f"driver host is empty.")
            return False
        sync_node = self.driver_host_list[0]
        ssh_info = SshInfo(sync_node, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
        sync_client = SshTool.get_ssh_client(ssh_info)
        sync_cmd = [
            "export PATH=$PATH:/usr/local/sbin:/usr/sbin",
            f"echo '{self.cps_admin_pwd}' | python {constant.FSP_BIN_PATH}/"
            f"eBackupDriver_{constant.EBACKUP_VERSION}/action/actions.py cps_admin sync_config"
        ]
        sync_cmd_info = CmdInfo(sync_client, sync_cmd)
        SshTool.send_ssh_command(sync_cmd_info, 100)
        return True

    def driver_add_key(self):
        """DMK addkey.yml 功能迁移"""
        logger.info(f"start to execute addkey.yml.")
        for cps_ip in self.cps_host_list:
            ssh_info = SshInfo(cps_ip, constant.FSP_USER, self.fsp_pwd, self.fsp_root_pwd)
            cps_client = SshTool.get_ssh_client(ssh_info)
            add_key_cmd = [
                f"sed -i '/ebackup_login_password_v2/d' \"{constant.KEYLIST_FILE}\"",
                f"sed -i '/ebackup_storage_username_v2/d' \"{constant.KEYLIST_FILE}\"",
                f"sed -i '/ebackup_storage_password_v2/d' \"{constant.KEYLIST_FILE}\"",
                r"sed -i '/\[default\]/aebackup_login_password_v2\nebackup_storage_username_v2\n"
                f"ebackup_storage_password_v2' \"{constant.KEYLIST_FILE}\""
            ]
            addkey_cmd_info = CmdInfo(cps_client, add_key_cmd)
            SshTool.send_ssh_command(addkey_cmd_info, 100)

    def driver_patch_exec(self):
        """
        DMK patch.yml 功能迁移
        当前版本无补丁需求  暂不实现
        """
        pass

    def driver_patch_rollback_exec(self):
        """
        DMK patch rollback.yml 功能迁移
        当前版本无补丁需求  暂不实现
        """
        pass

    def driver_install_exec(self):
        """DMK Driver 安装入口"""
        # hcci的参数DriverNode_one2all固定为1 无需再次判断
        self.driver_precheck_driver_node_list()
        install_mode = self.param_map.get("ebk_installmode")
        # 0:全量安装 1: 仅在新节点安装Driver 2：仅安装以及同步配置
        if install_mode == "0":
            self.driver_add_key()
        self.driver_install()
        if install_mode == "2":
            self.driver_sync_config()
        elif install_mode == "0":
            self.driver_config()
        return True

    def driver_upgrade_exec(self):
        """DMK Driver 升级入口"""
        self.is_deploy = False
        self.driver_add_key()
        self.driver_upgrade()
        self.driver_sync_config()
        logger.info("Upgrade successful")
        return True

    def driver_rollback_exec(self):
        """DMK Driver 回退入口"""
        self.is_deploy = False
        self.driver_precheck_driver_node_list()
        self.driver_rollback()
        logger.info("Rollback successful")
        return True

    def execute_action(self, action):
        """DMK 入口函数"""
        try:
            SshTool.verify_turnkey_agent(self.cps_host_list + self.driver_host_list)
            if action == "upgrade":
                return self.driver_upgrade_exec()
            elif action == "patch":
                return self.driver_patch_exec()
            elif action == "patch_rollback":
                return self.driver_patch_rollback_exec()
            elif action == "install":
                return self.driver_install_exec()
            elif action == "rollback":
                return self.driver_rollback_exec()
            return False
        finally:
            self.clean_history()
