import json
import time
import utils.common.log as logger
from utils.business.iam_util import IamApi
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean

from plugins.eBackup.common import constant
from plugins.eBackup.common.model import Account
from plugins.eBackup.common.param_tool import ParamTool


class IamUtil(object):
    class IAMAccount(Account):
        def __init__(self, account_name, account_pwd, domain_name, is_default=True):
            super().__init__(account_name, account_pwd)
            self.domain_name = domain_name
            self.is_default = is_default

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_util = ParamUtil()
        self.service_name = constant.EBACKUP_SERVICE_NAME
        self.param_tool = ParamTool(self.project_id, self.pod_id)

    def _get_op_service_default_account(self):
        account_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_karbor_admin')
        if not account_name:
            raise Exception("Failed to obtain the op_service account name.")
        account_pwd = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_karbor_admin_pwd')
        if not account_pwd:
            raise Exception("Failed to obtain the op_service account pwd.")
        return self.IAMAccount(account_name, account_pwd, domain_name="op_service")

    def _get_op_svc_default_account(self):
        account_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_op_svc_csbs_account')
        if not account_name:
            raise Exception("Failed to obtain the op_svc account name.")
        account_pwd = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_op_svc_csbs_pwd')
        if not account_pwd:
            raise Exception("Failed to obtain the op_svc account pwd.")
        return self.IAMAccount(account_name, account_pwd, domain_name="op_svc_csbs")

    def get_iam_account_info(self, account_type="op_service"):
        if account_type not in ("op_service", "op_svc_csbs"):
            raise Exception("Unsupported account_type, the input "
                            f"account_type parameter is {account_type}.")
        if account_type == "op_service":
            account = self._get_op_service_default_account()
        else:
            account = self._get_op_svc_default_account()

        # 首region且部署场景， 直接返回默认账号密码
        if get_project_condition_boolean(self.project_id, "PrimaryRegion & ProjectDeploy"):
            logger.info("The current scene is the primary region deployment scene, "
                        "use the default account information.")
            return account

        if self.can_authed(account):
            logger.info("The default account is used for authentication.")
            return account

        if self.can_authed(account):
            return account
        return self._get_modified_pwd_account(account)

    def _get_modified_pwd_account(self, account):
        iam_pwd_list = []
        for region_id in self.param_tool.get_all_region_code():
            try:
                account_info = self.param_tool.get_account_info_from_unify_pwd(
                    region_id, [(account.account_name, 5)],
                    component_dict={"component": "CSBS-VBS", "sub_component": "CSBS-VBS"})
            except Exception as err:
                logger.warning(f"Failed to obtain the account information of {region_id} "
                               f"from the unify_pwd, err_msg:{str(err)}.")
                continue
            for item in account_info:
                if item.get('accountName') != account.account_name or item.get('newPasswd') == account.account_pwd:
                    continue
                iam_pwd_list.append(item.get('newPasswd'))
        authed_count = 1
        for pwd in set(iam_pwd_list):
            account.account_pwd = pwd
            if self.can_authed(account):
                account.is_default = False
                return account
            # 15分钟内鉴权失败5次，账号会被锁15分钟
            authed_count += 1
            if authed_count >= 5:
                time.sleep(15 * 60 + 10)
                authed_count = 0
        raise Exception(f"Failed to obtain the {account.account_name} pwd from the unify_pwd "
                        f"and the {account.account_name} pwd is not the default password.")

    def get_iam_token(self, account):
        """

        :param account: IAMAccount() object instance
        :return: iam token, type:str
        """
        if not hasattr(account, "domain_name") or account.domain_name not in ("op_service", "op_svc_csbs"):
            raise Exception("The domain_name of the account object is not expected, "
                            f"the domain_name is {getattr(account, 'domain_name', None)}.")
        account_info = {"domain_name": account.domain_name,
                        "name": account.account_name}
        body = {"auth": {"identity": {"methods": ["password"], "password": {
            "user": {"name": account_info.get("name"), "password": account.account_pwd,
                     "domain": {"name": account_info.get("domain_name")}}}},
                         "scope": {"domain": {"name": account_info.get("domain_name")}}}}
        return IamApi.get_iam_token(self.pod_id, json.dumps(body))

    def can_authed(self, account):
        """

        :param account: IAMAccount() object instance
        :return: bool
        """
        try:
            if self.get_iam_token(account):
                logger.info("Success to use the current account for authentication, "
                            f"account name:{account.account_name}")
                return True
        except Exception as err:
            logger.warning(f"Failed to use the current account for authentication, "
                           f"account name:{account.account_name}, err_msg:{str(err)}")
        return False
