# -*- coding:utf-8 -*-
import datetime
import ipaddress
import json
import os
import re
import stat
import time
import socket

import requests
from IPy import IP
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.ssh_util2 import Ssh
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil

from plugins.eBackup.common.model import AccountQualityInfo, SshInfo
from plugins.eBackup.common.constant import IPV6_NUMBER


CMD_OUTPUT_LENGTH = 2
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2

HCP_ACCOUNT_DESC = '{"zh-cn":"使用该帐户登录eBackup' \
                   '服务器进行运维操作，查看节点配置，查看日志，' \
                   '巡检和信息收集等。","en-us":"Used to log in ' \
                   'to an eBackup server to perform O&M operations, ' \
                   'view node configurations and logs, perform inspection, ' \
                   'and collect information."} '
ROOT_ACCOUNT_DESC = '{"zh-cn":"考虑到对系统安全的影响，在登录节点时，' \
                    '不能使用root用户直接登录，需要由hcp' \
                    '用户登录后通过执行命令 su - ' \
                    'root切换为root用户。切换到root' \
                    '帐号后，可对服务进行日常操作和维护，例如执行进程查看、' \
                    '日志查看、密码修改、配置文件修改等操作。",' \
                    '"en-us":"To ensure system security, you cannot log in ' \
                    'to the node as user root. You need to log in to the ' \
                    'node as user hcp and run the su - root command to ' \
                    'switch to user root. After switching to user root, ' \
                    'you can perform routine operations and maintenance on ' \
                    'services, such as viewing processes and logs, changing ' \
                    'passwords and configuration files."} '


def check_string_param(name, max_len=255, min_len=1,
                       expr=r"^[a-zA-Z0-9.\-_]+$", allow_null=False):
    if not name:
        return allow_null
    if len(name) < min_len or len(name) > max_len:
        return False
    pattern = re.compile(expr)
    if not re.match(pattern, name):
        return False
    return True


def check_url_param(endpoint, max_len=255, min_len=1, allow_null=False):
    if not check_string_param(endpoint, max_len, min_len,
                              expr=r"^[a-zA-Z0-9./\-_:=?#%]+$",
                              allow_null=allow_null):
        raise HCCIException(653103, endpoint)
    return True


def check_ip(host_ip):
    if not isinstance(host_ip, (str, int)):
        return False
    try:
        ipaddress.ip_address(host_ip)
        return True
    except Exception:
        return False


def parse_rest_response(result):
    def do_parse_rest_response():
        rsp_dict = {}
        for index, val in enumerate(result):
            if -1 != str(val).find("Data"):
                rsp_dict = json.loads(result[index].replace('\n', ''))
                break
        if not rsp_dict:
            logger.error(f"There is no Data in response result, the response is {str(result)}")
            raise Exception(f"There is no Data in response result,the response is {str(result)}")

        if rsp_dict.get("Error").get("Code") != 0:
            raise Exception(f"Request failed,the error code is {rsp_dict.get('Error', dict()).get('Code')}")
        return rsp_dict

    try:
        return do_parse_rest_response()
    except Exception as err:
        logger.error(f"Request failed, the result is {str(result)}")
        raise Exception("Convert result to dict failed.") from err


class AccountTool(object):
    @staticmethod
    def build_unify_pwd_body_os(os_account_info, region_id, user_account, operation_type=0, user_pwd=""):
        """构建请求体
        生成注册的请求体, admin账户。
        """
        account_decs = ""
        account_type = 0
        sub_com_name = "eBackupServer"
        data = {"componentName": "eBackup",
                "subComponents": [{"subComponentName": sub_com_name, "createdAccountList": []}]}
        account_pwd = None
        modify_type = None
        if user_account == "hcp":
            account_decs = HCP_ACCOUNT_DESC
            modify_type = 1
            account_pwd = user_pwd if user_pwd else os_account_info.ssh_info.ssh_passwd
            account_type = 1

        elif user_account == "root":
            account_decs = ROOT_ACCOUNT_DESC
            modify_type = 2
            account_pwd = user_pwd if user_pwd else os_account_info.ssh_info.sudo_passwd
            account_type = 1
        account_old_pwd = account_pwd
        account_dic = {"accountName": user_account,
                       "region": region_id,
                       "accountType": account_type,
                       "accountDescription": account_decs,
                       "passwd": account_pwd,
                       "oldPasswd": account_old_pwd,
                       "lastPasswdChange": os_account_info.last_modify_time,
                       "modifyType": modify_type,
                       "passwdExpires": os_account_info.expire_time,
                       "riskMessage": '{"zh-cn":"","en-us":""}',
                       "passwdComplexity": os_account_info.quality_item,
                       "ip": os_account_info.node_ip,
                       "usedScene": os_account_info.node_ip,
                       "operationType": operation_type}
        data.get("subComponents")[0]["createdAccountList"].append(account_dic)
        return data

    @staticmethod
    def get_hcp_or_root_time_about(account, ssh_info):
        """
        获得hcp或root账号的时间相关信息
        Args:
            account: 获取时间信息的账户名
            ssh_info: 节点ssh信息

        Returns:

        """
        last_time, expire_time, _ = AccountTool.get_password_about_time(account, ssh_info)
        quality_item = AccountTool.build_unify_pwd_qulity(account)
        os_account_info = AccountQualityInfo(last_time, expire_time, quality_item, ssh_info)
        return os_account_info

    @staticmethod
    def build_unify_pwd_qulity(user_name):
        account_min_time = 0
        account_max_time = 0
        if user_name == "hcp":
            account_min_time = 0
            account_max_time = 90
        elif user_name == "root":
            account_min_time = 0
            account_max_time = 99999
        quality_item = {
            "difok": 3,
            "minlen": 8,
            "passwdChangeMinBetweenTime": account_min_time,
            "passwdChangeMaxBetweenTime": account_max_time
        }
        return quality_item

    @staticmethod
    def get_password_about_time(account, ssh_info):
        """获得账户密码相关时间，过期时间，失效时间，上一次修改时间
        Args:
            account: 账户名称
            ssh_info: 该节点ssh信息
        Returns:
            expired_time: 密码过期时间
            change_time: 上一次密码修改时间
            inactive_time:密码失效时间
        """
        output_list = []
        if account == "hcp":
            sign = "$"
            ssh_client = Utils.get_ssh_client(ssh_info, need_root_login=False)
            cmd = "chage -l {}".format(account)
            output_list = Ssh.ssh_send_command(ssh_client, cmd, sign, 20)
        if account == "root":
            sign = "#"
            cmd = "chage -l {}".format(account)
            ssh_client = Utils.get_ssh_client(ssh_info, need_root_login=True)
            output_list = Ssh.ssh_send_command(ssh_client, cmd, sign, 20)
        # 只要命令返回的结果
        if len(output_list) < SSH_RET_LENGTH:
            return ERROR_CODE, '', ''
        try:
            expired_time, change_time, inactive_time = AccountTool.deal_with_output(output_list)
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. Error Message: {error}")
            return EXCEPT_CODE, '', ''
        change_time = int(time.mktime(change_time.timetuple())) * 1000
        if expired_time != -1:
            expired_time = int(time.mktime(expired_time.timetuple())) * 1000
        if inactive_time != -1:
            inactive_time = int(time.mktime(inactive_time.timetuple())) * 1000
        return change_time, expired_time, inactive_time

    @staticmethod
    def deal_with_output(output_list):
        expired_time = -1
        change_time = ""
        inactive_time = -1
        for out_put_line in output_list:
            split_line = out_put_line.split(":")
            if len(split_line) < 2:
                continue
            temp_name = split_line[0].strip()
            temp_value = split_line[1].strip()
            if temp_name == 'Password expires':
                if temp_value == 'never':
                    expired_time = -1
                else:
                    expired_time = datetime.datetime.strptime(temp_value, '%b %d, %Y')
            elif temp_name == 'Last password change':
                change_time = datetime.datetime.strptime(temp_value, '%b %d, %Y')
            elif temp_name == 'Password inactive':
                if temp_value == 'never':
                    inactive_time = -1
                else:
                    inactive_time = datetime.datetime.strptime(temp_value, '%b %d, %Y')
        return expired_time, change_time, inactive_time


class Utils(object):
    sshclient_list = dict()

    @classmethod
    def login_openstack(cls, fsp_env_pwd, ssh_client):
        Ssh.ssh_send_command(ssh_client, "TMOUT=0", "#", 300)
        Ssh.ssh_send_command(ssh_client, "source set_env", "1|2", 300)
        Ssh.ssh_send_command(ssh_client, "2", "CPS_USERNAME", 300)
        Ssh.ssh_send_command(ssh_client, "cps_admin", "CPS_PASSWORD", 300)
        resp = Ssh.ssh_send_command(ssh_client, fsp_env_pwd, "#", 300)
        if [x for x in resp if x.find("Authenticate Failed") >= 0]:
            raise HCCIException(650051)

    @classmethod
    def close_ssh_clinet(cls, ssh_client):
        def do_close_ssh_clinet():
            Ssh().ssh_close(ssh_client)
            need_to_remove = None
            for key, client in cls.sshclient_list.items():
                if client == ssh_client:
                    need_to_remove = key
                    break
            if need_to_remove:
                cls.sshclient_list.pop(need_to_remove)
                logger.info("Remove ssh client from sshclient_list")

        try:
            do_close_ssh_clinet()
        except Exception as err:
            logger.error(f"Close ssh client failed,the reason is {str(err)}")

    @staticmethod
    def find_float_ip(ssh_info):
        def do_find_float_ip():
            nonlocal ssh_client
            ssh_client = Utils.get_ssh_client(ssh_info, login_ip=host_ip)
            result = ssh.ssh_exec_command_return_list(ssh_client, find_float_ip_cmd)
            logger.info(f"The result is {result}.")
            if -1 == str(result).find("isSingleMode"):
                float_ip_res = result[0].strip().replace("\n", "")
                if float_ip_res != "":
                    logger.info(f"The float ip of host({ssh_info.node_ips}) is {float_ip_res}.")
                    return float_ip_res
            return ''

        ssh = Ssh()
        find_float_ip_cmd = 'cat /opt/huawei-data-protection/ebackup/conf/floatIpBak || echo "isSingleMode"'
        for host_ip in ssh_info.node_ips:
            ssh_client = None
            try:
                float_ip = do_find_float_ip()
            except Exception as err:
                logger.info(f"Find float ip failed,the reason is {str(err)}")
                return ""
            finally:
                if ssh_client:
                    Utils.close_ssh_clinet(ssh_client)
            if float_ip:
                return float_ip
        else:
            logger.info("It is in SingleMode,use admin node ip as float ip.")
            float_ip = Utils.find_admin_ip(ssh_info)
            if float_ip != "":
                logger.info("It is in SingleMode,The admin node ip is %s.", float_ip)
                return float_ip
            logger.error("Find float ip of host(%s) failed.", str(ssh_info.node_ips))
            return ''

    @staticmethod
    def find_admin_ip(ssh_info):
        def do_find_admin_ip():
            nonlocal ssh_client
            ssh_client = Utils.get_ssh_client(ssh_info, login_ip=host_ip)
            result = str(ssh.ssh_exec_command_return(ssh_client, is_admin_node))
            if -1 != result.find("isAdminNode"):
                ret = str(host_ip)
                logger.info(f"The admin node of hosts({ssh_info.node_ips}) is {host_ip}")
                return ret
            else:
                return ''

        ssh = Ssh()
        is_admin_node = 'ps -ef | grep -v "grep" | grep "AdminNode" >/dev/null 2>&1 && echo "isAdminNode"'
        for host_ip in ssh_info.node_ips:
            ssh_client = None
            try:
                admin_ip = do_find_admin_ip()
            except Exception as err:
                logger.info(f"Find admin node ip failed,the reason is {err}")
                return ""
            finally:
                if ssh_client:
                    Utils.close_ssh_clinet(ssh_client)
            if admin_ip:
                return admin_ip
            else:
                continue
        else:
            logger.info(f"Find admin node of host({ssh_info.node_ips}) failed.")
            return ''

    @staticmethod
    def find_internal_ip(data_mover_ip, hcp_password, root_password):
        def do_find_internal_ip():
            nonlocal ssh_client
            cmd = "echo last get internal ip cmd result: $?;" \
                  "cat /opt/huawei-data-protection/ebackup/conf/hcpconf.ini " \
                  "| grep 'LoadbalanceAddress=' | awk -F'//' '{print $2}' |" \
                  " awk -F':' '{print $1}'"
            ssh_client = ssh.ssh_create_client(data_mover_ip, "hcp", hcp_password)
            ssh.ssh_send_command(ssh_client, 'su', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, root_password, '#', 100)
            result = ssh.ssh_exec_command_return_list(ssh_client, cmd)
            if "last get internal ip cmd result: 0" not in result:
                logger.error("Get ebk internal ip fail.")
                ssh.ssh_close(ssh_client)
                raise HCCIException("Get ebk internal ip fail.")
            return result[1]

        ssh = Ssh()
        ssh_client = None
        try:
            return do_find_internal_ip()
        except Exception as err:
            logger.info(f"Find admin node ip failed,the reason is {str(err)}")
            return ""
        finally:
            if ssh_client:
                ssh.ssh_close(ssh_client)

    @staticmethod
    def parse_login_response(rsp, web_ip):
        rsp_dict = rsp.json()
        error_code = rsp_dict["error"]["code"]
        if error_code is None:
            logger.error("Errcode is empty.")
            raise HCCIException(650009, web_ip)
        if error_code != 0:
            if error_code == -1007:
                logger.error(f"Login {web_ip} failed,the description is:{rsp_dict['error']['description']}")
                raise HCCIException(650041, web_ip)
            logger.error(f"Login {web_ip} failed,the description is:{rsp_dict['error']['description']}")
            raise HCCIException(650009, web_ip)
        account_state = rsp_dict["data"]["accountstate"]
        if account_state == 3:
            logger.error(f"Login {web_ip} failed,description:the account password is expired!")
            raise HCCIException(650025, web_ip)
        if account_state == 5:
            logger.error(f"Login {web_ip} failed,description:the account password will be expired!")
            raise HCCIException(650043, web_ip)
        token = json.loads(rsp.text)["data"]["iBaseToken"]
        session = rsp.headers.get('Set-Cookie').split(';')[0]
        return token, session

    @staticmethod
    def login(web_ip, password, ip_version=4, user="admin", scope=0):
        def do_login():
            if ip_version == 4:
                url = f'https://{web_ip}:8088/rest/dev/login'
            else:
                url = f'https://[{web_ip}]:8088/rest/dev/login'
            header = {'Accept': 'application/json', 'Accept-Language': 'en'}
            data = '{"scope":%s,"username":"%s","password":"%s"}' % (scope, user, password)

            logger.info(f"Begin to login ebackup({str(web_ip)})")
            rsp = requests.post(url, headers=header, data=data, verify=False)
            if rsp.status_code == 200:
                token, session = Utils.parse_login_response(rsp, web_ip)
            else:
                logger.error(f"Request status code is {str(rsp.status_code)},login {web_ip} failed")
                raise Exception(f"Login {web_ip} failed.Request Request status code is {str(rsp.status_code)}")
            logger.info(f"Login {web_ip} successfully.")
            return token, session

        try:
            return do_login()
        except Exception as err:
            logger.info(f"Login {web_ip} failed.")
            raise err

    @staticmethod
    def logout(web_ip, token, session, role, ip_version=4):
        def do_logout():
            if not token and not session:
                return True

            if ip_version == 4:
                url = f"https://{web_ip}:8088/rest/dev/sessions"
            else:
                url = f"https://[{web_ip}]:8088/rest/dev/sessions"
            header = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': token,
                'Cookie': 'language=en; %s; DEVICE_ID=dev;'
                          'sessionIdleTime=60000; MACHINEROLE=%s;'
                          'CSRF_IBASE_TOKEN=%s' % (session, role, token)
            }

            rsp = requests.delete(url, headers=header, verify=False)
            if rsp.status_code != 200:
                logger.error(f"Logout {web_ip} failed.")
                return False
            logger.info(f"Logout {web_ip} successfully.")
            return True

        try:
            return do_logout()
        except Exception as err:
            logger.error(f"Logout {web_ip} failed.The reason is {str(err)}")
            return False

    @staticmethod
    def login_governance(base_url, password, ssh, ssh_client):
        def do_login_governance():
            url = '%s/v3/auth/tokens' % base_url
            login_data = ''' '{"auth": {"identity" : {"methods" : ''' \
                         '''["password"],"password" : {"user" : {"name" : ''' \
                         ''' "admin","password" : "%s","iam_type" : 0} ''' \
                         '''}}}}' ''' % password
            header = '"Accept:application/json"'
            get_token_cmd = f'export LD_LIBRARY_PATH=/opt/huawei-data-protection/ebackup/libs && /opt/huawei-data-' \
                            f'protection/ebackup/sbin/SecurityTool dmk post {url} {login_data} {header} 2>/dev/null'
            result = ssh.ssh_exec_command_return(ssh_client, get_token_cmd)
            if -1 == str(result).find("ResponseCode OK"):
                logger.error("Response Code is not 200")
                raise Exception("Response Code is not 200")
            response_dict = parse_rest_response(result)
            token = response_dict.get("Data").get("token")
            return token

        try:
            return do_login_governance()
        except Exception as err:
            logger.error(f"login governance failed, error:{str(err)}.")
            raise err

    @staticmethod
    def logout_governance(base_url, token, ssh, ssh_client):
        def do_logout_governance():
            if not token or not ssh_client:
                return True

            url = "%s/v3/auth/tokens" % base_url
            security_tool = '/opt/huawei-data-protection/ebackup/sbin/SecurityTool'
            logout_cmd = '%s dmk delete %s "" "X-Auth-Token:%s" 2>/dev/null' % (security_tool, url, token)
            result = ssh.ssh_exec_command_return(ssh_client, logout_cmd)
            if str(result).find("ResponseCode OK") == -1:
                logger.error("Response Code is not 200")
                raise Exception("Response Code is not 200")
            parse_rest_response(result)
            logger.info(f"Logout governance {base_url} successfully")
            return True

        try:
            return do_logout_governance()
        except Exception as login_err:
            logger.error(f"Logout governace failed,the reason is: {login_err}")
            return False

    @staticmethod
    def check_ip_version(common_ip):
        try:
            m_ip = IP(common_ip)
        except Exception as err:
            logger.error(f"{common_ip} is not a valid ip address.The reason is {str(err)}")
            return -1
        return m_ip.version()

    @staticmethod
    def init_system_params(project_id, region_id):
        def do_init_system_params():
            util = ParamUtil()
            params_dict = util.get_service_cloud_param(project_id, "eBackup", region_id)
            if not params_dict.get("eBackup_Datamover_nodes"):
                params_dict.update({"eBackup_Datamover_nodes":
                                    Utils.get_value_from_config_file("eBackup_Datamover_nodes")})
            Utils.update_param(project_id, region_id, util, params_dict)
            logger.info("Init eBackup system parameters successfully.")
            return params_dict

        try:
            return do_init_system_params()
        except Exception as err:
            logger.error(f"Init eBackup system parameters failed.the reason is {str(err)}")
            raise err

    @staticmethod
    def update_param(project_id, region_id, util, params_dict):
        common_params = {
            "dmk_floatIp": util.get_value_from_cloud_param(project_id, "DMK_public_params", "dmk_floatIp", region_id),
            "dmk_os_business_username": util.get_value_from_cloud_param(project_id, "DMK_public_params",
                                                                        "dmk_os_business_username", region_id),
            "dmk_os_business_user_password": util.get_value_from_cloud_param(project_id, "DMK_public_params",
                                                                             "dmk_os_business_user_password",
                                                                             region_id),
            "dmk_ui_username": util.get_value_from_cloud_param(project_id, "DMK_public_params", "dmk_ui_username",
                                                               region_id),
            "dmk_ui_password": util.get_value_from_cloud_param(project_id, "DMK_public_params", "dmk_ui_password",
                                                               region_id),
            "openstack_reverse_proxy_ip": util.get_value_from_cloud_param(project_id, 'OpenStack',
                                                                          'openstack_reverse_proxy_ip', region_id),
            "openstack_fsp_pwd": util.get_value_from_cloud_param(project_id, 'OpenStack', 'openstack_fsp_pwd',
                                                                 region_id),
            "openstack_root_pwd": util.get_value_from_cloud_param(project_id, 'OpenStack', 'openstack_root_pwd',
                                                                  region_id),
            "openstack_external_api_cps_web_port": util.get_value_from_cloud_param(
                project_id, 'OpenStack', 'openstack_external_api_cps_web_port', region_id),
            "openstack_cloud_admin_pwd": util.get_value_from_cloud_param(project_id, 'OpenStack',
                                                                         'openstack_cloud_admin_pwd', region_id),
            "current_region_id": region_id
        }
        params_dict.update(common_params)
        params_dict.update(
            {"eBackup_Datamover_nodes": params_dict.get("eBackup_Datamover_nodes").replace(",", ";").lower()})

    @staticmethod
    def query_openstack_info(float_ip, password):
        token = None
        session = None
        ret_dict = {}

        def do_query_openstack_info():
            nonlocal token, session, ret_dict
            logger.info("Begin to query openstack info at host:[%s]", float_ip)
            ip_version = Utils.check_ip_version(float_ip)
            if ip_version == 4:
                url = f"https://{float_ip}:8088/rest/dev/openstack_settings/default"
            else:
                url = f"https://[{float_ip}]:8088/rest/dev/openstack_settings/default"

            token, session = Utils.login(float_ip, password, ip_version)
            header = {'Accept': 'application/json;version=2.2;charset=UTF-8',
                      'iBaseToken': token, 'Accept-Language': 'en',
                      'Cookie': f'language=en; {session};DEVICE_ID=dev;sessionIdleTime='
                                f'60000;MACHINEROLE=2;CSRF_IBASE_TOKEN={token}'}
            rsp = requests.get(url=url, headers=header, verify=False)
            if rsp.status_code == 200:
                rsp_dict = rsp.json()
                error_code = rsp_dict["error"]["code"]
                if error_code is None:
                    logger.error("errcode is empty.")
                    raise Exception("errcode is empty.")
                if error_code != 0:
                    logger.error(f"Query openstack info failed on [{float_ip}],"
                                 f"the description is:{rsp_dict['error']['description']}")
                    raise Exception(rsp_dict["error"]["description"])
                ret_dict = rsp_dict["data"]
            else:
                logger.error(f"Response status code is not 200.The url is {url}")
                raise Exception("Response status code is not 200.")

            logger.info(f"Query openstack info succ, iam url:{ret_dict['AUTHURL']}, nova url:{ret_dict['NOVAURL']}, "
                        f"cinder url:{ret_dict['CINDERURL']}, neutron:{ret_dict['NEUTRONURL']}.")

        try:
            do_query_openstack_info()
        except Exception as err:
            logger.error(f"Query openstack info failed.The reason is: {str(err)}")
            return {}
        finally:
            Utils.logout(float_ip, token, session, 0)
        return ret_dict

    @staticmethod
    def query_alarm_info(float_ip, password):
        token = None
        session = None

        def do_query_alarm_info():
            nonlocal token, session
            logger.info("Begin to query alarm info.")
            manageone_oc_base_url = ''
            ip_version = Utils.check_ip_version(float_ip)
            if ip_version == 4:
                url = f"https://{float_ip}:8088/rest/dev/alarm_settings/default"
            else:
                url = f"https://[{float_ip}]:8088/rest/dev/alarm_settings/default"

            token, session = Utils.login(float_ip, password, ip_version)
            header = {'Accept': 'application/json;version=2.2;charset=UTF-8',
                      'iBaseToken': token, 'Accept-Language': 'en',
                      'Cookie': f'language=en; {session};DEVICE_ID=dev;sessionIdleTime='
                                f'60000;MACHINEROLE=2;CSRF_IBASE_TOKEN={token}'}
            rsp = requests.get(url=url, headers=header, verify=False)
            if rsp.status_code == 200:
                rsp_dict = rsp.json()
                error_code = rsp_dict["error"]["code"]
                if error_code is None:
                    logger.error("errcode is empty.")
                    raise Exception("errcode is empty.")
                if error_code != 0:
                    logger.error(f"Query alarm info failed on [{float_ip}],"
                                 f"the description is:{rsp_dict['error']['description']}")
                    raise Exception(rsp_dict["error"]["description"])
                if "ALARMSERVERURL" not in rsp_dict["data"] or 0 == len(rsp_dict["data"]["ALARMSERVERURL"]):
                    logger.error("Alarm report is not configured.")
                    raise Exception("Alarm report is not configured.")
                if rsp_dict["data"]["ISALARMON"] != '1':
                    logger.error("Alarm report is not enable.")
                    raise Exception("Alarm report is not enable.")
                manageone_oc_base_url = rsp_dict["data"]["ALARMSERVERURL"]
            logger.info("The oc url is:" + str(manageone_oc_base_url))
            return "https://" + manageone_oc_base_url.split(':')[0]

        try:
            oc_domain_name = do_query_alarm_info()
        except Exception as query_err:
            logger.error(f"Query alarm info failed.The reason is: {query_err}")
            return ""
        finally:
            Utils.logout(float_ip, token, session, 0)
        logger.info(f"The oc domain name is: {str(oc_domain_name)}")
        return oc_domain_name

    @classmethod
    def get_ssh_client(cls, ssh_info, need_root_login=True, login_ip=""):
        ssh_client = None
        host_ip = ''
        if isinstance(ssh_info.node_ips, list):
            if login_ip:
                host_ip = login_ip
            elif len(ssh_info.node_ips) == 1:
                host_ip = ssh_info.node_ips[0]
            else:
                logger.error(f"ip is not correct: ips: {ssh_info.node_ips}, login_ip: {login_ip}.")
                raise Exception("ssh client ip is invalid.")
        elif isinstance(ssh_info.node_ips, str):
            host_ip = ssh_info.node_ips

        if not host_ip:
            logger.error(f"ip is not correct: ips: {ssh_info.node_ips}.")
            raise Exception("ssh client ip is invalid.")

        root_pwd = ssh_info.sudo_passwd
        user = ssh_info.ssh_user
        user_pwd = ssh_info.ssh_passwd
        need_root_login = bool(root_pwd) if need_root_login else False
        if (host_ip, need_root_login) in cls.sshclient_list:
            ssh_client = cls.sshclient_list[(host_ip, need_root_login)]
        if ssh_client:
            try:
                Ssh.ssh_send_command(ssh_client, "echo checkssh", "checkssh", 1)
            except Exception as ex:
                logger.warn(f"ssh socket is closed, need reconnect. stdout: {str(ex)}.")
            else:
                return ssh_client

        def create_ssh_client():
            ssh_client = Ssh.ssh_create_client(host_ip, user, user_pwd, timeout=30)
            if need_root_login:
                Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 60)
                Ssh.ssh_send_command(ssh_client, root_pwd, '#', 60)
                Ssh.ssh_send_command(ssh_client, "TMOUT=0", '#', 60)
            logger.info(f"Create ssh client for {host_ip} successfully.")
            cls.sshclient_list[(host_ip, need_root_login)] = ssh_client
            return ssh_client

        try:
            return create_ssh_client()
        except Exception as ex:
            Utils.close_ssh_clinet(ssh_client)
            logger.error(f"Create ssh client for {host_ip} failed, error info:{str(ex)}.")
            return ''

    @staticmethod
    def ssh_send_comand_return(ssh_client, cmd, ebk_timeout=20):
        """
        执行ssh命令
        :param ssh_client:  ssh客户端
        :param cmd:  执行命令
        :param ebk_timeout:  超时时间单位分钟
        :return: 命令执行情况
        """
        def do_ssh_send_comand_return():
            if not ssh_client:
                logger.error("ssh_client is invalid.")
                return ''
            return Ssh.ssh_exec_command_return(ssh_client, cmd, timeout=ebk_timeout)

        try:
            return do_ssh_send_comand_return()
        except Exception as err:
            logger.error(f"Failed to execute cmd. {err}")
            return ''

    @staticmethod
    def parse_config_str(config_str):
        config_dict = dict()
        for config_item in config_str.split("\n"):
            if not config_item.strip():
                continue
            key = config_item.split('=')[0].strip()
            value = config_item.split('=')[1].strip()
            config_dict[key] = value
        return config_dict

    @staticmethod
    def parse_dict_to_str(config_dict):
        config_str = ""
        for key, value in config_dict.items():
            config_str += f"{key}={value}\n"
        return config_str

    @staticmethod
    def set_value_to_config_file(key, value):
        logger.info(f"set value({value}) to key({key}).")
        config_file_path = os.path.realpath(__file__ + "/../config.ini")
        if os.path.exists(config_file_path):
            with open(config_file_path, 'r') as config_file:
                config_str = config_file.read()
        else:
            config_str = ""
        config_dict = Utils.parse_config_str(config_str)
        config_dict[key] = value
        file_flags = os.O_WRONLY | os.O_CREAT
        file_modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(config_file_path, file_flags, file_modes), 'w') as config_file:
            config_file.truncate()
            config_file.write(Utils.parse_dict_to_str(config_dict))

    @staticmethod
    def get_value_from_config_file(key):
        logger.info(f"get key{key} from config file.")
        config_file_path = os.path.realpath(__file__ + "/../config.ini")
        if not os.path.exists(config_file_path):
            return ''
        with open(config_file_path, 'r') as config_file:
            config_str = config_file.read()
            config_dict = Utils.parse_config_str(config_str)
            return config_dict.get(key, '')

    @staticmethod
    def get_server_url(management_float_ip):
        if IP(management_float_ip).version() == IPV6_NUMBER:
            return f"https://[{management_float_ip}]:8088/rest/dev"
        return f"https://{management_float_ip}:8088/rest/dev"

    @staticmethod
    def check_ssh_port(ip_address):
        ssh_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            logger.info(f"check ssh port on node {ip_address}")
            result = ssh_socket.connect_ex((ip_address, 22))
        except Exception as err:
            logger.error(f"Failed to connect node{ip_address} 22 port. "
                         f"reason {str(repr(err))}.")
            result = 1
        finally:
            ssh_socket.close()
        return not bool(result)

    @classmethod
    def get_node_manage_ip(cls, ip_address):
        if len(ip_address) == 1:
            return ip_address[0].get('ip')
        for _ip in ip_address:
            if cls.check_ssh_port(_ip.get('ip')):
                return _ip.get('ip')
        return ip_address[0].get('ip')

    @classmethod
    def get_ebackup_datamover_nodes_from_cmdb(cls, project_id):
        ebackup_datamover_nodes_dict = {}
        data = ParamUtil().get_need_check_cloud_params(
            project_id, "ManageOne_public_params")
        current_region_id = BaseOps().get_regionId_by_projectId(project_id)[0]
        cmdb_ip = data.get("cmdb_ip")
        cmdb_password = data.get("cmdb_password")
        cmdb_account = data.get("cmdb_account")
        cmdb = ManageOneCmdbUtil.get_instance(cmdb_ip, cmdb_password,
                                              cmdb_account, project_id)
        cmdb_nodes_list = cmdb.get_deploy_node_info(current_region_id)
        if not cmdb_nodes_list:
            logger.error("Query cmdb nodes info failed.")
            raise Exception("Query cmdb nodes info failed.")
        for node in cmdb_nodes_list:
            if node["name"].find("eBackup_service") != -1:
                group_name = node.get("name")[:-1] + 's'
                manage_ip = cls.get_node_manage_ip(node.get('ipAddresses'))
                if group_name not in ebackup_datamover_nodes_dict:
                    ebackup_datamover_nodes_dict.update({group_name: [manage_ip]})
                else:
                    ebackup_datamover_nodes_dict[group_name].append(manage_ip)
            else:
                logger.info(f"Skip non-ebackup nodes{node.get('name')} info.")
        if not ebackup_datamover_nodes_dict:
            logger.error("Get ebackup datamover nodes info failed.")
            raise Exception("Get ebackup datamover nodes info failed.")
        return '|'.join([';'.join(nodes) for nodes in ebackup_datamover_nodes_dict.values()])

