# -*- coding:utf-8 -*-
import os
import time

import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.common.util import Utils
from plugins.eBackup.scripts.upgrade.util import CommonConfig
from plugins.eBackup.scripts.upgrade.util import get_cipher_status

CHECK_SERVICE_STATUS_CMD = 'service hcp status 2>/dev/null | grep -v ' \
                           '"dsware_agent" | grep "isn\'t running" ' \
                           '|| echo "successful"'

FORBID_HA_SWITCH_CMD = 'export LD_LIBRARY_PATH=/opt/huawei-data-protection' \
                       '/ebackup/libs/ &&  /opt/huawei-data-protection/' \
                       'ebackup/ha/module/hacom/tools/ha_client_tool' \
                       ' --forbidswitch   --name=adminnode --time=60 '

RESTORE_HA_SWITCH_CMD = 'export LD_LIBRARY_PATH=/opt/huawei-data-protection/' \
                        'ebackup/libs/ &&  /opt/huawei-data-protection/' \
                        'ebackup/ha/module/hacom/tools/ha_client_tool' \
                        ' --cancelforbidswitch --name=adminnode '

CHECK_CIPHER_CHANGE_FINISH_CMD = 'find /opt/huawei-data-protection/ebackup/ ' \
                                 '-name "hcpconf.ini" 2>/dev/null|' \
                                 ' xargs cat | grep -E "Re.*Encrypt=1" ' \
                                 '&& echo "changing"'

STOP_EBACKUP_SERVICE_CMD = 'service hcp stop force>/dev/null 2>&1' \
                           ' && echo "successful"'

START_EBACKUP_SERVICE_CMD = 'service hcp start force>/dev/null 2>&1' \
                            ' && echo "successful"'

CHANGE_CIPHER_CMD = 'find /opt/huawei-data-protection/ebackup/ ' \
                    '-name "hcpconf.ini" 2>/dev/null |' \
                    ' xagrs sed -i \'s/KmcAlgMode=.*/KmcAlgMode=%s/\''


class DatamoverChange(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverChange, self).__init__(project_id, pod_id,
                                              regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.change_alg_mode = get_cipher_status(self.project_id)
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            host_info = self.__db_param_dict["eBackup_Datamover_nodes"]
            host_info = host_info.replace(" ", "").split("|")

            for group in host_info:
                ebackup_changer = EbackupCipherChanger(group, self.__db_param_dict, self.project_id,
                                                       region_id=self.regionid_list[0], pod_id=self.pod_id)
                ebackup_changer.do_change(self.change_alg_mode)
            return Message(200)

        try:
            return do_execute()
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn=f"切换Datamover加密算法出现异常,请查看日志并重试，异常信息：{str(ex)}",
                           error_msg_en=f"Exception occurs when changing Datamover cipher alg, "
                                        f"please see log and try again, error info:{str(ex)}")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class UpdateCMDB(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(UpdateCMDB, self).__init__(project_id, pod_id,
                                         regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.change_alg_mode = get_cipher_status(self.project_id)
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            ebackup_changer = EbackupCipherChanger(None, self.__db_param_dict, self.project_id,
                                                   region_id=self.regionid_list[0], pod_id=self.pod_id)
            ebackup_changer.update_cmdb_info(self.change_alg_mode)
            return Message(200)

        try:
            return do_execute()
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn=f"更新CMDB信息出现异常,请查看FCU日志并重试，异常信息：{str(ex)}",
                           error_msg_en=f"Exception occurs when changing Datamover cipher, "
                                        f"please see log and try again, error info:{str(ex)}")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class EbackupCipherChanger(object):
    def __init__(self, host_ips, param_dict, project_id, region_id=None, pod_id=None):
        self.primary_ip = None
        self.host_ips = host_ips
        self.param_dict = param_dict
        self.project_id = project_id
        self.region_id = region_id
        self.pod_id = pod_id
        self.ssh = Ssh()
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.cipher_change_file_path = os.path.realpath(__file__ + '/../../../shell_tool/cipher_change.sh')

    def do_change(self, change_alg_mode):
        # 1.forbid ha switch
        ret = self.ha_switch_execute(FORBID_HA_SWITCH_CMD)
        if not ret[0]:
            raise HCCIException(650045, self.host_ips, ret[1])

        # 2.check cipher alg change status.

        # 3.stop eBackup standby|proxy Service
        for host_ip in self.host_ips.split(';'):
            if host_ip != self.primary_ip:
                ret = self.execute_ssh_cmd(host_ip, STOP_EBACKUP_SERVICE_CMD)
                if not ret[0]:
                    raise HCCIException(650047, host_ip, ret[1])

        # 4. stop primary node:
        ret = self.execute_ssh_cmd(self.primary_ip, STOP_EBACKUP_SERVICE_CMD)
        if not ret[0]:
            raise HCCIException(650047, self.primary_ip, ret[1])

        # 5. DO CHANGE;
        for host_ip in self.host_ips.split(';'):
            self.ssh.put_file(host_ip, 'hcp',
                              self.param_dict['eBackup_hcp_pwd'],
                              self.cipher_change_file_path, '/home/hcp')
            ret = self.execute_ssh_cmd(
                host_ip, 'sh /home/hcp/cipher_change.sh "%s" && echo "successful"' % change_alg_mode)
            if not ret[0]:
                raise HCCIException(650048, host_ip, ret[1])
            if -1 == str(ret[1]).find('successful'):
                logger.error("execute cipher change script on node(%s) "
                             "failed. reason: %s" % (host_ip, ret[1]))
                raise HCCIException(650048, host_ip, ret[1])
            logger.info(
                "success execute cipher change script on node(%s)" % host_ip)

        # 6.Set kms info;
        common_config = CommonConfig(self.param_dict, region_id=self.region_id, project_id=self.project_id,
                                     pod_id=self.pod_id)
        common_config.config_kms_info(self.host_ips.split(';'), True)

        # 7. start service
        ret = self.execute_ssh_cmd(self.primary_ip, START_EBACKUP_SERVICE_CMD)
        if not ret[0]:
            raise HCCIException(650047, self.primary_ip, ret[1])
        for host_ip in self.host_ips.split(';'):
            if host_ip != self.primary_ip:
                ret = self.execute_ssh_cmd(host_ip, START_EBACKUP_SERVICE_CMD)
                if not ret[0]:
                    raise HCCIException(650047, host_ip, ret[1])

        # 8. check cipher alg change finish

        # 9. cancel ha switch
        ret = self.ha_switch_execute(RESTORE_HA_SWITCH_CMD)
        if not ret[0]:
            raise HCCIException(650049, self.host_ips, ret[1])

        # 10.reencrypt the nginx cert
        for host_ip in self.host_ips.split(';'):
            self.reencrypt_nginx_cert(host_ip)

    def reencrypt_nginx_cert(self, host_ip):
        cmd = "sh /opt/root_tool/common/" \
              "sudo_common_func.sh cert_cipher_reencrypt" \
              " && echo 'reencrypt cert successful.'"
        flag, result = self.execute_ssh_cmd(host_ip, cmd)
        if flag and str(result).find("reencrypt cert successful.") != -1:
            logger.info("reencrypt nginx cert successful on node %s." % host_ip)
            return
        else:
            logger.error("Failed to reencrypt nginx cert on node %s."
                         " stdout: %s." % (host_ip, str(result)))
            raise HCCIException(650055, host_ip)

    def execute_ssh_cmd(self, host_ip, cmd):
        account_name = 'hcp'
        account_pwd = self.param_dict['eBackup_hcp_pwd']
        root_passwd = self.param_dict['eBackup_root_pwd']

        ssh_client = self.ssh.ssh_create_client(host_ip, account_name, account_pwd)

        def do_execute_ssh_cmd():
            self.ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            self.ssh.ssh_send_command(ssh_client, root_passwd, '#', 100)
            result = self.ssh.ssh_exec_command_return_list(ssh_client, cmd)
            self.ssh.ssh_close(ssh_client)
            return True, result

        try:
            return do_execute_ssh_cmd()
        except Exception as err:
            des = 'execute cmd on node(%s) fail. reason: %s' % (host_ip, str(err))
            logger.error(des)
            return False, des

    def check_cipher_change_status(self):
        for host_ip in self.host_ips.split(';'):
            count = 0
            logger.info("start check cipher alg change status on node(%s)" % host_ip)
            while count < 6:
                ret = self.execute_ssh_cmd(host_ip, CHECK_CIPHER_CHANGE_FINISH_CMD)
                if not ret[0]:
                    return host_ip, ret[1]

                if -1 == str(ret[1]).find('changing'):
                    logger.info('node(%s) changing the cipher alg finish.' % host_ip)
                    count = -1
                    break
                logger.info('node(%s) is still changing the cipher alg.' % host_ip)
                count += 1
                time.sleep(10)

            if count != -1:
                des = "node(%s) cipher alg change status error, check service status." % host_ip
                logger.error(des)
                return host_ip, des
        return '', ''

    def ha_switch_execute(self, switch_cmd):
        hcp_passwd = self.param_dict['eBackup_hcp_pwd']
        root_passswd = self.param_dict['eBackup_root_pwd']
        server_ips = self.host_ips.split(';')
        ssh_client = SshInfo(server_ips, "hcp", hcp_passwd, root_passswd)
        self.primary_ip = Utils.get_value_from_config_file("eBackup_primary_node")
        if self.primary_ip:
            ret = ['Success']
        else:
            if len(server_ips) > 1:
                self.primary_ip = Utils.find_admin_ip(ssh_client)
            else:
                self.primary_ip = server_ips[0]
            ret = self.execute_ssh_cmd(self.primary_ip, switch_cmd)
        if not ret[0]:
            logger.error('Node(%s) execute ha command fail.' % self.primary_ip)
            return False, ret[1]
        logger.info("Node(%s) execute ha command success." % self.primary_ip)
        Utils.set_value_to_config_file("eBackup_primary_node", self.primary_ip)
        return True, ""

    def get_cipher_name(self, change_alg_mode):
        if change_alg_mode == '0':
            return "generalCipher"
        elif change_alg_mode == '1':
            return "SMCompatible"
        elif change_alg_mode == '2':
            return "SMOnly"
        else:
            return ''

    def update_cmdb_info(self, change_alg_mode):
        logger.info('start update cmdb info')
        self.cmdb_util = ManageOneCmdbUtil(self.project_id)
        node_infos = self.cmdb_util.get_cloud_service_info(self.param_dict['current_region_id'], "eBackup")
        cipher_name = self.get_cipher_name(change_alg_mode)
        for node_info in node_infos:
            extend_infos = node_info.get("extendInfos")
            update_extend_info = []
            for extend_info in extend_infos:
                if extend_info.get('key') != 'CipherType':
                    update_extend_info.append(extend_info)

            update_extend_info.append({"key": "CipherType", "value": cipher_name})
            node_info['extendInfos'] = update_extend_info
            self.cmdb_util.set_cloud_service_info(self.param_dict['current_region_id'], node_info)
