# -*- coding:utf-8 -*-
import json
import time

import utils.common.log as logger
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common.iam_util import IamUtil
from plugins.eBackup.scripts.common.ebackup_util import CustomParametersUtil
from plugins.eBackup.scripts.common.ebackup_util import CommonTools
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.common.util import AccountTool

logger.init("eBackup")
ADMIN_ACCOUNT_DESC = '{"zh-cn":"用于登录GUI和CLI，具有超级管理员权限。",' \
                     '"en-us":"Used to log in to the GUI and CLI and ' \
                     'has the super administrator permissions."}'

IAM_ACCOUNT_DESC = '{"zh-cn":"eBackup服务使用的IAM账号。",' \
                   '"en-us":"The IAM account for eBackup service."}'
REST_URI_GOV_UNIFYPWD_MODIY = "/v1/srv_admin/modifypwd"
REST_URI_GOV_UNIFYPWD_RESET = "/v1/srv_admin/resetpwd"
REST_URI_GOV_UNIFYPWD_QUERYJOB = "/v1/srv_admin/querypwdstat"
REST_URI_GOV_UNIFYPWD_VERIFY = "/v1/srv_admin/verifypwd"
HCP_ACCOUNT_DESC = '{"zh-cn":"使用该帐户登录eBackup' \
                   '服务器进行运维操作，查看节点配置，查看日志，' \
                   '巡检和信息收集等。","en-us":"Used to log in ' \
                   'to an eBackup server to perform O&M operations, ' \
                   'view node configurations and logs, perform inspection, ' \
                   'and collect information."} '
ROOT_ACCOUNT_DESC = '{"zh-cn":"考虑到对系统安全的影响，在登录节点时，' \
                    '不能使用root用户直接登录，需要由hcp' \
                    '用户登录后通过执行命令 su - ' \
                    'root切换为root用户。切换到root' \
                    '帐号后，可对服务进行日常操作和维护，例如执行进程查看、' \
                    '日志查看、密码修改、配置文件修改等操作。",' \
                    '"en-us":"To ensure system security, you cannot log in ' \
                    'to the node as user root. You need to log in to the ' \
                    'node as user hcp and run the su - root command to ' \
                    'switch to user root. After switching to user root, ' \
                    'you can perform routine operations and maintenance on ' \
                    'services, such as viewing processes and logs, changing ' \
                    'passwords and configuration files."} '

RETRY_TIME_LIMIT = 5
# 命令返回的标准长度
SSH_RET_LENGTH = 3
EXCEPT_CODE = -3
ERROR_CODE = -2


class RegisterCertToManageone(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.params = ParamUtil()
        self.pararms_util = CustomParametersUtil(project_id, pod_id)
        self.rest_api = EbackupRest()
        self.ssh_obj = Ssh()
        self.headers = None
        self.ebk_iam_headers = None
        self.if_ipv6 = None
        self.openstack_config_url = ""
        self.auth_url = ""
        self.nova_url = ""
        self.cinder_url = ""
        self.neutron_url = ""
        self.__config_dict = self.params.get_service_cloud_param(
            pod_id, "eBackup")
        self.clear_history_cmd = ">~/.bash_history&&history -c&&rm -rf /home/hcp/RegisterCertToManageone.sh"
        self.hcp_passwd = self.__config_dict.get('hcp_ssh_password')
        self.root_pwd = self.__config_dict.get('eBackup_os_root_password')
        self.ebackup = CommonTools(self.project_id, self.pod_id)
        self.iam_util = IamUtil(self.project_id, self.pod_id)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            if get_project_condition_boolean(
                    project_id, 'eBackup_ServerProxy_Driver'):
                self.get_network_type()
                self.ebackup_register_cert()
                self.update_datamover_config()
                self.ebackup_register_unify_pwd()
                self.ebackup_register_unify_pwd_hcp_and_root()
            elif get_project_condition_boolean(
                    project_id, 'eBackup_Install_Complete'):
                self.get_network_type()
                self.update_datamover_config()
                self.ebackup_register_cert()
                self.ebackup_register_unify_pwd()
                self.ebackup_register_unify_pwd_hcp_and_root()

            elif get_project_condition_boolean(
                    project_id, 'eBackup_Proxy'
            ):
                self.update_proxy_config()
                # 把新增的节点直接注册
                self.ebackup_register_unify_pwd_hcp_and_root()
            return Message(200)
        except HCCIException as err:
            logger.error(str(err))
            return Message(500, HCCIException("640016", str(err)))
        except Exception as err:
            logger.error("execute error:%s" % str(err))
            return Message(500, err)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
         标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_network_type(self):
        self.if_ipv6 = 0
        pro_dic = get_project_conditions(self.project_id)
        if 'ManageIPV6' in pro_dic:
            self.if_ipv6 = pro_dic["ManageIPV6"]

    def login(self, mo_ip, pwd, component):
        """
        改写的login 方法
        :param ip:
        :param pwd:
        :param component: 子部件名称
        :return:
        """
        logger.info("begin to login %s." % component)
        login_url = "https://%s:8088/rest/dev/login" % mo_ip
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"scope":0,"username":"admin","password":"' + pwd + '"}'
        rsp = self.rest_api.post(login_url, headers, data)
        if component == 'ebackup':
            machinerole = 0
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            self.headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': token,
                'Cookie': 'language=en;%s;DEVICE_ID=dev; '
                          'sessionIdleTime=60000; MACHINEROLE=%s; '
                          'CSRF_IBASE_TOKEN=%s' % (
                              session_content, machinerole,
                              token)}
            logger.info("login ebackup succ.")
        else:
            description = rsp.json()["error"]["description"]
            logger.error("login failed,description:" + description)

    def logout_ebk_iam(self, lb_listen_addr):
        if not self.ebk_iam_headers:
            return
        url = "https://%s/v3/auth/tokens" % lb_listen_addr
        self.rest_api.delete(url, self.ebk_iam_headers)
        logger.info("log out ebk_iam succ.")

    def register_cert(self, lb_listen_addr):
        tanent_info = ManageOneUtil2().getMoTanentInfo(self.pod_id)
        cert_ip = tanent_info[0]
        domain_name = 0
        if tanent_info.__len__() >= 5:
            domain_name = 1
            cert_ip = tanent_info[4]
        if self.if_ipv6 and domain_name == 0:
            cert_ip = '[%s]' % cert_ip
        cert_url = "https://" + cert_ip + ":26335"
        om_url = "https://" + self.pararms_util.get_oc_domain_name()
        logger.info("cert url is %s, om url is %s." % (cert_url, om_url))
        register_url = "https://%s/v1/srv_governance_inner/settings" % \
                       lb_listen_addr
        region_id = self.__config_dict.get('region_id')
        iam_account = self.iam_util.get_iam_account_info("op_service")
        data = '{"Settings":[{"ConfDomain":"UnifyCert","ConfItem":' \
               '"ManageOneAddr","ConfValue":"%s"},{"ConfDomain":"UnifyCert",' \
               '"ConfItem":"ManageOneOCAddr","ConfValue":"%s"},' \
               '{"ConfDomain":"UnifyCert","ConfItem":"CurrentRegion",' \
               '"ConfValue":"%s"},{"ConfDomain":"UnifyCert","ConfItem":' \
               '"OpenstackIAMUserName","ConfValue":"%s"},{"ConfDomain":' \
               '"UnifyCert","ConfItem":"OpenstackIAMUserPassword",' \
               '"ConfValue":"%s"}]}' % (cert_url, om_url, region_id,
                                        iam_account.account_name, iam_account.account_pwd)
        rsp = self.rest_api.patch(register_url, self.ebk_iam_headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode != 0:
            description = rsp.json()["Error"]["Description"]
            logger.error("register cert failed, description:%s" % description)
            raise HCCIException(653064, lb_listen_addr, description)

    def ebackup_register_cert(self):
        ip_list = self.__config_dict.get('datamover_externalom_iplist', "").lower()
        datamover_iplist = ip_list.split(',')
        lb_listen_addr = "%s:8090" % self.__config_dict.get('datamover_internal_float_ip')
        self.no_limit_8090_iptables(datamover_iplist[:2], self.__config_dict.get('datamover_internal_float_ip'))
        self.login_ebk_iam(lb_listen_addr)
        self.register_cert(lb_listen_addr)
        logger.info("register ebackup cert to manageone succ.")

    def ebackup_register_unify_pwd(self):
        ip_list = self.__config_dict.get('datamover_externalom_iplist').lower()
        datamover_iplist = ip_list.split(',')
        lb_listen_addr1 = "%s:8090" % self.__config_dict.get('datamover_internal_float_ip')
        if len(datamover_iplist) > 1:
            lb_listen_addr2 = "%s:8088" % self.__config_dict.get('datamover_management_float_ip')
        else:
            lb_listen_addr2 = "%s:8088" % datamover_iplist[0]
        delete_data = self.build_unify_pwd_body_admin(lb_listen_addr1, lb_listen_addr2, 2)
        self.send_unify_pwd_req(delete_data)
        data = self.build_unify_pwd_body_admin(lb_listen_addr1, lb_listen_addr2)
        result = self.send_unify_pwd_req(data)
        if not result:
            raise HCCIException(653089)
        self.logout_ebk_iam(lb_listen_addr1)
        self.limit_8090_iptables(datamover_iplist[0:2], self.__config_dict.get('datamover_internal_float_ip'))
        logger.info("register ebackup unify pwd to manageone succ.")

    def get_admin_pwd_quality(self, lb_listen_addr):
        url = f"https://{lb_listen_addr}/v3/auth/security_policy"
        rsp = self.rest_api.get(url, self.ebk_iam_headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            return ""
        errcode = rsp.json()["Error"]["Code"]
        if errcode != 0:
            description = rsp.json()["Error"]["Description"]
            logger.error(
                "get admin pwd quality failed, description:%s" % description)
            raise HCCIException(653085, description)
        data_info = rsp.json()["Data"]
        quality_item = {
            "difok": int(data_info["PwdComplex"]) + 3,
            "minlen": int(data_info["PasswordMinLength"]),
            "ucredit": 1,
            "lcredit": 1,
            "dcredit": 1,
            "ocredit": 1,
            "passwdChangeMinBetweenTime": int(data_info["MinValidPeriod"]),
            "passwdChangeMaxBetweenTime": int(data_info["MaxValidPeriod"])
        }
        return quality_item

    def test_login_ebk_iam(self, lb_listen_addr):
        errcode = '0'
        logger.info("begin to login ebk_iam micro service.")
        login_url = "https://%s:8090" % lb_listen_addr
        headers = {'Accept': 'application/json'}

        try:
            self.rest_api.get(login_url, headers)
        except Exception as err:
            errcode = '1'
            logger.error("Test the connection to the IAM failed:%s" % str(err))
        return errcode

    def no_limit_8090_iptables(self, ip_list, float_ip):
        if self.test_login_ebk_iam(float_ip) == '1':
            cmd = "iptables -I INPUT  -p tcp -s 0.0.0.0/0" \
                  " -d 0.0.0.0/0 --dport 8090 -j ACCEPT"
            for ebk_ip in ip_list:
                self.ssh_obj.ssh_cmds(
                    ebk_ip, cmd, "hcp", self.hcp_passwd,
                    self.root_pwd, "", "")

    def limit_8090_iptables(self, ip_list, float_ip):
        if self.test_login_ebk_iam(float_ip) == '0':
            cmd = "iptables -D INPUT  -p tcp -s 0.0.0.0/0" \
                  " -d 0.0.0.0/0 --dport 8090 -j ACCEPT"
            for ebk_ip in ip_list:
                self.ssh_obj.ssh_cmds(
                    ebk_ip, cmd, "hcp", self.hcp_passwd,
                    self.root_pwd, "", "")

    def login_ebk_iam(self, lb_listen_addr):
        logger.info("begin to login ebk_iam micro service.")
        login_url = "https://%s/v3/auth/tokens" % lb_listen_addr
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"auth":{"identity":{"methods":["password"],"password":' \
               '{"user":{"name":"admin","password":"%s",' \
               '"iam_type":0}}}}}' % \
               self.__config_dict.get('eBackup_weblogin_password')
        rsp = self.rest_api.post(login_url, headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            token = rsp.json()["Data"]["token"]
            self.ebk_iam_headers = {
                "Accept": "application/json",
                "X-Auth-Token": token}
            logger.info("login ebk_iam micro service succ.")
        else:
            description = rsp.json()["Error"]["Description"]
            logger.error("login failed, description:%s" % description)
            raise HCCIException(653086, lb_listen_addr, description)

    def get_admin_change_time(self, lb_listen_addr):
        url = "https://%s/v3/auth/singleuser?Id=admin" % lb_listen_addr
        rsp = self.rest_api.get(url, self.ebk_iam_headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        description = rsp.json()["Error"]["Description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            raise HCCIException(653020)
        if errcode != 0:
            logger.error("get admin pwd change time failed, "
                         "description:%s" % description)
            raise HCCIException(653087, lb_listen_addr, description)
        data = rsp.json()["Data"]
        pwd_change_time = data["Users"][0]["CHANGETIME"]
        return pwd_change_time

    def get_mo_iam_float_ip(self):
        """Get mo_tenent_float_ip for iam.
        :return:
        """
        if_ipv6 = 0
        try:
            project_dic = get_project_conditions(self.project_id)
        except Exception as err:
            logger.error("Get project conditions err:%s" % err)
            raise HCCIException("Get project conditions err:%s" % err) from err

        if "ManageIPV6" in project_dic:
            if_ipv6 = project_dic["ManageIPV6"]

        try:
            tenent_info = ManageOneUtil2().getMoTanentInfo(self.pod_id)
        except Exception as err:
            logger.error("Get Mo tenent info err:%s" % err)
            raise HCCIException("Get Mo tenent info err:%s" % err) from err

        if tenent_info:
            iam_float_ip = tenent_info[0]
            if if_ipv6:
                return "[%s]" % iam_float_ip
            else:
                return iam_float_ip
        else:
            logger.error("Get iam float ip failed.")
            raise Exception("Get iam float ip failed.")

    def build_unify_pwd_body_admin(self, lb_listen_addr1, lb_listen_addr2, operation_type=0):
        sub_com_name = "eBackupServer"
        data = {
            "componentName": "eBackup",
            "subComponents": [
                {
                    "subComponentName": sub_com_name,
                    "createdAccountList": []
                }
            ]
        }
        unify_pwd_url_ip = lb_listen_addr2.split(":")[0]
        unify_pwd_modify_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_MODIY)
        unify_pwd_reset_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_RESET)
        unify_pwd_sync_status_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_QUERYJOB)
        unify_pwd_verify_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_VERIFY)

        quality_item = self.get_admin_pwd_quality(lb_listen_addr1)
        last_time_second = self.get_admin_change_time(lb_listen_addr1)
        last_time = int(last_time_second) * 1000
        expire_time = (int(last_time_second) + int(
            quality_item["passwdChangeMaxBetweenTime"]) * 86400) * 1000

        account_pwd = self.__config_dict.get('eBackup_weblogin_password')
        account_old_pwd = self.__config_dict.get('eBackup_weblogin_password')
        account_dic = {
            "accountName": "admin",
            "region": self.__config_dict.get('region_id'),
            "accountType": 6,
            "accountDescription": ADMIN_ACCOUNT_DESC,
            "passwd": account_pwd,
            "oldPasswd": account_old_pwd,
            "lastPasswdChange": last_time,
            "modifyType": 1,
            "passwdExpires": expire_time,
            "riskMessage": '{"zh-cn":"","en-us":""}',
            "passwdComplexity": quality_item,
            "ip": unify_pwd_url_ip,
            "usedScene": unify_pwd_url_ip,
            "urlModify": unify_pwd_modify_url,
            "urlReset": unify_pwd_reset_url,
            "urlSyncStatus": unify_pwd_sync_status_url,
            "urlValid": unify_pwd_verify_url,
            "operationType": operation_type
        }
        try:
            data.get("subComponents")[0].get("createdAccountList").append(account_dic)
        except Exception as err:
            logger.error(f"Occurred an exception when obtaining the dictionary value. error : {err}")
            raise err

        return data

    def register_unify_pwd(self, oc_address, data, iam_token):
        register_url = "https://%s/rest/mounpwdservice/v1/account" % oc_address
        header = {"Content-Type": "application/json", "X-Auth-Token": iam_token}
        rsp = self.rest_api.post(register_url, data=json.dumps(data), headers=header)
        logger.info("Register unify pass_word result code:%s" % rsp.status_code)
        if rsp.status_code <= 300 and rsp.content:
            rsp_content = json.loads(rsp.content)
        else:
            return True, False
        if rsp_content.get("code") == "00000000":
            logger.info("Register service to mo unify pass_word success, response code:%s" % rsp_content.get("code"))
            return True, True
        return False, True

    def send_unify_pwd_req(self, data):
        iam_token = self.iam_util.get_iam_token(self.iam_util.get_iam_account_info("op_service"))
        oc_address = self.pararms_util.get_oc_domain_name()

        register_retry_times = 0
        logger.info("Register unify pass_word float_ip:%s" % oc_address)
        req_ret = False
        ret = False
        while True:
            try:
                ret, req_ret = self.register_unify_pwd(oc_address, data, iam_token)
            except Exception as err:
                logger.error(f"Register service to mo unify pass_word occur exc, info: {str(err)}")

            if ret:
                return req_ret

            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of retries, so do not try again")
                return False
            time.sleep(30)

    def ebackup_register_unify_pwd_hcp_and_root(self):
        """
        ebackup hcp, root 注册统一密码
        Returns:

        """
        ip_list = \
            self.__config_dict.get('datamover_externalom_iplist').lower()
        password_list = {"hcp": self.hcp_passwd, "root": self.root_pwd}
        region_id = self.__config_dict['region_id']
        datamover_iplist = ip_list.split(',')
        for server_ip in datamover_iplist:
            user_account = ['hcp', 'root']
            for account in user_account:
                ssh_info = SshInfo(server_ip, "hcp", password_list.get("hcp"), password_list.get("root"))
                os_account_info = AccountTool.get_hcp_or_root_time_about(account, ssh_info)
                delete_data = AccountTool.build_unify_pwd_body_os(os_account_info, region_id, account, 2)
                self.send_unify_pwd_req(delete_data)
                data = AccountTool.build_unify_pwd_body_os(os_account_info, region_id, account)
                result = self.send_unify_pwd_req(data)
                if not result:
                    raise HCCIException(653089)

    def update_proxy_config(self):
        """
        更新扩proyx场景下,新增节点的配置。
        Returns:
        """
        ip_list = \
            self.__config_dict.get('datamover_externalom_iplist').lower()
        add_node_list = ip_list.split(',')
        self.update_node_config(add_node_list)

    def update_datamover_config(self):
        """修改datamover部件下的节点配置文件

        Returns:

        """
        ip_list = self.__config_dict[
            'datamover_externalom_iplist'].lower().split(',')

        self.update_node_config(ip_list)

    def update_node_config(self, ip_list):
        """
        修改节点ip
        Args:
            ip_list:

        Returns:

        """

        for node_ip in ip_list:
            cmd = "sed -i 's/^HasRegistedToMo=0/HasRegistedToMo=1/g' " \
                  "/opt/huawei-data-protection/ebackup/microservice" \
                  "/ebk_governance/conf/hcpconf.ini;echo last result $?"
            resultout = self.ssh_obj.ssh_cmds(node_ip, cmd, 'hcp', self.hcp_passwd, self.root_pwd, "", "")
            if 'last result 0' in resultout:
                logger.info("update 'hcpconf.ini' succ")
            pwd_cmd = "sed -i 's/dcredit=.*/dcredit=0 ucredit=0 lcredit=0 ocredit=0/g' /etc/pam.d/system-auth;echo " \
                      "updating the password complexity $?"
            update_result = self.ssh_obj.ssh_cmds(node_ip, pwd_cmd, 'hcp', self.hcp_passwd, self.root_pwd, "", "")
            if 'updating the password complexity 0' in update_result:
                logger.info("Password complexity updated successfully.")
            else:
                raise HCCIException("Failed to update the password complexity.")

    def get_datamover_ips(self, headers, node_ip):
        node_info_url = r'https://%s/rest/dev/vbackup_server' % node_ip
        rsp = self.rest_api.get(node_info_url, headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            raise HCCIException(653019)
        if errcode != 0:
            logger.error("get node info failed,description:" + description)
            raise HCCIException(653029, description)
        logger.info("get node info succ.")
        data = rsp.json()["data"]
        ip_lists = []
        for a_dict in data:
            datamover_ip = a_dict.get("IPADDR")
            ip_lists.append(datamover_ip)
        # 去重
        ip_lists = list(set(ip_lists))
        return ip_lists
