# -*- coding:utf-8 -*-
import time
import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common.util import Utils
from plugins.eBackup.scripts.upgrade.ebackup_upgrade import EbackupUpgrader
from plugins.eBackup.scripts.upgrade.util import CommonConfig

RETRY_TIME_LIMIT = 4


class DatamoverUpgrade(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverUpgrade, self).__init__(project_id, pod_id,
                                               regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])
        self.failed_node_key = f"Datamover_fail_node_{str(regionid_list[0])}"
        self.ssh = Ssh()
        self.failed_ips = []
        self.service_down_ips = []
        self.exception_list = []
        self.hcp_password = self.__db_param_dict["eBackup_hcp_pwd"]
        self.root_password = self.__db_param_dict["eBackup_root_pwd"]

    def check_ebackup_status(self, server_ips):
        check_cmd = "service hcp status>/dev/null;echo execute check_status_cmd result: $?"
        for ebk_ip in server_ips:
            result = self.ssh.ssh_cmds(ebk_ip, check_cmd, "hcp", self.hcp_password, self.root_password, "", "")
            if "execute check_status_cmd result: 0" not in result:
                logger.error(f"datamover node({ebk_ip}) is not running correctly.")
                return False
        return True

    def check_datamover_status(self, host_ips):
        logger.info("Checking service status.")
        time.sleep(60)
        retry_times = 1
        while retry_times < RETRY_TIME_LIMIT:
            status_check = self.check_ebackup_status(host_ips)
            if status_check is False:
                return False
            time.sleep(15)
            retry_times = retry_times + 1
        return True

    def do_execute(self):
        host_info = self.__db_param_dict["eBackup_Datamover_nodes"]
        host_info = host_info.replace(" ", "").split("|")
        Utils.set_value_to_config_file(self.failed_node_key, "")

        logger.info("Begin to upgrade all datamover nodes(%s)" % str(host_info))
        for group in host_info:
            host_ips = group.split(";")
            common_config = CommonConfig(self.__db_param_dict, pod_id=self.pod_id, region_id=self.regionid_list[0],
                                         project_id=self.project_id)
            logger.info("Begin to set kms info on datamover node.")
            common_config.config_kms_info(host_ips, False, scene="upgrade")
            common_config.handle_icagent_process(host_ips, 'disable')
            logger.info("Begin to upgrade datamover nodes:%s", str(host_ips))
            ebackup_upgrade = EbackupUpgrader(host_ips, self.__db_param_dict, "datamover")
            fail_reason = "DMK upgrade fail."
            try:
                is_true = ebackup_upgrade.upgrade()
            except Exception as err:
                fail_reason = str(err)
                is_true = False
            if is_true:
                if self.check_datamover_status(host_ips):
                    logger.info("Upgrade eBackup datamover(%s) successfully." % str(host_ips))
                    continue
                self.service_down_ips.append(str(host_ips))
                fail_reason = "Services are not all running correctly."
            logger.error("Fail to upgrade eBackup datamover(%s).reason: %s" % (str(host_ips), fail_reason))
            self.failed_ips.append(str(host_ips))
            fail_node = Utils.get_value_from_config_file(self.failed_node_key)
            if group not in fail_node:
                if fail_node:
                    fail_node = fail_node + '|' + group
                else:
                    fail_node = group
            Utils.set_value_to_config_file(self.failed_node_key, fail_node)
            self._dealing_with_tmp(host_ips)

        if self.failed_ips:
            self.exception_list.append(HCCIException(650083, str(self.failed_ips)))
            logger.error("Fail to upgrade all eBackup datamover nodes.")
        if self.service_down_ips:
            self.exception_list.append(HCCIException(650084, str(self.service_down_ips)))
            logger.error("Services are not all running correctly.")
        if self.exception_list:
            return Message(500, error_msg_cn='\n'.join([exception.error_msg_cn for exception in self.exception_list]),
                           error_msg_en='\n'.join([exception.error_msg_en for exception in self.exception_list]))
        logger.info("Upgrade all eBackup datamover successfully.")
        Utils.set_value_to_config_file(self.failed_node_key, "")
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.do_execute()
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn="升级Datamover出现异常,请查看升级日志并重试，异常信息：%s" % str(ex),
                           error_msg_en=f"Exception occurs when upgrade Datamover, "
                                        f"please see log and try again, error info:{str(ex)}")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        def do_retry():
            host_group_list = Utils.get_value_from_config_file(self.failed_node_key)
            if host_group_list:
                host_group_list = host_group_list.split('|')
            for group in host_group_list:
                host_ips = group.split(";")

                ebackup_upgrade = EbackupUpgrader(host_ips, self.__db_param_dict, "datamover")
                is_true = ebackup_upgrade.rollback_force()
                if not is_true:
                    logger.error("Rollback %s failed." % str(group))
                    return Message(500, error_msg_cn=f"回退Datamover[{str(group)}]失败，请联系技术支持",
                                   error_msg_en=f"Rollback eBackup Datamover[{str(group)}]failed, "
                                                f"please contact support engineers.")
            return self.execute(project_id, pod_id, regionid_list)

        try:
            return do_retry()
        except HCCIException as ex:
            logger.error("Exception occurs when retry to upgrade eBackup: %s", str(ex))
            return Message(500, ex)
        except Exception as err:
            logger.error("Exception occurs when retry to upgrade eBackup: %s", str(err))
            return Message(500, error_msg_cn=f"重试出现异常，请联系技术支持，异常信息:{str(err)}",
                           error_msg_en=f"Exception occurs when retry to upgrade eBackup, "
                                        f"please contact support engineers, error info：{str(err)}")

    def _dealing_with_tmp(self, server_ips):
        logger.info("Start to deal with the /tmp folder")
        check_cmd = "df -h /tmp"
        for ebk_ip in server_ips:
            ssh_client = self.ssh.ssh_create_client(ebk_ip, "hcp", self.hcp_password)
            self.ssh.ssh_send_command(ssh_client, 'su', 'assword:', 30)
            self.ssh.ssh_send_command(ssh_client, self.root_password, '#', 30)
            result = self.ssh.ssh_exec_command_return(ssh_client, check_cmd)
            if str(result).find('tmpfs') != -1:
                logger.info(f"The /tmp folder on node({ebk_ip}) was not mounted properly. Start to make a copy.")
                copy_tmp_cmd = "cp -r /tmp/upgrade/ /opt/upgrade_bak/"
                self.ssh.ssh_send_command(ssh_client, copy_tmp_cmd, '#', 100)

        logger.info("Successfully dealt with /tmp folder")
