# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.business.dmk_util import DmkApi
from utils.business.project_util import ProjectApi
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh
from utils.common.agent_status import is_autoops_agent_availiable

from plugins.eBackup.common.util import Utils
from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.scripts.upgrade.ebackup_upgrade import EbackupUpgrader
from plugins.eBackup.scripts.upgrade.util import check_access_status
from plugins.eBackup.scripts.upgrade.util import check_versions_compatibility
from plugins.eBackup.scripts.upgrade.util import g_check_item
from plugins.eBackup.scripts.upgrade.util import get_dmk_precheck_result
from plugins.eBackup.scripts.upgrade.util import precheck_ebackup_os
from plugins.eBackup.scripts.upgrade.util import format_check_results


class DatamoverPrecheck(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverPrecheck, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.__dmk_util = DmkApi()
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.region_code = regionid_list[0]

    def precheck_on_opsagent(self, server_ips, check_results):
        logger.info("Begin to precheck OpsAgent on nodes(%s)" % server_ips)
        node_without_agent = []
        check_fail_node = []
        precheck_result = {"ops_agent_status": {}}
        for server_ip in server_ips:
            try:
                is_true = is_autoops_agent_availiable(ip=server_ip, region_code=self.region_code)
                if not is_true:
                    node_without_agent.append(server_ip)
            except Exception as err:
                logger.error("Exception occurs when get Ops Agent precheck result:" + str(err))
                check_fail_node.append(server_ip)
        if node_without_agent:
            logger.error("Pre-check on OpsAgent failed, "
                         f"please manually install OpsAgent on nodes in the list: {str(node_without_agent)}")
            for node in node_without_agent:
                precheck_result.get("ops_agent_status").update(
                    {node: f"Pre-check on Ops Agent failed on node {str(node_without_agent)}"})
        if check_fail_node:
            logger.error("Failed to execute pre-check on OpsAgent, "
                         "please verify the contents of HCC system and the parts of ManageOne, "
                         f"and you can manually check the status of OpsAgent on node({str(check_fail_node)})")
            for node in check_fail_node:
                precheck_result.get("ops_agent_status").update(
                    {node: f"Failed to execute pre-check on Ops Agent on node {str(check_fail_node)}"})
        format_check_results(check_results, precheck_result)

    def do_execute(self):
        self._delete_old_backup_data()
        check_results = []
        server_ips = self.param_dict.get('eBackup_Datamover_nodes')
        server_ips = server_ips.replace(" ", "").replace("|", ";").split(";")
        self.save_ebackup_version(server_ips[0])

        # 1.precheck by precheck_ebackup.sh
        logger.info("Begin to precheck nodes(%s) by HCU firstly." % str(server_ips))
        is_true, error_info = check_versions_compatibility(server_ips, self.param_dict.get('eBackup_hcp_pwd'),
                                                           self.param_dict.get('eBackup_root_pwd'))
        if not is_true:
            logger.error("version is not same.")
            raise HCCIException(650044, error_info)

        ssh_info = SshInfo(
            server_ips, "hcp", self.param_dict.get('eBackup_hcp_pwd'), self.param_dict.get('eBackup_root_pwd'))
        is_true = precheck_ebackup_os(check_results, ssh_info,
                                      self.param_dict.get('update_scene'),
                                      self.param_dict.get('dmk_floatIp'),
                                      self.param_dict.get('eBackup_Version'))
        if not is_true:
            logger.error("Precheck by HCU failed.")
            return Message(500, check_results=check_results)

        # 2.check acess status
        for group in self.param_dict.get('eBackup_Datamover_nodes').split('|'):
            group_ips = group.split(';')
            if not self.precheck_access_status(group_ips, check_results):
                logger.error("Precheck assess status by FCU failed.")
                return Message(500, check_results=check_results)
        key = 'access_status'
        check_result = CheckResult(itemname_ch=g_check_item.get(key, dict()).get('name_cn'),
                                   itemname_en=g_check_item.get(key, dict()).get('name_en'), status="success")
        check_results.append(check_result)

        # 3.precheck by DMK
        self.precheck_on_dmk(server_ips, check_results)

        # 4.precheck whether OpsAgent is installed on eBackup nodes
        self.precheck_on_opsagent(server_ips, check_results)
        return Message(200, check_results=check_results)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.do_execute()
        except HCCIException as err:
            logger.error("Precheck failed: " + str(err))
            return Message(500, err)

        except Exception as err:
            logger.error("Failed to exec precheck ,the reason is %s" % err)
            return Message(500, error_msg_cn="升级前检查出现异常，请查看日志并重试，异常信息:%s" % str(err),
                           error_msg_en="Exception occurs while prechecking,"
                                        "please check the log and try again, "
                                        "error info:%s." % str(err))

    def _delete_old_backup_data(self):
        clear_bak_dir_cmd = "rm -rf /var/ebackup_bak /opt/link_var_ebackup_bak"
        hcp_pwd = self.param_dict.get("eBackup_hcp_pwd")
        root_pwd = self.param_dict.get("eBackup_root_pwd")
        server_ips = self.param_dict.get('eBackup_Datamover_nodes')
        server_ips = server_ips.replace(" ", "").replace("|", ";").split(";")
        for server_ip in server_ips:
            server_ssh_info = SshInfo(server_ip, 'hcp', hcp_pwd, root_pwd)
            ssh_client = Utils.get_ssh_client(server_ssh_info)
            Ssh.ssh_exec_command_return(ssh_client, clear_bak_dir_cmd)
            Utils.close_ssh_clinet(ssh_client)
        logger.info("Clear old back data success.")

    def precheck_access_status(self, server_ips, check_results):
        server_ssh_info = SshInfo(
            server_ips, 'hcp', self.param_dict.get('eBackup_hcp_pwd'), self.param_dict.get('eBackup_root_pwd'))
        web_ip = Utils.find_float_ip(server_ssh_info)
        is_true = check_access_status(check_results, web_ip,
                                      self.param_dict.get('eBackup_admin_pwd'),
                                      1)
        return is_true

    def precheck_on_dmk(self, server_ips, check_results):
        logger.info("Begin to precheck nodes(%s) by DMK." % server_ips)
        ebackup_upgrade = EbackupUpgrader(
            server_ips, self.param_dict, "datamover")
        is_true = ebackup_upgrade.precheck()
        if not is_true:
            logger.error("Precheck by DMK failed.")
            raise HCCIException(650036)

        logger.info("Begin to get DMK precheck result.")
        is_true = get_dmk_precheck_result(
            check_results,
            server_ips,
            self.param_dict.get('eBackup_hcp_pwd'),
            self.param_dict.get('eBackup_root_pwd'),
            self.param_dict.get('eBackup_Version'))
        if not is_true:
            logger.error("Prechecking is unpassed by DMK.")
        logger.info("Precheck nodes(%s) completed." % server_ips)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)

    def save_ebackup_version(self, server_ip):
        def do_save_ebackup_version():
            account_hcp = "hcp"
            account_hcp_passwd = self.param_dict.get('eBackup_hcp_pwd')
            account_root_passwd = self.param_dict.get('eBackup_root_pwd')
            version_check_cmds = "showsys  | grep -w \"Product " \
                                 "Version\" | awk -F '|' '{print $2}' " \
                                 "|sed s/[[:space:]]//g"
            ssh = Ssh()
            ssh_client = ssh.ssh_create_client(server_ip, account_hcp,
                                               account_hcp_passwd)
            ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, account_root_passwd, '#', 100)
            result = ssh.ssh_exec_command_return_list(ssh_client,
                                                      version_check_cmds)
            ssh.ssh_close(ssh_client)
            origin_version = result[0].strip().replace('\n', '')
            logger.info("save eBackup version is %s" % origin_version)
            projectapi = ProjectApi()
            projectapi.save_version_info(self.project_id, "eBackup",
                                         {"Originalversion": origin_version,
                                          "Targetversion":
                                              self.param_dict["eBackup_Version"]})
        try:
            do_save_ebackup_version()
        except Exception as err:
            raise err
