# -*- coding:utf-8 -*-
import base64
import ipaddress
import json
import os
import re
import socket
import subprocess
import time

import requests
import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.service_monitor.service_monitor import ServiceMonitorCfg
from utils.business.service_monitor.service_monitor import ServiceNodeCfg
from utils.business.service_monitor.service_monitor import set_service_m_cfg
from utils.client.webUI_client import g_webui_client_ctrl
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.ssh_util2 import Ssh

from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.iam_util import IamUtil
from plugins.eBackup.common.util import Utils

SYS_VERSION_CMD = '''grep "System Version" /opt/huawei-data-''' \
                  '''protection/ebackup/conf/versions.conf''' \
                  ''' | awk -F "=" '{print $2}' '''
TOOL_REMOVER_CMD = '''sed -i '/SecurityTool/ d' ~/.bash_history >/dev/null 2>&1;''' \
                   '''sed -i '/SecurityTool/ d' /var/log/messages >/dev/null 2>&1;''' \
                   '''sed -i '/OpSvcPassword/ d' ~/.bash_history;sed -i '/OpSvcPassword/ d' ''' \
                   '''/var/log/messages >/dev/null 2>&1;history -c >/dev/null 2>&1'''
g_check_item = {
    "cpu_health": {"name_cn": "CPU健康度", "name_en": "CPU Health",
                   "error_code": '650026'},
    "disk_health": {"name_cn": "硬盘健康度", "name_en": "Disk Health",
                    "error_code": '650027'},
    "disk_space": {"name_cn": "硬盘空闲空间", "name_en": "Free Disk Space",
                   "error_code": '650003'},
    "network_health": {"name_cn": "网络健康度", "name_en": "Network Health",
                       "error_code": '650028'},
    "service_health": {"name_cn": "服务健康度", "name_en": "Service Health",
                       "error_code": '650004'},
    "ha_health": {"name_cn": "HA健康度", "name_en": "HA Health",
                  "error_code": '650029'},
    "micro_service_health": {"name_cn": "微服务健康度",
                             "name_en": "Micro Service Health",
                             "error_code": '650030'},
    "version_compatibility": {"name_cn": "兼容性",
                              "name_en": "Micro Service Health",
                              "error_code": '650019'},
    "gaussdb_health": {"name_cn": "数据库健康度", "name_en": "Database Health",
                       "error_code": '650040'},
    "alarm_reporting": {"name_cn": "对接OC告警", "name_en": "Connect OC alarm",
                        "error_code": '650031'},
    "access_status": {"name_cn": "可访问状态", "name_en": "Accessibility Status",
                      "error_code": '650033'},
    "os_upgrade_precheck_status": {"name_cn": "OS可升级检查", "name_en": "Check whether the OS can be upgraded.",
                                   "error_code": '650078'},
    "ops_agent_status": {"name_cn": "OpsAgent状态检查", "name_en": "OpsAgent status",
                         "error_code": '650085'}
}

MAXFILESIZE = 1024 * 1024 * 1024 * 5


def get_reverse_proxy_file(current_region_id, reverse_proxy_ip):
    return os.path.realpath(
        __file__ + '/../' + '%s' + '_' + '%s' + '_cascaded_ips.txt') % \
           (current_region_id, reverse_proxy_ip)


def parse_rest_response(result):
    def do_parse_rest_response():
        rsp_dict = {}
        for index, val in enumerate(result):
            if -1 != str(val).find("Data"):
                rsp_dict = json.loads(result[index].replace('\n', ''))
                break
        if not rsp_dict:
            logger.error("There is no Data in response result,"
                         "the response is " + str(result))
            raise Exception(
                "There is no Data in response result,the "
                "response is " + str(result))

        if rsp_dict.get("Error", dict()).get("Code") != 0:
            raise Exception("request failed,the error code is" + rsp_dict.get("Error", dict()).get("Code"))
        return rsp_dict

    try:
        return do_parse_rest_response()
    except Exception as err:
        logger.error("request failed, the result is %s" % str(result))
        raise Exception("Convert result to dict failed.") from err


def parse_driver_node_info(result):
    return_server_dict = {"eBackupDriver": [], "cpsnode": []}
    driver_node_dict = {}
    cps_server_dict = {}

    def do_parse_res(val):
        nonlocal cps_server_dict, driver_node_dict
        if -1 != val.find("cpsnode"):
            cps_server_dict = json.loads(val)
            return True
        if -1 != val.find("eBackupDriver"):
            driver_node_dict = json.loads(val)
            return True
        return False

    for res in result:
        try:
            if do_parse_res(res):
                continue
        except Exception as err:
            logger.error("Convert string to dict failed.The reason is:%s." % str(err))
            raise err
    return_server_dict.update(cps_server_dict)
    return_server_dict.update(driver_node_dict)
    logger.info(f"eBackupDriver:{return_server_dict.get('eBackupDriver')}")
    logger.info(f"CpsServer:{return_server_dict.get('cpsnode')}")
    logger.info("Query eBackup driver nodes successfully.")
    return return_server_dict


def get_ebackup_driver_nodes(reverse_proxy_ip, db_param_dict):
    def do_get_ebackup_driver_nodes():
        file_path = os.path.realpath(__file__ + '/../../../shell_tool/query_driver_nodes.py')
        openstack_fsp_pwd = db_param_dict.get('openstack_fsp_pwd')
        openstack_root_pwd = db_param_dict.get('openstack_root_pwd')
        cps_admin_pwd_unicode = db_param_dict.get('openstack_cps_admin_pwd')
        cps_admin_pwd = base64.b64encode(cps_admin_pwd_unicode.encode("utf-8")).decode(encoding="utf-8")
        ssh = Ssh()
        ret = ssh.put_file(reverse_proxy_ip, "fsp", openstack_fsp_pwd, file_path, "/home/fsp/")
        if not ret:
            logger.error("Upload %s to %s failed." % (file_path, reverse_proxy_ip))
            raise HCCIException(650007, reverse_proxy_ip, '/home/fsp', "fsp")
        logger.info("Upload queryDriverNode.py to %s successfully." % reverse_proxy_ip)
        fsp_ssh_info = SshInfo(reverse_proxy_ip, "fsp", openstack_fsp_pwd, openstack_root_pwd)
        ssh_client = Utils.get_ssh_client(fsp_ssh_info)
        Ssh.ssh_send_command(ssh_client, "cd /home/fsp", "#", 300)
        Ssh.ssh_send_command(ssh_client, "chmod +x query_driver_nodes.py", "#", 300)
        Ssh.ssh_send_command(ssh_client, "python query_driver_nodes.py cps_admin upgrade && echo 'Success'",
                             "CPS_PASSWORD:", 300)
        result = Ssh.ssh_send_command(ssh_client, cps_admin_pwd, "Success", 300)
        Ssh.ssh_send_command(ssh_client, "rm -f /home/fsp/query_driver_nodes.py", "#", 300)

        clear_cmd = '''sed -i '/query_driver_nodes.py/ d' ~/.bash_history > /dev/null 2>&1;''' \
                    '''history -c >/dev/null 2>&1;''' \
                    '''sed -i '/query_driver_nodes.py/ d' /var/log/messages >/dev/null 2>&1'''
        Ssh.ssh_send_command(ssh_client, clear_cmd, "#", 300)
        Utils.close_ssh_clinet(ssh_client)
        if -1 != str(result).find("Unsupported-eBackup"):
            logger.info("driver nodes is empty.")
            return {}
        if -1 == str(result).find("Success"):
            logger.error("Query eBackup driver nodes failed,the reason is:%s" % str(result))
            raise Exception(str(result))
        return parse_driver_node_info(result)

    try:
        return do_get_ebackup_driver_nodes()
    except Exception as err:
        logger.error(err)
        raise err


def do_update_driver_cert_path(driver_ip, db_param_dict):
    openstack_fsp_pwd = db_param_dict["openstack_fsp_pwd"]
    openstack_root_pwd = db_param_dict["openstack_root_pwd"]
    cps_admin_pwd = db_param_dict["openstack_cps_admin_pwd"]
    logger.info("begin to update cert content")
    client = Ssh.ssh_create_client(driver_ip, "fsp", openstack_fsp_pwd)
    logger.info("finish create SSH client use fsp user.")
    Ssh.ssh_send_command(client, "su - root", "Password:", 30, retry_times=3)
    Ssh.ssh_send_command(client, openstack_root_pwd, "#", 30)
    logger.info("finish login to root on %s node." % driver_ip)
    python37_path = "/usr/lib/python3.7/site-packages/cinder/backup/ebackupconfig.py"
    python_path = "/usr/lib/python/site-packages/cinder/backup/ebackupconfig.py"
    get_ebackupconfig_path_cmd = f"grep update_ebk_cert {python37_path}; echo \"python3 $?\"; " \
                                 f"grep update_ebk_cert {python_path}; echo \"python $?\""
    # 由于HCS821底座python版本升级，并且兼容备份单升（python版本未升级）场景，需做python or python3.7路径的判断
    Ssh.ssh_send_command(client, "TMOUT=0", "#", 300)
    resp = Ssh.ssh_send_command(client, get_ebackupconfig_path_cmd, "#", 300)
    if str(resp).find("python 0") != -1:
        py_path = python_path
    elif str(resp).find("python3 0") != -1:
        py_path = python37_path
    else:
        logger.error(f"Failed to find ebackupconfig.py path, stdout: {str(resp)}.")
        raise HCCIException(650074, str(resp))

    Ssh.ssh_send_command(client, f"python {py_path} update_ebk_cert", "CPS_USERNAME=", 300)
    Ssh.ssh_send_command(client, "cps_admin", "CPS_PASSWORD", 300)
    resp = Ssh.ssh_send_command(client, cps_admin_pwd, "#", 300)
    if str(resp).find("configuration eBackup driver finish.") == -1:
        logger.error(f"Failed to config Drvier cert, stdout: {str(resp)}.")
        raise HCCIException(650073, driver_ip)
    logger.info("Success to update ebackup driver cert config.")
    return client


def update_driver_cert_path(driver_ip, db_param_dict):
    """
    更新cinder证书路径
    :param reverse_proxy_ip: 反向代理IP
    :param db_param_dict: 密码参数
    """
    ssh_client = None
    try:
        ssh_client = do_update_driver_cert_path(driver_ip, db_param_dict)
    except Exception as ex:
        raise HCCIException(650052, repr(ex)) from ex
    finally:
        # 校验完毕关闭ssh连接
        if ssh_client:
            Ssh.ssh_close(ssh_client)
            logger.info("OpenStack SSH client is closed")


def get_cascading_fsp_web_client(reverse_ip, port, web_passwd):
    if reverse_ip is None or web_passwd is None:
        raise Exception("openstack_reverse_proxy_ip or "
                        "openstack_cpsweb_pwd can not be None.")
    auth_provider = {
        "ip": reverse_ip,
        "port": port,
        "user_name": "admin",
        "pwd": web_passwd
    }
    fsp_web_info = (auth_provider, "webUI_rest")
    return g_webui_client_ctrl.apply_client(fsp_web_info, None)


def get_reverse_ip(rest_client, cascaded_region_name):
    logger.info('start get_reverse_ip %s ' % cascaded_region_name)
    url = "api/v1/config/domain"
    cmd = {'body': None, 'method': 'GET', 'uri': url}
    ret = rest_client.exec_cmd(cmd)
    if ret[0] != 200:
        logger.error("Request to get cascaded reverse proxy ip failed,"
                     "status code is %s." % str(ret[0]))
        return ''
    result = ret[2]
    if 'dns_address' in result and result.get('dns_address'):
        logger.info('The dns_address are:%s' % str(result.get('dns_address')))
        for dns_address in result.get('dns_address'):
            if cascaded_region_name in dns_address.get('domain'):
                return dns_address.get('ip')
    return ''


class EbackupConfig(object):
    def __init__(self, hosts, db_param_dict, node_type, action="upgrade"):
        logger.init("eBackup config")
        self.__db_param_dict = db_param_dict
        self.__node_type = node_type
        self.__action = action
        self.__hosts = hosts

    def get_config(self):
        logger.info("dmk: enter get dmk config step.")
        # 获取安装时候的配置模板信息

        if self.__action == "config":
            return self._config_dmk_config()
        elif self.__action == "upgrade":
            return self._upgrade_dmk_config()
        elif self.__action == "precheck":
            return self._precheck_dmk_config()
        # 装备config块

        logger.info("dmk: compute dmk params failed.")
        return "---", "---"

    def _upgrade_dmk_config(self):
        config_val = "---"
        if self.__node_type == "datamover":
            config_val = "---"

        host_val = "[eBackupService]\n"
        for host in self.__hosts:
            host_val += host + "\n"
        return config_val, host_val

    def _precheck_dmk_config(self):
        config_val = "no_need_backup: 0"

        host_val = "[eBackupService]\n"
        for host in self.__hosts:
            host_val += host + "\n"
        return config_val, host_val

    def _config_dmk_config(self):
        config_val = "---"
        if self.__node_type == "datamover":
            config_val = "---"

        host_val = "[eBackupService]\n"
        for host in self.__hosts:
            host_val += host + "\n"
        return config_val, host_val


class DriverConfig(object):
    def __init__(self, hosts_dict, param_dict, one2all=0, action="upgrade"):
        logger.init("eBackup driver config")
        self.__db_param_dict = param_dict
        self.__action = action
        self.__hosts_dict = hosts_dict
        self.__one2all = one2all

    def get_config(self):
        logger.info("dmk: enter get dmk config step.")
        # 获取安装时候的配置模板信息

        if self.__action == "config":
            return self._config_dmk_config()
        elif self.__action == "upgrade":
            return self._upgrade_dmk_config()
        # 装备config块

        logger.info("dmk: compute dmk params failed.")
        return "---", "---"

    def _upgrade_dmk_config(self):
        cps_user_pwd = self.__db_param_dict.get('openstack_cps_admin_pwd')
        config_val = "CpsUsername: " + "cps_admin\n" + "CpsPasswd: " + \
                     cps_user_pwd + "\n" + "DriverUpgradeNode_" \
                                           "one2all: %s\n" % self.__one2all

        host_val = ""
        if self.__one2all == 1:
            host_val += "[eBackupDriver]\n"
            host_val += self.__hosts_dict["eBackupDriver"][0]
        else:
            if 0 != len(self.__hosts_dict["cpsnode"]):
                host_val += "[cpsnode]\n"
                for host_ip in self.__hosts_dict["cpsnode"]:
                    host_val += host_ip + "\n"
            if 0 != len(self.__hosts_dict["eBackupDriver"]):
                host_val += "[eBackupDriver]\n"
                for host_ip in self.__hosts_dict["eBackupDriver"]:
                    host_val += host_ip + "\n"

        return config_val, host_val

    def _config_dmk_config(self):
        cps_user_pwd = self.__db_param_dict.get('openstack_cps_admin_pwd')
        config_val = "CpsUsername: " + "cps_admin\n" + "CpsPasswd: " + \
                     cps_user_pwd + "\n" + "DriverUpgradeNode_one2all: 0\n"
        host_val = "[eBackupDriver]\n"

        host_val += self.__hosts_dict["eBackupDriver"][0]
        return config_val, host_val


class CommonConfig(object):
    def __init__(self, param_dict, pod_id=None, region_id=None, project_id=None):
        self.param_dict = param_dict
        self.pod_id = pod_id
        self.region_id = region_id
        self.project_id = project_id
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.rest_api = EbackupRest()
        self.manageone = ManageOneUtil2()
        self.mo_domain_list = self.get_mo_domain_name_list(self.project_id)
        self.config_dict = ParamUtil().get_service_cloud_param(pod_id, "eBackup")

    def get_cert_url(self, float_ip):
        ip_version = Utils.check_ip_version(float_ip)
        if -1 != Utils.check_ip_version(self.param_dict["ManageOne_global_domain_name"]):
            silvan_url = self.param_dict["ManageOne_global_domain_name"]
        else:
            silvan_url = "console-silvan." + self.param_dict["ManageOne_global_domain_name"]
        if ip_version == 4:
            cert_url = "https://%s:%s" % (silvan_url, "26335")
        else:
            cert_url = "https://[%s]:%s" % (silvan_url, "26335")
        return cert_url

    def ha_sync_file(self, ip_list):
        logger.info("start to sync file.")
        cmd = "export LD_LIBRARY_PATH=/opt/huawei-data-protection/ebackup" \
              "/libs/ && /opt/huawei-data-protection/ebackup/ha/module/" \
              "hacom/tools/ha_client_tool --syncallfile"
        account_hcp = "hcp"
        account_hcp_passwd = self.param_dict.get('eBackup_hcp_pwd')
        account_root_passwd = self.param_dict.get('eBackup_root_pwd')
        for ha_ip in ip_list:
            ssh_client = Ssh.ssh_create_client(ha_ip, account_hcp, account_hcp_passwd)
            try:
                Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
                Ssh.ssh_send_command(ssh_client, account_root_passwd, '#', 100)
                Ssh.ssh_send_command(ssh_client, cmd, '#', 100)
            finally:
                Ssh.ssh_close(ssh_client)

    def get_config_kms_info_cmd(self, raise_flag, scene, iam_account, iam_pwd):
        db_opt = ParamUtil()
        mo_public_param_list = db_opt.get_service_cloud_param(self.project_id, "ManageOne_public_params",
                                                              self.region_id)
        mo_global_domain_name = mo_public_param_list.get("ManageOne_global_domain_name", None)
        if not mo_global_domain_name:
            logger.error("get global domain name from mo failed")
            if raise_flag:
                raise HCCIException(650053)
            return ''
        cmd = "(iam_conf=/opt/huawei-data-protection/ebackup/microservice/ebk_iam/conf/hcpconf.ini;iam_passwd=%s && " \
              "(sed -i '/OpSvcPassword/d;/Regionid/d;/GlobalDomainName/d;" \
              "/OpSvcName/d' ${iam_conf};gen_line=`sed -n '/\[General\]/=' ${iam_conf}`;" \
              "sed -i \"${gen_line}c \[General\]\\nRegionid=%s\\nGlobalDomainName=%s\\nOpSvcName=%s\\n" \
              "OpSvcPassword=${iam_passwd}\\n\" ${iam_conf})) && echo 'config kms info success'" % (
                  iam_pwd, self.region_id, mo_global_domain_name, iam_account.account_name)
        if scene == "upgrade":
            cmd = "(KMS_conf=/tmp/upgrade/KMS_CONF;iam_passwd=%s && " \
                  "(echo -e \"Regionid=%s\\nGlobalDomainName=%s\\nOpSvcName=%s\\n" \
                  "OpSvcPassword=${iam_passwd}\\n\" > ${KMS_conf})) && echo 'config kms info success'" % (
                      iam_pwd, self.region_id, mo_global_domain_name, iam_account.account_name)
        return cmd

    def config_kms_info(self, ip_list, raise_flag, scene="change"):
        """  设置 kms 信息
            ip_list  需要设置信息的ip列表
            raise_flag  是否需要在出错时抛出异常，避免堵塞升级流程，算法切换流程需抛异常
        """
        iam_account = self.iam_util.get_iam_account_info("op_svc_csbs")
        cmd1 = "export LD_LIBRARY_PATH=/opt/huawei-data-protection/ebackup/libs/:" \
               "/opt/huawei-data-protection/ebackup/microservice/ebk_iam/lib/:${LD_LIBRARY_PATH};"
        get_pwd_cmd = "/opt/huawei-data-protection/ebackup/microservice/ebk_iam/bin/SecurityTool" \
                      " enc_normal_pass_ms ebk_iam"
        account_hcp = "hcp"
        account_hcp_passwd = self.param_dict.get('eBackup_hcp_pwd')
        account_root_passwd = self.param_dict.get('eBackup_root_pwd')
        for host_ip in ip_list:
            ssh_client = None
            try:
                ssh_client = Ssh.ssh_create_client(host_ip, account_hcp, account_hcp_passwd)
                Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
                Ssh.ssh_send_command(ssh_client, account_root_passwd, '#', 100)
                Ssh.ssh_send_command(ssh_client, cmd1, '#', 100)
                Ssh.ssh_send_command(ssh_client, get_pwd_cmd, 'Enter password:', 100)
                Ssh.ssh_send_command(ssh_client, iam_account.account_pwd, 'Enter password again:', 100)
                iam_pwd = Ssh.ssh_send_command(ssh_client, iam_account.account_pwd, '#', 100)[1].rstrip('\n')
                cmd = self.get_config_kms_info_cmd(raise_flag, scene, iam_account, iam_pwd)
                if not cmd:
                    return
                result = Ssh.ssh_send_command(ssh_client, cmd, '#', 100)[1].rstrip('\n')
                Ssh.ssh_send_command(ssh_client, TOOL_REMOVER_CMD, '#', 100)
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)
            if result.find("config kms info success") == -1:
                logger.error("Failed to config kms info on node %s. stdout: %s.", host_ip, str(result))
                if raise_flag:
                    raise HCCIException(650054, host_ip)
                continue
            logger.info("config kms info success on node %s", host_ip)

    def handle_icagent_process(self, node_ips, action):
        """  处理icagent进程
            升级到HCS821时由于root脚本路径改变需要在升级前将ICAgent脚本禁用，防止root用户锁定
            ssh_info  需要禁用的节点ssh信息
            action  禁用还是恢复
        """
        logger.info("Enter handle icagent process func.")
        icagent_scripts = "/home/moicagent_start/bin/manual/mstart.sh"
        stop_agent_scripts = "/home/moicagent_start/bin/manual/mstop.sh"
        disable_icagent_cmd = f"[ -f '{icagent_scripts}' ] && mv -f '{icagent_scripts}' '{icagent_scripts}.bak' "
        enable_icagent_cmd = f"[ -f '{icagent_scripts}.bak' ] && mv -f '{icagent_scripts}.bak' '{icagent_scripts}' "
        stop_icagent_process = f"[ -f '{stop_agent_scripts}' ] && (sh {stop_agent_scripts};echo 'exec shell res:$?')"

        account_hcp = "hcp"
        account_hcp_passwd = self.param_dict.get('eBackup_hcp_pwd')
        account_root_passwd = self.param_dict.get('eBackup_root_pwd')
        ssh_info = SshInfo(node_ips, account_hcp, account_hcp_passwd, account_root_passwd)

        for datamover_ip in ssh_info.node_ips:
            ssh_client = Utils.get_ssh_client(ssh_info, login_ip=datamover_ip)
            if action == "disable":
                logger.info(f"Begin disable icagent process on node {datamover_ip}")
                Ssh.ssh_send_command(ssh_client, disable_icagent_cmd, "#", 100)
                result = Ssh.ssh_send_command(ssh_client, stop_icagent_process, "#", 100)
                if str(result).find("exec shell res:1") != -1:
                    logger.error(f"Stop MOIcAgent process failed. stdout: {result}")
                    raise HCCIException(650072, datamover_ip)
            elif action == "enable":
                # 将脚本移回去后会进行自动拉起，此处无需手动拉起。
                Ssh.ssh_send_command(ssh_client, enable_icagent_cmd, "#", 100)
            Utils.close_ssh_clinet(ssh_client)
        logger.info("Handle icagent process success.")

    def config_service_monitor(self):
        datamover_ip = self.param_dict["eBackup_Datamover_nodes"].replace("|", ";").split(";")
        hosts = []
        templates = ["eBackup_server_template_01",
                     "eBackup_server_template_02", "os_template"]
        for node_ip in datamover_ip:
            for template_id in templates:
                ebackup_node = ServiceNodeCfg(node_ip, template_id, "eBackup_log")
                hosts.append(ebackup_node)
        service_m_cfg = ServiceMonitorCfg("eBackup",
                                          "eBackup",
                                          self.param_dict["region_id"],
                                          "eBackup",
                                          "IAAS",
                                          hosts)
        return not set_service_m_cfg(self.pod_id, service_m_cfg)

    @staticmethod
    def get_lb_float_ip(ssh, ssh_client):
        get_lb_float = '''lb_address=`grep -w LoadbalanceAddress ''' \
                       '''/opt/huawei-data-protection/ebackup/conf/''' \
                       '''hcpconf.ini | awk -F "=" '{print $2}'`; ''' \
                       '''echo $lb_address'''
        lb_address = ssh.ssh_exec_command_return_list(ssh_client,
                                                      get_lb_float)
        logger.info("Get lb address:" + str(lb_address))
        lb_float_ip = lb_address[0].strip().replace("\n", "")
        return lb_float_ip

    def config_alarm_info(self, float_ip):
        token = ''
        session = ''

        def do_config_alarm_info():
            nonlocal token, session
            logger.info("Begin to config alarm info at host:[%s]" % float_ip)
            oc_domain_name = self.param_dict["ManageOne_oc_external_global_domain_name"]
            if oc_domain_name is None or 0 == len(oc_domain_name) or -1 != Utils.check_ip_version(oc_domain_name):
                logger.info("ManageOne_oc_external_global_domain_name is null or ip address,skip to config alram info.")
                return True
            ip_version = Utils.check_ip_version(float_ip)
            if ip_version == 4:
                url = "https://%s:8088/rest/dev/alarm_settings/default" % float_ip
            else:
                url = "https://[%s]:8088/rest/dev/alarm_settings/default" % float_ip

            token_tmp, session_tmp = Utils.login(float_ip, self.param_dict["eBackup_admin_pwd"], ip_version)
            if token_tmp and session_tmp:
                token = token_tmp
                session = session_tmp
            header = {'Accept': 'application/json;version=2.2;charset=UTF-8',
                      'iBaseToken': token, 'Accept-Language': 'en',
                      'Cookie': 'language=en; %s;DEVICE_ID=dev;sessionIdleTime=60000'
                                ';MACHINEROLE=2;CSRF_IBASE_TOKEN=%s' % (session, token)}
            rsp = requests.get(url=url, headers=header, verify=False)
            self.handle_alarm_response(rsp, float_ip, url, header)
            logger.info("Config alarm info successfully.")
            return True

        try:
            res = do_config_alarm_info()
        except Exception as err:
            logger.error("Config alarm info failed.The reason is: " + str(err))
            return False
        finally:
            Utils.logout(float_ip, token, session, 0)
        return res

    def handle_alarm_response(self, rsp, float_ip, url, header):
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            error_code = rsp_dict["error"]["code"]
            if error_code is None:
                logger.error("errcode is empty.")
                raise Exception("errcode is empty.")
            if error_code != 0:
                logger.error(
                    "Query alarm info failed on [%s],the description is:"
                    "%s" % (float_ip, rsp_dict["error"]["description"]))
                raise Exception(rsp_dict["error"]["description"])

            if "ALARMSERVERURL" not in rsp_dict["data"] or 0 == \
                    len(rsp_dict["data"]["ALARMSERVERURL"]):
                logger.error("No alarm report is not configured.,"
                             "the step will be skipped.")
                return True
            oc_port = rsp_dict["data"]["ALARMSERVERURL"].split(":")[1]
            if oc_port is None or len(oc_port) == 0:
                logger.error("oc_port is empty.")
                raise Exception("oc_port is empty.")
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        if -1 == Utils.check_ip_version(
                rsp_dict["data"]["ALARMSERVERURL"].split(":")[0]):
            logger.info("The oc alarm is domain,no need to update.")
            return True
        data = '{"ALARMSERVERURL": "oc.%s:%s"}' % \
               (self.param_dict["ManageOne_oc_external_global_domain_name"],
                oc_port)
        rsp = requests.put(url=url, headers=header, data=data, verify=False)
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            error_code = rsp_dict["error"]["code"]
            if error_code is None:
                logger.error("errcode is empty.")
                raise Exception("errcode is empty.")
            if error_code != 0:
                logger.error(
                    "Config alarm info failed on [%s],the description "
                    "is:%s" % (float_ip, rsp_dict["error"]["description"]))
                raise Exception(rsp_dict["error"]["description"])
            return ''
        else:
            logger.error(
                "Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")

    def handle_response(self, rsp, float_ip, url, header):
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            error_code = rsp_dict["error"]["code"]
            if error_code is None:
                logger.error("errcode is empty.")
                raise Exception("errcode is empty.")
            if error_code != 0:
                logger.error(
                    "Query operation_log_reporting info failed on [%s],"
                    "the description is:%s"
                    % (float_ip, rsp_dict["error"]["description"]))
                raise Exception(rsp_dict["error"]["description"])
            if "SERVERURL" not in rsp_dict["data"] or \
                    len(rsp_dict["data"]["SERVERURL"]) == 0:
                logger.error("No operation log report is configured,"
                             "the step will be skipped.")
                return True
            lvs_port = rsp_dict["data"]["SERVERURL"].split(":")[1]
            if lvs_port is None or len(lvs_port) == 0:
                logger.error("lvs_port is empty.")
                raise Exception("lvs_port is empty.")
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        data = '{"ISON":"1","SERVERURL":"mo-lvs.%s:%s"}' \
               % (self.param_dict["ManageOne_global_domain_name"], lvs_port)
        rsp = requests.put(url=url, headers=header, data=data, verify=False)
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            error_code = rsp_dict["error"]["code"]
            if error_code is None:
                logger.error("errcode is empty.")
                raise Exception("errcode is empty.")
            if error_code != 0:
                logger.error(
                    "Config operation_log_reporting info failed on "
                    "[%s],the description is:%s"
                    % (float_ip, rsp_dict["error"]["description"]))
                raise Exception(rsp_dict["error"]["description"])
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        logger.info("Config operation_log_reporting info successfully.")
        return True

    def check_string_param(self, name, max_len=255, min_len=1,
                           expr=r"^[a-zA-Z0-9.\-_]+$", allow_null=False):
        if not name:
            return allow_null
        if len(name) < min_len or len(name) > max_len:
            return False
        pattern = re.compile(expr)
        if not re.match(pattern, name):
            return False
        return True

    def check_url_param(self, endpoint, max_len=255, min_len=1, allow_null=False):
        if not self.check_string_param(endpoint, max_len, min_len,
                                       expr=r"^[a-zA-Z0-9./\-_:=?#%]+$",
                                       allow_null=allow_null):
            raise Exception("Failed to check url param.")

    def check_ip(self, node_ip):
        if not isinstance(node_ip, (str, int)):
            return False
        try:
            ipaddress.ip_address(node_ip)
        except Exception as err:
            logger.info(f"check ip failed, {err}.")
            return False
        return True

    def no_limit_8090_iptables(self, ip_list, float_ip):
        cmd = "(ifconfig | grep -w '%s') && ((iptables -vnL |grep  -w " \
              "'tcp dpt:8090' |grep -v veth |" \
              "grep 0.0.0.0) || (DEST_SEGMENT=" \
              "`iptables -vnL |grep -v veth| grep  -w 'tcp dpt:8090' |" \
              " awk '{print $9}' " \
              "|uniq | head -n 1`;iptables -I INPUT  -p tcp -s 0.0.0.0/0 -d " \
              "$DEST_SEGMENT --dport 8090 -j ACCEPT))" % float_ip
        logger.info("the cmd is %s." % cmd)
        for ebk_ip in ip_list:
            Ssh.ssh_cmds(
                ebk_ip, cmd, "hcp", self.param_dict["eBackup_hcp_pwd"],
                self.param_dict["eBackup_root_pwd"], "", "")

    def limit_8090_iptables(self, ip_list, float_ip):
        cmd = "(ifconfig | grep -w '%s') && ((iptables -vnL " \
              "|grep -w 'tcp dpt:8090' " \
              "| grep -v veth | grep 0.0.0.0) && (DEST_SEGMENT=" \
              "`iptables -vnL |grep  -w 'tcp dpt:8090' | awk '{print $9}' " \
              "|uniq |head -n 1`;iptables -D INPUT  -p tcp -s 0.0.0.0/0 -d " \
              "$DEST_SEGMENT --dport 8090 -j ACCEPT))" % float_ip
        for ebk_ip in ip_list:
            Ssh.ssh_cmds(
                ebk_ip, cmd, "hcp", self.param_dict["eBackup_hcp_pwd"],
                self.param_dict["eBackup_root_pwd"], "", "")

    def get_mo_domain_name_list(self, project_id):
        '''
         返回当前region所对应的oc，sc ,以及IAM（线下IAM）所对应内部域名（oc域名是内部域名，非port域名）,
         region0_id和global_domain_name参数需要各自依赖出来
        :param pod_id:
        :return:
        '''
        region_id = self.param_dict["current_region_id"]
        global_domain_name = BaseOps().get_user_input_cloud_param_by_key(project_id, "ManageOne_global_domain_name")
        oc_domain = "oc.{}.{}".format(region_id, global_domain_name)
        sc_domain = "sc.{}".format(global_domain_name)
        iam_domain = "iam-cache-proxy.{}.{}".format(region_id, global_domain_name)
        return [oc_domain, sc_domain, iam_domain]

    def get_iam_domain_name(self):
        iam_domain = self.mo_domain_list[2] + ":26335"
        if not iam_domain:
            raise Exception("Failed to obtain iam domain.")
        logger.info("Succeed to obtain iam domain, "
                    f"the iam domain is: [{iam_domain}].")

        self.check_url_param(iam_domain)
        return iam_domain

    def get_oc_domain_name(self):
        oc_domain = self.mo_domain_list[0] + ":26335"
        if not oc_domain:
            raise Exception("Failed to obtain oc domain.")
        logger.info("Succeed to obtain oc domain, "
                    f"the oc domain is: [{oc_domain}].")

        self.check_url_param(oc_domain)
        return oc_domain

    def logout_ebk_iam(self, iam_addr, ebk_iam_header):
        url = "https://%s/v3/auth/tokens" % iam_addr
        self.rest_api.delete(url, ebk_iam_header)
        logger.info("login out ebk_iam succ.")

    def login_ebk_iam(self, iam_addr):
        logger.info("begin to login ebk_iam micro service.")
        url = "https://%s/v3/auth/tokens" % iam_addr
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"auth":{"identity":{"methods":["password"],"password":' \
               '{"user":{"name":"admin","password":"%s","iam_type":0}}}}}' % \
               self.config_dict.get('eBackup_admin_pwd')
        rsp = self.rest_api.post(url, headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            token = rsp.json()["Data"]["token"]
            ebk_iam_headers = {"Accept": "application/json", "X-Auth-Token": token}
            logger.info("login ebk_iam micro service succ.")
            return ebk_iam_headers
        else:
            description = rsp.json()["Error"]["Description"]
            logger.error("login failed, description:%s" % description)
            raise HCCIException(653086, url, description)

    def get_region_name_by_region_id(self, region_id):
        region_info_list = ManageOneCmdbUtil(self.project_id).get_region_info(region_id)
        region_name = None
        for region_info in region_info_list:
            if region_info.get("regionCode") == region_id:
                region_name = region_info.get("name")
                break
        if not region_name:
            raise Exception("Failed to obtain region name.")
        return region_name


def get_ebackup_driver_nodes_from_file(file_path):
    if not os.path.exists(file_path):
        logger.error("There is no %s existing." % file_path)
        return {}
    driver_host_dict = {}
    with open(file_path, mode='r') as file_point:
        driver_host_list = file_point.readlines()
    if 2 != len(driver_host_list):
        logger.error(
            "There is no cps-server node or eBackup "
            "driver node in:%s" % str(file_path))
        raise Exception(
            "There is no cps-server node or eBackup"
            "driver node in:%s" % str(file_path))
    driver_host_dict['cpsnode'] = driver_host_list[0].replace('\n', '').strip(
        ',').split(',')
    logger.info("Get cps node form %s:%s" %
                (file_path, str(driver_host_dict.get('cpsnode'))))
    driver_host_dict['eBackupDriver'] = driver_host_list[1]. \
        replace('\n', '').strip(',').split(',')
    logger.info("Get ebackup driver node from %s:%s." %
                (file_path, str(driver_host_dict.get('eBackupDriver'))))
    return driver_host_dict


def ping_in_windows(host, retry_times=3):
    def do_ping_in_windows():
        logger.info("Begin to ping %s." % host)
        cmd = 'ping -w 1000 -n %d %s' % (4, host)
        for _ in range(0, retry_times):
            time.sleep(2)
            ping_process = subprocess.Popen(args=cmd, shell=False,
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.STDOUT)
            (std_output, err_output) = ping_process.communicate(timeout=1200)
            if str(std_output).find('TTL=') >= 0:
                logger.info("Ping %s successfully." % host)
                return True
        logger.error("Ping %s failed." % host)
        return False

    try:
        return do_ping_in_windows()
    except Exception as err:
        logger.error("ping host failed,the reason is:" + str(err))
        raise HCCIException(650008, str(err)) from err


def ping_reverse_proxy_ips(check_result, key, host_list, reverse_proxy_ip, current_region_id):
    def do_ping_reverse_proxy_ips():
        logger.info("Begin to ping %s" % ','.join(host_list))
        unpassed_ips = []
        for host_ip in host_list:
            is_true = ping_in_windows(host_ip)
            if not is_true:
                unpassed_ips.append(host_ip)
                logger.error("ping %s failed." % host_ip)
        if 0 != len(unpassed_ips):
            cascaded_reverse_ips_file = get_reverse_proxy_file(
                current_region_id, reverse_proxy_ip)
            check_result.set_check_result(
                param_keys=[key],
                status=500,
                error_msg=HCCIException(650024, ','.join(unpassed_ips), ','.join(unpassed_ips),
                                        cascaded_reverse_ips_file))
            return False
        logger.info("Ping %s successfully." % ','.join(host_list))
        return True

    try:
        return do_ping_reverse_proxy_ips()
    except Exception as err:
        check_result.set_check_result(param_keys=[key], status=500, error_msg=HCCIException(650008, str(err)))
        return False


def format_check_results(check_results, precheck_result):
    logger.info("Begin to format precheck result...")
    for (key, val) in precheck_result.items():
        if 0 == len(val):
            check_result = CheckResult(
                itemname_ch=g_check_item.get(key, dict()).get('name_cn'),
                itemname_en=g_check_item.get(key, dict()).get('name_en'),
                status="success")
        else:
            unpassed_nodes = []
            error_msg_temp = {}
            error_msg = {}

            for (host_ip, msg) in list(val.items()):
                unpassed_nodes.append(host_ip)
                if msg not in error_msg:
                    error_msg_temp.update({msg: [host_ip]})
                else:
                    error_msg_temp.get(msg, []).append(host_ip)

            for (msg, ips) in list(error_msg_temp.items()):
                error_msg.update({','.join(ips): msg})
            error_description = str(list(error_msg.values())[0]) if 1 == len(
                error_msg) else str(error_msg)
            check_result = CheckResult(
                itemname_ch=g_check_item.get(key, dict()).get('name_cn'),
                itemname_en=g_check_item.get(key, dict()).get('name_en'),
                status="failure",
                error_msg_cn=HCCIException(g_check_item.get(key, dict()).get('error_code'),
                                           ','.join(unpassed_nodes),
                                           error_description))
        check_results.append(check_result)
    logger.info("Format precheck result completed.")


def precheck_ebackup_os(check_results, ssh_info: SshInfo, update_scene, dmk_float_ip, version):
    ret_ssh_client = None

    def do_precheck_ebackup_os():
        nonlocal ret_ssh_client
        precheck_info = {
            "precheck_result": {"cpu_health": {}, "disk_health": {}, "disk_space": {}, "network_health": {},
                           "service_health": {}, "ha_health": {}},
            "precheck_script": os.path.realpath(__file__ + '/../../../shell_tool/precheck_ebackup.sh'),
            "precheck_cmd": f'sh /home/hcp/precheck_ebackup.sh {update_scene} {dmk_float_ip} {version} &&'
                            f' echo "Success"'
        }
        logger.info("Begin to precheck eBackup: " + str(ssh_info.node_ips))
        check_result, ret_ssh_client = precheck_ebackup_nodes_all(precheck_info, ssh_info, check_results)
        return check_result

    try:
        res = do_precheck_ebackup_os()
    except Exception as err:
        logger.error("Exception occurs when do the precheck:" + str(err))
        raise err
    finally:
        if ret_ssh_client:
            Utils.close_ssh_clinet(ret_ssh_client)
    return res


def precheck_ebackup_nodes_all(precheck_info, ssh_info, check_results):
    check_flag = True
    for host_ip in ssh_info.node_ips:
        ssh_client = Utils.get_ssh_client(ssh_info, login_ip=host_ip)
        Ssh().ssh_send_command(ssh_client, 'rm -rf /home/hcp/precheck_ebackup.sh', '#', 100)
        ret = Ssh().put_file(
            host_ip, ssh_info.ssh_user, ssh_info.ssh_passwd, precheck_info.get("precheck_script"), "/home/hcp/")
        if not ret:
            logger.error(f"Upload {precheck_info.get('precheck_script')} to host({host_ip}) failed.")
            raise HCCIException(650007, host_ip, "/home/hcp", "hcp")
        ret = Ssh().ssh_exec_command_return_list(ssh_client, precheck_info.get('precheck_cmd'))
        logger.info("Excute precheck command result:" + str(ret))
        if -1 == str(ret).find('Success'):
            raise HCCIException(
                650008, "Exception occurs when excute cmd [%s] on %s" %
                        (precheck_info.get('precheck_cmd'), host_ip))
        ret = Ssh().ssh_exec_command_return_list(
            ssh_client, "cat /home/hcp/precheck_result")
        Utils.close_ssh_clinet(ssh_client)
        for item in ret:
            key_val = item.strip().replace('\n', '').split(':')
            key = key_val[0]
            val = key_val[1]
            logger.info(
                "[Precheck], IP:%s, check_item:%s, check_result:%s" %
                (host_ip, key, val))
            if val != 'pass':
                precheck_info.get("precheck_result").get(key, dict()).update({host_ip: val})
                check_flag = False
                logger.error(
                    "[Precheck], IP:%s, check_item:%s, "
                    "check_result:%s" % (host_ip, key, val))
            else:
                continue
        format_check_results(check_results, precheck_info.get("precheck_result"))
        logger.info("Precheck eBackup completed: " + str(ssh_info.node_ips))
        logger.info("The precheck result is:" + str(check_flag))
        return check_flag, ssh_client


def check_versions_compatibility(server_ips, account_hcp_passwd,
                                 account_root_passwd):
    account_hcp = "hcp"
    version_check_cmds = "showsys  | grep -w \"Product Version\" | " \
                         "awk -F '|' '{print $2}' |sed s/[[:space:]]//g"
    ssh = Ssh()
    version = ""
    start_ip = ""
    for ebk_ip in server_ips:
        ssh_client = ssh.ssh_create_client(ebk_ip, account_hcp,
                                           account_hcp_passwd)
        ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
        ssh.ssh_send_command(ssh_client, account_root_passwd, '#', 100)
        result = ssh.ssh_exec_command_return_list(ssh_client,
                                                  version_check_cmds)
        ssh.ssh_close(ssh_client)
        ebk_version = result[0].strip().replace('\n', '')
        logger.info("The node(%s) version is %s." % (ebk_ip, ebk_version))
        if version == "":
            version = ebk_version
            start_ip = ebk_ip
        elif ebk_version != version:
            return False, "the eBackup version of %s and %s is " \
                          "not same." % (start_ip, ebk_ip)

    return True, ''


def get_dmk_precheck_result(check_results, host_list, hcp_password, root_password, version):
    ssh_client = None

    def do_get_dmk_precheck_result():
        nonlocal ssh_client
        precheck_result = {"micro_service_health": {}, "version_compatibility": {}, "gaussdb_health": {}, }
        ssh = Ssh()
        flag = True
        logger.info("Begin to get dmk precheck result: " + str(host_list))
        for host_ip in host_list:
            ssh_client = ssh.ssh_create_client(host_ip, 'hcp', hcp_password)
            ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, root_password, '#', 100)
            clean_cache_cmd = 'touch /tmp/upgrade/upgrade_pkg/ms_upgrade/public/CleanRepCache.txt'
            ssh.ssh_send_command(ssh_client, clean_cache_cmd, '#', 100)
            ret = ssh.ssh_exec_command_return_list(ssh_client, SYS_VERSION_CMD)
            current_version = ret[0].strip().replace("\n", "")
            logger.info("Current version is " + str(current_version))
            if current_version == version:
                logger.info("Current version is %s,No need to do this." % current_version)
                continue
            ret = ssh.ssh_exec_command_return_list(ssh_client, "test -e /tmp/upgrade/prechk_result && echo Success")
            logger.info(
                "Check if the dmk precheck result file exists:" + str(ret))
            if -1 == str(ret).find("Success"):
                precheck_result.get('micro_service_health').update(
                    {host_ip: "Precheck failed,please view the details on DMK."})
                precheck_result.get('version_compatibility').update(
                    {host_ip: "Precheck failed, please view the details on DMK."})
                logger.error("There is no /tmp/upgrade/prechk_result existing.")
                continue
            ret = ssh.ssh_exec_command_return_list(ssh_client, "cat /tmp/upgrade/prechk_result")
            logger.info("Get DMK check result:" + str(ret))
            format_check_results(check_results, precheck_result)
        format_check_results(check_results, precheck_result)
        return flag

    try:
        check_flag = do_get_dmk_precheck_result()
    except Exception as err:
        logger.error("Exception occurs when get dmk precheck result:" + str(err))
        raise err
    finally:
        if ssh_client:
            Utils.close_ssh_clinet(ssh_client)
    logger.info("Get dmk precheck result completed: " + str(host_list))
    logger.info("The precheck result is:" + str(check_flag))
    return check_flag


def check_ssh_port(ip_address):
    ssh_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    logger.info("check ssh port on node(%s)" % ip_address)
    try:
        result = ssh_socket.connect_ex((ip_address, 22))
    except Exception as err:
        logger.info("Failed to connect node(%s) 22"
                    " port. reason %s." % (ip_address, str(repr(err))))
        result = 1
    finally:
        ssh_socket.close()
    if result == 0:
        return True
    else:
        return False


def get_node_manage_ip(ip_address):
    if len(ip_address) == 1:
        return ip_address[0].get('ip')
    for node_ip in ip_address:
        try:
            if check_ssh_port(node_ip.get('ip')):
                return node_ip.get('ip')
        except Exception as err:
            logger.info("Failed to check ssh port on"
                        " node(%s), reason: %s " % (node_ip, str(repr(err))))
    return ip_address[0].get('ip')


def get_deploy_info_from_cmdb(project_id):
    logger.info("Begin to query ebackup deploy info from cmdb...")
    deploy_info_dict = {'eBackup_Datamover_nodes': {}}

    data = ParamUtil().get_need_check_cloud_params(project_id, "ManageOne_public_params")
    current_region_id = BaseOps().get_regionId_by_projectId(project_id)[0]
    pod_id = BaseOps().get_pod_id_by_project_id(project_id)
    cmdb_ip = data.get("cmdb_ip")
    cmdb_password = data.get("cmdb_password")
    cmdb_account = data.get("cmdb_account")
    cmdb = ManageOneCmdbUtil.get_instance(cmdb_ip, cmdb_password, cmdb_account, project_id, pod_id)
    cmdb_nodes_list = cmdb.get_deploy_node_info(current_region_id)

    if 0 == len(cmdb_nodes_list):
        logger.error("Query cmdb nodes info failed.")
        raise Exception("Query cmdb nodes info failed.")
    for node in cmdb_nodes_list:
        if -1 != node["name"].find("eBackup_service"):
            group_name = node["name"][:-1] + 's'
            manage_ip = get_node_manage_ip(node.get('ipAddresses'))
            if group_name not in deploy_info_dict.get('eBackup_Datamover_nodes'):
                deploy_info_dict.get('eBackup_Datamover_nodes').update({group_name: [manage_ip]})
            else:
                deploy_info_dict.get('eBackup_Datamover_nodes')[group_name].append(manage_ip)
        else:
            logger.info("Skip non-ebackup nodes(%s) info." % node["name"])

    logger.info(
        "Query ebackup deploy info completed: " + str(deploy_info_dict))
    return deploy_info_dict


def check_access_status(check_results, web_ip, admin_pwd, role):
    running_status = {'offline': [], 'partial': []}
    token = ''
    session = ''

    def do_check_access_status():
        nonlocal token, session, running_status
        logger.info("Begin to check access status at host:[%s]" % web_ip)
        ip_version = Utils.check_ip_version(web_ip)
        if ip_version == 4:
            url = "https://%s:8088/rest/dev/vbackup_server" % web_ip
        else:
            url = "https://%s:8088/rest/dev/vbackup_server" % web_ip

        token_tmp, session_tmp = Utils.login(web_ip, admin_pwd, ip_version)
        if token_tmp and session_tmp:
            token = token_tmp
            session = session_tmp
        header = {'Accept': 'application/json;version=2.2;charset=UTF-8', 'iBaseToken': token, 'Accept-Language': 'en',
                  'Cookie': 'language=en; %s;DEVICE_ID=dev;sessionIdleTime=60000;MACHINEROLE=2;CSRF_IBASE_TOKEN=%s' % (
                      session, token)}
        rsp = requests.get(url=url, headers=header, verify=False)
        handle_ha_info(rsp, web_ip, running_status, url)
        key = 'access_status'
        if 0 != len(running_status.get('offline')) or 0 != len(running_status.get('partial')):
            logger.error("Some nodes access status is abnormal: " + str(running_status))
            description = "Login https://%s:8088 for details." % web_ip
            abnormal_nodes = running_status.get('offline') + running_status.get('partial')
            check_result = CheckResult(itemname_ch=g_check_item.get(key, dict()).get('name_cn'),
                                       itemname_en=g_check_item.get(key, dict()).get('name_en'), status="failure",
                                       error_msg_cn=HCCIException(g_check_item.get(key, dict()).get('error_code'),
                                                                  ','.join(abnormal_nodes), description))
            check_results.append(check_result)
            return False
        else:
            logger.info("All nodes access status is normal.")
            return True

    try:
        res = do_check_access_status()
    except Exception as err:
        logger.error("Query access status failed.The reason is: " + str(err))
        return False
    finally:
        Utils.logout(web_ip, token, session, role)
    return res


def handle_ha_info(rsp, web_ip, running_status, url):
    if rsp.status_code == 200:
        rsp_dict = rsp.json()
        error_code = rsp_dict["error"]["code"]
        if error_code is None:
            logger.error("errcode is empty.")
            raise Exception("errcode is empty.")
        if error_code != 0:
            logger.error(
                "Query access status failed on [%s],the description is:%s"
                % (web_ip, rsp_dict["error"]["description"]))
            raise Exception(rsp_dict["error"]["description"])
        for node in rsp_dict.get('data'):
            if node.get('RUNNINGSTATUS') == '28':
                running_status.get('offline').append(node.get('MANAGEMENTIP'))
                logger.error(
                    "The access status of {} is Inaccessible".format(
                        node.get('MANAGEMENTIP')))
            elif node.get('RUNNINGSTATUS') == '30':
                running_status.get('partial').append(node.get('MANAGEMENTIP'))
                logger.error("The access status of {} is Partially "
                             "accessible".format(node.get('MANAGEMENTIP')))
            else:
                logger.info("The access status of {} is Accessible".format(
                    node.get('MANAGEMENTIP')))
    else:
        logger.error("Response status code is not 200.The url is" + url)
        raise Exception("Response status code is not 200.")
    logger.info("Check access status completed,"
                "the check reuslt is: " + str(running_status))


def get_current_cipher_mode(server_ip, ebackup_hcp_pwd, ebackup_root_pwd):
    cmd = "grep --color=never KmcAlgMode /opt/huawei-data-protection/ebackup/conf/hcpconf.ini 2>/dev/null"
    ebackup_ssh_info = SshInfo(server_ip, "hcp", ebackup_hcp_pwd, ebackup_root_pwd)
    sshclient = Utils.get_ssh_client(ebackup_ssh_info)
    if not sshclient:
        raise HCCIException(650059, "hcp", server_ip)
    ciper_mode = Utils.ssh_send_comand_return(sshclient, cmd)
    if not ciper_mode or str(ciper_mode).find("KmcAlgMode") == -1:
        raise HCCIException(650060)
    ciper_mode = re.search('KmcAlgMode=([0-1])', str(ciper_mode))
    if not ciper_mode:
        raise HCCIException(650060)
    Utils.set_value_to_config_file("curKmcAlgMode", ciper_mode.group(1))


def ebackup_cipher_precheck(server_ips, ebackup_hcp_pwd, ebackup_root_pwd, update_scene, ebackup_version):
    check_results = []
    for server_ip in server_ips:
        ret = check_support_cipher_change(server_ip, ebackup_hcp_pwd, ebackup_root_pwd)
        if not ret[0]:
            logger.error("Node(%s) Precheck fail." % server_ip)
            return Message(500, ret[1])

    logger.info(f"Begin to precheck nodes({server_ips}) by HCSU firstly.")
    ssh_info = SshInfo(server_ips, "hcp", ebackup_hcp_pwd, ebackup_root_pwd)
    is_true = precheck_ebackup_os(check_results, ssh_info, update_scene, "127.0.0.1", ebackup_version)
    if not is_true:
        logger.error("Precheck by HCSU failed.")
        return Message(200, check_results=check_results)
    return Message(200, check_results=check_results)


def check_support_cipher_change(server_ip, account_hcp_passwd, account_root_passwd):
    account_hcp = "hcp"
    support_cipher_cmd = 'cat /opt/huawei-data-protection/ebackup/conf/hcpconf.ini | ' \
                         'grep "KmcAlgMode" && echo "successful"'
    check_euler_version_cmd = "uname -r | grep -E 'eulerosv2r9|eulerosv2r8|eulerosv2r10|eulerosv2r11' " \
                              "&& echo 'successful'"
    ssh = Ssh()
    ssh_client = ssh.ssh_create_client(server_ip, account_hcp, account_hcp_passwd)

    def do_send_cmd():
        ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
        ssh.ssh_send_command(ssh_client, account_root_passwd, '#', 100)
        check_cipher_result = ssh.ssh_exec_command_return_list(ssh_client, support_cipher_cmd)
        euler_version_result = ssh.ssh_exec_command_return_list(ssh_client, check_euler_version_cmd)
        return check_cipher_result, euler_version_result

    try:
        result_check_cipher, result_euler_version = do_send_cmd()
    except Exception as err:
        description = "PreCheck cipher change fail. Reason: %s." % str(err)
        logger.error(description)
        return False, description
    finally:
        ssh.ssh_close(ssh_client)

    if -1 == str(result_check_cipher).find("successful"):
        description = "eBackup on Node(%s) do not support cipher change." % server_ip
        logger.error(description)
        return False, description
    if -1 == str(result_euler_version).find("successful"):
        description = "os version on Node(%s) do not support cipher change." % server_ip
        logger.error(description)
        return False, description
    return True, ""


def get_cipher_status(project_id):
    cipher_conditions = ProjectApi().get_project_conditions(project_id)
    cipher_alg = ''
    if cipher_conditions.get('generalCipher'):
        cipher_alg = '0'
    elif cipher_conditions.get('SMCompatible'):
        cipher_alg = '1'
    return cipher_alg


def get_cur_cipher_status():
    cur_cipher_alg = Utils.get_value_from_config_file("curKmcAlgMode")
    if not cur_cipher_alg:
        raise HCCIException(650061, os.path.realpath(__file__ + "/../config.ini"))
    return cur_cipher_alg


class ManageOneOcApi(object):
    def __init__(self, project_id, region_id):
        self.project_id = project_id
        self.region_id = region_id
        self.oc_username = ""
        self.oc_password = ""
        self.oc_ip = ""
        self.oc_token = ""
        self._get_oc_account_info()

    def _get_oc_account_info(self):
        logger.info("Begin to get oc acount info.")

        is_primary_region = ProjectApi().get_project_conditions(
            self.project_id).get("PrimaryRegion", False)

        if is_primary_region:
            self.oc_ip = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                "ManageOne_moOMFloatIp",
                self.region_id)
            self.oc_username = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                'ManageOne_oc_username',
                self.region_id)
            self.oc_password = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                'ManageOne_oc_password',
                self.region_id)
        else:
            self.oc_ip = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                "moOMFloatIp",
                self.region_id)

            self.oc_username = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                'oc_username',
                self.region_id)
            self.oc_password = ParamUtil().get_value_from_cloud_param(
                self.project_id,
                "ManageOne_public_params",
                'oc_password',
                self.region_id)

        logger.info(
            "OC ip is:%s, oc username is:%s" % (self.oc_ip, self.oc_username))
        logger.info("Query OC account info successfully.")

    def login(self):
        def do_login():
            logger.info("Begin to login to manageone to get session.")
            url = "https://%s:26335/rest/plat/smapp/v1/sessions" % self.oc_ip
            header = {"Accept": "application/json", "Content-Type": "application/json;charset=UTF-8"}
            data = {"grantType": "password", "userName": self.oc_username, "value": self.oc_password}

            rsp = requests.put(url=url, headers=header, data=json.dumps(data), verify=False)
            if rsp.status_code == 200:
                rsp_data = rsp.json()
                if 'accessSession' not in list(rsp_data.keys()):
                    logger.error("Get session failed.The response body is:" + str(rsp_data))
                    return False
                self.oc_token = rsp_data["accessSession"]
                logger.info("Login to manageone to get session successfully.")
                return True
            else:
                logger.error(f"Get session failed,the status_code is:{str(rsp.status_code)}, text is:{str(rsp.text)}")
                return False

        try:
            return do_login()
        except Exception as err:
            logger.error("Login to manageone failed, the reason is:" + str(err))
            return False

    def logout(self):
        def do_logout():
            logger.info("Begin to logout manageone.")
            url = "https://%s:26335/rest/plat/smapp/v1/sessions" % self.oc_ip
            header = {"Accept": "application/json", "Content-Type": "application/json;charset=UTF-8",
                      "X-Auth-Token": self.oc_token}
            rsp = requests.delete(url=url, headers=header, verify=False)
            if rsp.status_code == 200:
                logger.info("Logout manageone successfully.")
                return True
            else:
                logger.error(f"Get session failed,the status_code is: {str(rsp.status_code)}. "
                             f"The text is:{str(rsp.text)}")
                return False

        try:
            return do_logout()
        except Exception as err:
            logger.error("Logout to manageone failed, the reason is:" + str(err))
            return False

    def report_system_info(self, service_info):
        def do_report_system_info():
            url = "https://%s:26335/rest/cmdb/v2/system-infos" % self.oc_ip
            header = {"Accept": "application/json", "Content-Type": "application/json;charset=UTF-8",
                      "X-Auth-Token": self.oc_token}
            data = {"system_infos": service_info}

            rsp = requests.put(url=url, headers=header, data=json.dumps(data), verify=False)
            if rsp.status_code == 200:
                rsp_data = rsp.json()
                if int(rsp_data["resultCode"]) != 0:
                    logger.error(f"Report system info failed.The error code is:{rsp_data.get('resultCode')}. "
                                 f"The error msg is:{rsp_data.get('resultData')}")
                    return False
                logger.info("Report system info successfully.")
                return True
            else:
                logger.error(f"Report system info to mo failed,the status_code is {str(rsp.status_code)}."
                             f" The text is:{str(rsp.text)}")
                return False

        try:
            return do_report_system_info()
        except Exception as err:
            logger.error("Report system info to manageone failed, the reason is:%s", str(err))
            return False
