# -*- coding:utf-8 -*-
import base64
import json
import os
import shutil
import threading
import zipfile
from collections import defaultdict
from dataclasses import dataclass
from time import sleep

import requests
import utils.common.log as logger
from tenacity import retry, stop_after_attempt, wait_fixed
from utils.business.check_utils import check_value_null
from utils.business.manageone_console_util import ManageOneConsoleUtil
from utils.business.param_util import CheckResult
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.cipher_api import API as CIPHER_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.client.iam_client import API as IAM_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.nginx_client import API as NGINX_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Action
from plugins.eReplication.common.constant import RECOVERY_PLAN_INTERMEDIATE_STATE
from plugins.eReplication.common.constant import RecoveryPlanState
from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import HTTP_STATUS_OK
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.constant import VERSION_MAP
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.model import DmkDeploymentInfo
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.monitor_api import API as MONITOR_API


class API(object):

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.params = Params(self.project_id, self.pod_id)
        self.sudo_client = None
        self.config_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        DR_API(Auth(self.nodes.server_ip_01, self.nodes.ssh_user, self.nodes.ssh_pwd,
                    self.nodes.sudo_user, self.nodes.sudo_pwd)).do_confirm_tool_env_config()
        self.config_tool = FILE_API(self.config_path)
        self.mo_api = MO_API(self.project_id, self.pod_id)

    @retry(stop=stop_after_attempt(5), wait=wait_fixed(30), reraise=True)
    def backup_data(self):
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            logger.info(f"Backup data on {host} in thread.")
            self.sudo_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                                self.nodes.sudo_user,
                                                                self.nodes.sudo_pwd))
            func = (self._backup_data, thread_name,
                    (self.sudo_client, self.params.data_encrypt_pwd), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    @staticmethod
    def _backup_data(sudo_client, encrypt_pwd):
        # 此目录存在则代表已备份过完整数据，无需备份
        sudo_dir_backup_path = f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_SUDO_DIR_NAME}"
        cmd = f"[ -d '{sudo_dir_backup_path}' ] && echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return(sudo_client, cmd)
        if 'CMD_RESULT=0' in str(result):
            logger.info("Complete data has been backed up, no need to backup.")
            return
        data_back_file = DR_API.backup_dr_system_data(
            sudo_client, encrypt_pwd, is_upgrade=True)
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {Path.DATA_BACKUP_PATH} && "
            f"mkdir -p {Path.DATA_BACKUP_PATH} && echo CMD_RESULT=$?",
            "CMD_RESULT=0")
        # 备份系统配置数据
        dest_file = f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {dest_file} && cp -frp {data_back_file} {dest_file} && "
            f"echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 备份安装文件RDInstalled.xml
        install_back_file = f"{Path.DATA_BACKUP_PATH}/{Path.INSTALL_FILE_NAME}"
        SSH_API.send_command(
            sudo_client,
            f"cp -frp {Path.INSTALL_FILE_SRC} {install_back_file} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 备份运行环境
        runtime_path_back = f"{Path.DATA_BACKUP_PATH}/{Path.RUNTIME_BACK_PATH}"
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {runtime_path_back} && "
            f"cp -frp /opt/BCManager/Runtime {runtime_path_back} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 备份高斯数据
        gaussdb_path_back = f"{Path.DATA_BACKUP_PATH}/{Path.GAUSSDB_BACK_PATH}"
        gaussdb_data_path_back = \
            f"{Path.DATA_BACKUP_PATH}/{Path.GAUSSDB_DATA_BACK_PATH}"
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {gaussdb_path_back} {gaussdb_data_path_back} && "
            f"cp -frp {Path.GAUSSDB_PATH} {gaussdb_path_back} && "
            f"cp -frp {Path.GAUSSDB_DATA_PATH} {gaussdb_data_path_back} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 备份sudo配置
        sudo_config_backup_path = \
            f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_SUDO_CONFIG_NAME}"
        SSH_API.send_command(
            sudo_client,
            rf"\cp -rf {Path.SYS_SUDO_CONFIG_PATH} "
            rf"{sudo_config_backup_path} && \cp -rfp {Path.SYS_SUDO_DIR_PATH} "
            rf"{sudo_dir_backup_path} && echo CMD_RESULT=$?", "CMD_RESULT=0")

    def check_env(self):
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            logger.info(f"Check env on {host} in thread.")
            func = (self._check_env, thread_name, (host,), {})
            funcs.append(func)
        result = ExcThread.exec_func_in_thread_and_return(funcs)
        self._backup_nginx_data()
        DR_API.sync_env_config_to_remote(
            self.nodes.server_ip_01, self.nodes.ssh_user, self.nodes.ssh_pwd,
            self.config_path)
        return result

    def _check_env(self, host):
        """检查环境信息

        :return:
        """
        result_check_list = list()
        dr_api = DR_API(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                             self.nodes.sudo_user, self.nodes.sudo_pwd))
        old_version = dr_api.get_current_version()
        pkg_version = PKG_API.get_pkg_version()
        # 将升级前的Server版本号、目标版本号写入env.ini
        self.config_tool.set_value_by_key_and_sub_key(
            Component.REPLICATION, 'current_version', old_version)
        self.config_tool.set_value_by_key_and_sub_key(
            Component.REPLICATION, 'target_version', pkg_version)
        # 升级包版本号与当前环境版本号一致, 说明是新版本,不用升级,
        # 也就不用再检查备份数据
        if pkg_version == old_version:
            logger.info(
                "The current version is equals upgrade package version, "
                "no need check backup data.")
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查运行环境[{host}]",
                    itemname_en=f"Check runtime environment[{host}]",
                    status="success"))
            return result_check_list
        # 如果备份文件已经存在,需要用户手工判断数据备份文件是否有用，此处情况较多，
        # 工具无法正常判断，但在一个正常升级流程结束后的数据清理动作会删除该目录
        # 此处如果检查到文件存在，说明上一次数据清理未做或操作失败
        dest_file = f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        result = SSH_API.exec_command_return_list(
            dr_api.sudo_client, f"[ -f '{dest_file}' ] && echo CMD_RESULT=$?")
        if "CMD_RESULT=0" in str(result):
            logger.error("The backup data is already exists, please confirm.")
            exception = HCCIException('665003', Path.DATA_BACKUP_PATH, host)
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查备份文件[{host}]",
                    itemname_en=f"Check the backup file[{host}]",
                    status="failure", error_msg_cn=exception))
        else:
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查备份文件[{host}]",
                    itemname_en=f"Check the backup file[{host}]",
                    status="success"))
        result_check_list = self._check_disk(
            dr_api.sudo_client, host, result_check_list)
        return result_check_list

    def _backup_nginx_data(self):
        nginx_api = NGINX_API()
        nginx_data = nginx_api.get_dr_nginx_config(self.pod_id)
        if not nginx_data:
            return
        self.config_tool.set_value_by_key_and_sub_key(
            "nginx_backup", "data", json.dumps(nginx_data))
        logger.info("Backup nginx data success.")

    @staticmethod
    def _check_disk(sudo_client, host, result_check_list):
        # 检查运行环境备份文件是否存在
        runtime_path = f"{Path.DATA_BACKUP_PATH}/{Path.RUNTIME_BACK_PATH}"
        result_check_list = DR_API.check_whether_backup_file_exists(
            sudo_client, host, runtime_path, result_check_list)
        # 检查磁盘空间是否足够: /opt
        opt_disk = "/opt"
        opt_size = 4
        result_check_list = DR_API.check_whether_disk_free_space_is_enough(
            sudo_client, host, opt_disk, opt_size, result_check_list)
        # 检查磁盘空间是否可写: /opt
        result_check_list = DR_API.check_whether_disk_can_be_writen(
            sudo_client, host, opt_disk, result_check_list)
        # 检查磁盘空间是否足够: /var
        var_disk = "/var"
        var_size = 4
        result_check_list = DR_API.check_whether_disk_free_space_is_enough(
            sudo_client, host, var_disk, var_size, result_check_list)
        # 检查磁盘空间是否可写: /var
        result_check_list = DR_API.check_whether_disk_can_be_writen(
            sudo_client, host, var_disk, result_check_list)
        # 检查磁盘空间是否足够: /tmp
        tmp_disk = "/tmp"
        tmp_size = 2
        result_check_list = DR_API.check_whether_disk_free_space_is_enough(
            sudo_client, host, tmp_disk, tmp_size, result_check_list)
        return result_check_list

    def check_recovery_plan(self):
        """检查恢复计划数据是否有中间状态"""
        logger.info("Start checking the recovery plan status.")
        result_check_list = list()
        request_api = DR_API.get_dr_request_api(self.project_id, self.pod_id)
        uri = "ws/recoveryplans?startPage=0&planType=0&pageSize=1000"
        state, body = request_api.send_get_request(uri)
        if state != 200:
            logger.info("Query recovery plans failed.")
            exception = HCCIException(675039, f"Request failed, status code is {state}.")
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查恢复计划状态[{request_api.node_ip}]",
                    itemname_en=f"Check the Recovery Plan Status[{request_api.node_ip}]",
                    status="failure", error_msg_cn=exception))
            return result_check_list
        for plan in body.get("records"):
            if plan.get("planStatus", RecoveryPlanState.DEFAULT_STATE) in RECOVERY_PLAN_INTERMEDIATE_STATE:
                exception = HCCIException(675037, "Some recovery plans are not in stable state.")
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查恢复计划状态[{request_api.node_ip}]",
                        itemname_en=f"Check the Recovery Plan Status[{request_api.node_ip}]",
                        status="failure", error_msg_cn=exception))
                return result_check_list
        logger.info("Check the recovery plan status success.")
        return result_check_list

    def rollback_data(self):
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            logger.info(f"Rollback data on {host} in thread.")
            func = (self._rollback_data, thread_name, (host,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def modify_openstack(self):
        iam_pwd = self.params.iam_account_preset_pwd
        iam_users = [f"{service.lower()}_service"
                     for service in self._get_all_installed_service()]
        iam_user = "csdr_service" if Condition(self.project_id). \
            is_primary_hcs_global else iam_users[0]
        self._modify_openstack(iam_user, iam_pwd)

    def _rollback_data(self, host):
        """恢复环境到升级前版本

        :return:
        """
        # 获取系统当前版本号
        auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                    self.nodes.sudo_user, self.nodes.sudo_pwd)
        dr_api = DR_API(auth)
        system_current_version = dr_api.get_current_version()
        SSH_API.send_command(
            dr_api.sudo_client,
            f"echo y|sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh",
            "successfully")
        # 恢复运行环境
        logger.info("Start recovery runtime.")
        pkg_version = PKG_API.get_pkg_version()
        logger.info(
            f"Package_version is {pkg_version}, "
            f"system_current_version is {system_current_version}.")
        # 检查/home/DRManager/.FCU_AUTO_BACK/Runtime_Bak和
        # /home/DRManager/.FCU_AUTO_BACK/FCDConfigData.zip是否存在
        dest_file = f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        runtime_path_back = f"{Path.DATA_BACKUP_PATH}/{Path.RUNTIME_BACK_PATH}"
        result = SSH_API.exec_command_return_list(
            dr_api.sudo_client, f"[ -d '{runtime_path_back}' ] && "
                                f"[ -f '{dest_file}' ] && echo CMD_RESULT=$?")
        if 'CMD_RESULT=0' not in str(result):
            # 如果未找到备份数据，且版本号不同，说明此时是重试，
            # 是已经回退过了把备份的数据删除了(因为删除备份数据是最后一步，
            # 很有可能上次失败的原因是删除备份数据成功，但是向FCU返回结果的时候出现
            # 了网络异常，导致上次步骤执行失败，所以现在直接return，不抛出异常）
            # 否则则说明还未回退但是备份的数据不存在
            if system_current_version == \
                    self.config_tool.get_value_by_key_and_sub_key(
                        Component.REPLICATION, "current_version"):
                logger.info("Can not find the backup data file.")
                return
            else:
                logger.error(
                    "Can not find the backup data file, "
                    "rollback failed.")
                raise Exception("Can not find the backup data.")
        if Condition(self.project_id).is_cloud_platform6:
            # 回退到651时 重建Tomcat用户
            self._rebuild_tomcat_user(dr_api.sudo_client)
            logger.info("Rollback user reinforce.")
            # 回退系统加固项，ICUser是否可登陆配置回退
            # 8.0.6及以后版本ICUser为no login，所以回退到低版本的时候需要还原加固项
            SSH_API.send_command(
                dr_api.sudo_client,
                "usermod -s /bin/bash ICUser;echo EXEC_RESULT=$?",
                "EXEC_RESULT=0", 20)
            logger.info("Rollback user reinforce success.")

        # 回退runtime目录
        self._recover_runtime_path(
            dr_api.sudo_client, pkg_version, system_current_version)
        # 回退RDInstall.xml
        self._recover_install_xml(
            dr_api.sudo_client, pkg_version, system_current_version)
        # 回退gauss数据
        self._recover_gauss_data(dr_api.sudo_client)
        # 821版本以上回退需要先停止icruntime进程
        self._stop_icruntime(dr_api.sudo_client, pkg_version, auth)
        # 回退sudo配置项
        self._recover_sudo_config(dr_api.sudo_client)
        # 回退系统数据
        self._recover_system_data(
            dr_api.sudo_client, self.params.data_encrypt_pwd)
        # 删除备份文件
        self._clear_backup_file(dr_api.sudo_client)

    @staticmethod
    def _stop_icruntime(sudo_client, pkg_version, auth):
        logger.info(f"Begin to stop icruntime")
        if pkg_version < VERSION_MAP.get("8.2.1"):
            logger.info(f"The eReplication version earlier than 8.3.1, no need stop icruntime.")
            return
        script_name = "stop_icruntime.sh"
        script_file = os.path.join(os.path.dirname(
            os.path.dirname(__file__)), "shell_tools", script_name)
        SSH_API.put_file(
            auth.host, auth.ssh_user, auth.ssh_pwd, script_file, Path.DR_USER_PATH)
        result = SSH_API.exec_command_return_list(sudo_client, f"[ -d {Path.TMP_PATCH_PATH} ];echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(result):
            SSH_API.send_command(sudo_client, f"mkdir -p {Path.TMP_PATCH_PATH};echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 拷贝文件
        SSH_API.send_command(sudo_client,
                             fr"\cp -rf {Path.DR_USER_PATH}/{script_name} {Path.TMP_PATCH_PATH} && echo CMD_RESULT=$?",
                             "CMD_RESULT=0")
        SSH_API.send_command(sudo_client, f"cd {Path.TMP_PATCH_PATH} && sh {script_name}",
                             "Stop icruntime service success.")
        # 清理
        SSH_API.send_command(sudo_client, f"rm -rf {Path.TMP_PATCH_PATH};echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.DR_USER_PATH}/{script_name};echo CMD_RESULT=$?",
                             "CMD_RESULT=0")
        logger.info(f"Stop icruntime success.")

    @staticmethod
    def _rebuild_tomcat_user(ssh_client):
        user_name = "Tomcat"
        check_cmd = f"cat /etc/passwd | grep {user_name}"
        result = SSH_API.exec_command_return_list(ssh_client, check_cmd)
        if result:
            logger.info(f"{user_name} already exists, no need rebuild.")
            return True
        rebuild_cmd = \
            f"useradd {user_name} -u 20001 -m && " \
            f"chage -M 99999 {user_name} && " \
            f"usermod -g 20000 {user_name} && echo EXEC_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, rebuild_cmd)
        if not result or "EXEC_RESULT=0" not in result:
            logger.error(
                f"{user_name} rebuild failed, cmd is {rebuild_cmd}, "
                f"result is {result}.")
            raise Exception(f"{user_name} rebuild failed.")
        adjust_cmd = \
            f"usermod -s /sbin/nologin {user_name} && " \
            f"echo 'umask 027' >> /home/{user_name}/.bashrc && " \
            f"echo EXEC_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, adjust_cmd)
        if not result or "EXEC_RESULT=0" not in result:
            logger.error(
                f"{user_name} adjust failed, cmd is {rebuild_cmd}, "
                f"result is {result}.")
            raise Exception(f"{user_name} adjust failed.")
        return True

    @staticmethod
    def _recover_runtime_path(
            sudo_client, pkg_version, system_current_version):
        # 如果版本号相同，则说明还未回退，则要先删除升级后的资源信息
        if pkg_version != system_current_version:
            logger.info(f"Rollback completed, skip.")
            return
        runtime_path_back = f"{Path.DATA_BACKUP_PATH}/{Path.RUNTIME_BACK_PATH}"
        SSH_API.send_command(
            sudo_client, "chattr -iR /opt/BCManager/Runtime/*")
        SSH_API.send_command(
            sudo_client, "rm -fr /opt/BCManager/Runtime")
        SSH_API.send_command(
            sudo_client, "mkdir -p /opt/BCManager/Runtime")
        SSH_API.send_command(
            sudo_client, "chown -Rh ICUser:LEGO /opt/BCManager")
        SSH_API.send_command(
            sudo_client, "chmod -R 750 /opt/BCManager")
        SSH_API.send_command(
            sudo_client, "rm -fr /opt/BCManager/Runtime/*")
        logger.info("Clear resource success.")
        result = SSH_API.exec_command_return_list(
            sudo_client,
            rf"\cp -frp {runtime_path_back}/* "
            rf"/opt/BCManager/Runtime/ && echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(result):
            logger.error("Restore runtime path failed.")
            raise Exception(str(result))
        logger.info("Cp runtime_path_back success.")

    @staticmethod
    def _recover_install_xml(sudo_client, pkg_version, system_current_version):
        # 检查/home/ICUser/RDInstalled.xml
        install_file = f"{Path.DATA_BACKUP_PATH}/{Path.INSTALL_FILE_NAME}"
        result1 = SSH_API.exec_command_return_list(
            sudo_client, f"[ -f '{install_file}' ] && echo CMD_RESULT=$?")
        if 'CMD_RESULT=0' not in str(result1):
            # 如果未找到备份的RDInstalled.xml，且版本号不同，说明此时是重试，
            # 是已经回退过了把备份的数据删除了，不抛出异常
            # 否则则说明还未回退但是备份的数据不存在
            if pkg_version != system_current_version:
                logger.info("Can not find the backup RDInstalled.xml.")
                return
            else:
                logger.error(
                    "RDInstalled.xml not found, rollback failed.")
                raise Exception("RDInstalled.xml not found.")
        # 恢复安装文件
        logger.info("Start recover RDInstalled.xml.")
        result = SSH_API.exec_command_return_list(
            sudo_client,
            f"rm -f {Path.INSTALL_FILE_SRC} && "
            f"cp -frp {install_file} {Path.INSTALL_FILE_SRC} && "
            "echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(result):
            logger.error("Restore RDInstalled.xml failed.")
            raise Exception(str(result))

    @staticmethod
    def _recover_gauss_data(sudo_client):
        # 恢复高斯文件
        logger.info("Start recover gaussdb path.")
        gaussdb_path_back = f"{Path.DATA_BACKUP_PATH}/{Path.GAUSSDB_BACK_PATH}"
        gaussdb_path = Path.GAUSSDB_PATH
        logger.info("Begin to restore GaussDB.")
        SSH_API.send_command(sudo_client, f"chattr -iR {gaussdb_path}")
        SSH_API.send_command(
            sudo_client,
            rf"rm -fr {gaussdb_path} && \cp -frp {gaussdb_path_back} "
            rf"{gaussdb_path} && echo CMD_RESULT=$?", "CMD_RESULT=0")
        gaussdb_data_path_back = \
            f"{Path.DATA_BACKUP_PATH}/{Path.GAUSSDB_DATA_BACK_PATH}"
        gaussdb_data_path = Path.GAUSSDB_DATA_PATH
        SSH_API.send_command(sudo_client, f"chattr -iR {gaussdb_data_path}")
        SSH_API.send_command(
            sudo_client,
            rf"rm -fr {gaussdb_data_path} && \cp -frp "
            rf"{gaussdb_data_path_back} {gaussdb_data_path} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            sudo_client,
            rf"\cp -d {Path.GAUSSDB_PATH}/lib/libpq.so* "
            rf"/usr/lib && \cp -d "
            rf"{Path.GAUSSDB_PATH}/lib/libpq.so* "
            rf"/usr/lib64 && echo CMD_RESULT=$?", "CMD_RESULT=0")

    @staticmethod
    def _recover_system_data(sudo_client, encrypt_pwd):
        # 恢复系统配置数据
        logger.info("Start recovery config data.")
        # 将数据文件备份目录的权限设置为系统用户可正常访问
        SSH_API.send_command(
            sudo_client, f"chown -h ICUser:LEGO {Path.DATA_BACKUP_PATH} && chmod 740 {Path.DATA_BACKUP_PATH} &&"
                         f"echo CMD_RESULT=$?", "CMD_RESULT=0")
        dest_file = f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sh import.sh"
        SSH_API.send_command(sudo_client, cmd, 'do you want to continue')
        SSH_API.send_command(sudo_client, "y", 'Please input the restore file')
        SSH_API.send_command(
            sudo_client, dest_file, "encrypt password", 50)
        SSH_API.send_command(
            sudo_client, encrypt_pwd, "Restore successfully", 300)

    @staticmethod
    def _recover_sudo_config(sudo_client):
        # 回退/etc/sudoers
        # 8.1.0版本做了安全加固并移动了sudo脚本路径，回退时需要回退/etc/sudoers
        sudo_config_backup_path = \
            f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_SUDO_CONFIG_NAME}"
        sudo_dir_backup_path = \
            f"{Path.DATA_BACKUP_PATH}/{Path.BACKUP_SUDO_DIR_NAME}"
        SSH_API.send_command(
            sudo_client,
            rf"\cp -rf {sudo_config_backup_path} "
            rf"{Path.SYS_SUDO_CONFIG_PATH} && rm -rf "
            rf"{Path.SYS_SUDO_DIR_PATH} && \cp -rfp "
            rf"{sudo_dir_backup_path} {Path.SYS_SUDO_DIR_PATH} && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")

    @staticmethod
    def _clear_backup_file(sudo_client):
        # 回滚成功后,删除备份文件
        logger.info("Start delete server backup data")
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {Path.DATA_BACKUP_PATH} {Path.UPGRADE_TEMP_PATH} "
            f"{Path.DR_USER_PATH}/env.ini && echo CMD_RESULT=$?",
            "CMD_RESULT=0")
        logger.info("Delete server backup data success.")

    def stop_service(self):
        version = DR_API.get_bcm_version(self.project_id, self.pod_id)
        all_hosts = self.nodes.all_hosts
        dmk_float_ip = self.params.dmk_float_ip
        dmk_deploy_user = self.params.dmk_deploy_user
        dmk_user_new_pwd = self.params.dmk_user_new_pwd
        ssh_user = self.nodes.ssh_user
        DMK_API.login_dmk(dmk_float_ip, dmk_deploy_user, dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(dmk_deploy_user, ssh_user)
        if check_value_null(account_id):
            raise Exception("Get dmk account id return false.")
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)),
                         "conf", "server_hosts"))
        host_info = host_info.format(",".join(all_hosts))
        config_info = \
            f"---\nversion: {version}\n\nremote_ssh_user: {ssh_user}"
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        result = DMK_API.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, version, Action.STOP_SERVER,
                              host_info, config_info, account_id))
        if not result:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def upgrade_server_submit(self):
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            logger.info(f"Submit server on {host} in thread.")
            self.sudo_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                                self.nodes.sudo_user,
                                                                self.nodes.sudo_pwd))
            func = (
                self._clear_backup_file, thread_name, (self.sudo_client,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def check_server_status_before_upgrade(self):
        """升级前检查服务状态，返回值为CheckResult列表"""
        result_check_list = list()
        try:
            self._check_server_status()
        except HCCIException as err:
            logger.error(str(err))
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查服务状态",
                    itemname_en=f"Check the server status",
                    status="failure", error_msg_cn=err))
        return result_check_list

    @retry(stop=stop_after_attempt(5), wait=wait_fixed(60), reraise=True)
    def check_server_status(self):
        """升级后检查服务状态，无返回值"""
        self._check_server_status()

    def _check_server_status(self):
        """检查服务状态：
        1.主备状态:sh showSystemStatus.sh
        2.一主三备状态:sh configOneAThreeS.sh -a
        """
        logger.info("Start to check the service status.")
        sudo_client = None
        try:
            physical_ip, _ = DR_API.get_server_node(
                self.nodes.all_hosts, self.nodes.ssh_user, self.nodes.ssh_pwd, self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            if not physical_ip:
                err_msg = "No service listening port is found."
                logger.error(err_msg)
                raise HCCIException(675038, err_msg)
            sudo_client = SSH_API.get_sudo_ssh_client(
                Auth(physical_ip, self.nodes.ssh_user, self.nodes.ssh_pwd, self.nodes.sudo_user, self.nodes.sudo_pwd))
            logger.info("Start to check the service listening port.")
            result = SSH_API.exec_command_return_list(sudo_client, "netstat -ntlp | grep 9443 && echo CMD_RESULT=$?")
            if "CMD_RESULT=0" not in str(result):
                err_msg = "No service listening port is found."
                logger.error(err_msg)
                raise HCCIException(675038, err_msg)
            # 检查主备状态
            self._check_active_standby_status(sudo_client)
            # 检查一主三备状态
            self._check_one_active_and_three_standby_status(sudo_client, physical_ip)
        except Exception as err:
            logger.error(f"Failed to check the server status [{err}]")
            raise err
        finally:
            SSH_API.close_ssh(sudo_client)
        logger.info("Checking the service status is complete.")

    def _check_active_standby_status(self, sudo_client):
        logger.info("Start checking the active/standby status.")
        active_node = self._check_active_standby_total_status(sudo_client)
        self._check_active_standby_subprocess_status(sudo_client, active_node)
        logger.info("Checking the active/standby status is complete.")

    @staticmethod
    def _check_active_standby_total_status(sudo_client):
        logger.info("Start checking the number of running services.")
        cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sh showSystemStatus.sh|awk 'NR==8,NR==9 {{print $1,$6,$7,$8}}'"
        res = SSH_API.exec_command_return_list(sudo_client, cmd)
        logger.info(f"Query result:{res}")
        if len(res) != 2 or '--' in str(res) or 'HA failed' in str(res):
            err_msg = "The number of running services is not equal to 2."
            logger.error(err_msg)
            raise HCCIException(675038, err_msg)
        logger.info("Start to check the general running status of the service.")
        node_name1, ha_active1, ha_all_res_ok1, ha_run_phase1 = res[0].strip().split(" ")
        node_name2, ha_active2, ha_all_res_ok2, ha_run_phase2 = res[1].strip().split(" ")
        # 主端HAActive=active, HAAllResOK=normal,HARunPhase=Actived
        # 备端HAActive=standby,HAAllResOK=normal,HARunPhase=Deactived
        # node1是主时的异常情况
        node1_active_abnormal = ha_all_res_ok1 != 'normal' or ha_run_phase1 != 'Actived'
        node2_standby_abnormal = ha_active2 != 'standby' or ha_all_res_ok2 != 'normal' or ha_run_phase2 != 'Deactived'
        node1_active_total_is_abnormal = ha_active1 == 'active' and (node1_active_abnormal or node2_standby_abnormal)

        # node1是备时的异常情况
        node1_standby_abnormal = ha_all_res_ok1 != 'normal' or ha_run_phase1 != 'Deactived' or ha_active2 != 'active'
        node2_active_abnormal = ha_all_res_ok2 != 'normal' or ha_run_phase2 != 'Actived'
        node1_standby_total_is_abnormal = ha_active1 == 'standby' and (node1_standby_abnormal or node2_active_abnormal)
        running_abnormal = ha_active1 not in ['active', 'standby'] or ha_active2 not in ['active', 'standby']
        if running_abnormal or node1_active_total_is_abnormal or node1_standby_total_is_abnormal:
            err_msg = "Failed to check the overall running status of the service."
            logger.error(err_msg)
            raise HCCIException(675038, err_msg)
        logger.info("Checking the number of running services is complete.")
        return node_name1 if ha_active1 == 'active' else node_name2

    @staticmethod
    def _check_active_standby_subprocess_status(sudo_client, active_node):
        logger.info("Start checking the status of each subprocess.")
        output = SSH_API.exec_command_return_list(
            sudo_client, f"cd {Path.BCM_SCRIPTS_PATH} && sh showSystemStatus.sh|awk 'NR==12,NR==23 {{print $1,$2,$3}}'")
        logger.info(f"Query result:{output}")
        # 主端eReplicationServer、floatIP、nginx、webServer、nodeAgent为Normal,gaussdb为Active_normal
        # 备端eReplicationServer、floatIP、nginx、webServer为Stopped,nodeAgent为Normal,gaussdb为Standby_normal
        other_process = ["eReplicationServer", "floatIP", "nginx", "webServer"]
        for node_name, res_name, res_status in [record.strip().split(" ") for record in output if record]:
            node_agent_abnormal = res_name == "nodeAgent" and res_status != 'Normal'
            node_is_active = node_name == active_node
            active_gaussdb_abnormal = node_is_active and res_name == 'gaussdb' and res_status != "Active_normal"
            standby_gaussdb_abnormal = not node_is_active and res_name == 'gaussdb' and res_status != "Standby_normal"
            active_other_abnormal = node_name == active_node and res_name in other_process and res_status != "Normal"
            standby_other_abnormal = node_name != active_node and res_name in other_process and res_status != "Stopped"
            active_process_normal = active_gaussdb_abnormal or active_other_abnormal
            standby_process_normal = standby_gaussdb_abnormal or standby_other_abnormal
            if node_agent_abnormal or active_process_normal or standby_process_normal:
                err_msg = f"{node_name}'s {res_name} status is {res_status}, abnormal!"
                logger.error(err_msg)
                raise HCCIException(675038, err_msg)
            logger.info(f"{node_name}'s {res_name} status is {res_status}, OK!")
        logger.info("Checking the status of each subprocess is complete.")

    def _check_one_active_and_three_standby_status(self, sudo_client, physical_ip):
        logger.info("Start to check the status of one active and three standby services.")
        if len(self.nodes.all_hosts) == 2:
            logger.info("No need check the status of one active and three standby services, service is OK!")
            return
        ad_output = SSH_API.exec_command_return_list(
            sudo_client, f"cd {Path.BCM_SCRIPTS_PATH} && sh configOneAThreeS.sh -a|awk 'NR==7,NR==10 {{print $2, $3}}'")
        logger.info(f"Query result:{ad_output}")
        output_lst = [record.strip().split(" ") for record in ad_output if record]
        if len(ad_output) != 4:
            err_msg = "The number of one active and three standby services is not equal to 4."
            logger.error(err_msg)
            raise HCCIException(675040, err_msg)
        for node_ip, node_status in output_lst:
            active_node_abnormal = node_ip == physical_ip and node_status != "Active"
            standby_node_abnormal = node_ip != physical_ip and node_status != "Standby"
            if active_node_abnormal or standby_node_abnormal:
                err_msg = "The status of one active node and three standby nodes is abnormal."
                logger.error(err_msg)
                raise HCCIException(675040, err_msg)
        logger.info("Checking the status of one active node and three standby nodes is complete.")

    def start_service(self):
        version = DR_API.get_bcm_version(self.project_id, self.pod_id)
        all_hosts = self.nodes.all_hosts
        dmk_float_ip = self.params.dmk_float_ip
        dmk_deploy_user = self.params.dmk_deploy_user
        dmk_user_new_pwd = self.params.dmk_user_new_pwd
        ssh_user = self.nodes.ssh_user
        DMK_API.login_dmk(dmk_float_ip, dmk_deploy_user, dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(dmk_deploy_user, ssh_user)
        if check_value_null(account_id):
            raise Exception("Get dmk account id return false.")
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)),
                         "conf", "server_hosts"))
        host_info = host_info.format(",".join(all_hosts))
        config_info = \
            f"---\nversion: {version}\n\nremote_ssh_user: {ssh_user}"
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        result = DMK_API.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, version, Action.START_SERVER,
                              host_info, config_info, account_id))
        if not result:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def upgrade_service(self):
        logger.info("Enter upgrade_service.")
        pkg_version = PKG_API.get_pkg_version()
        self._update_os_soft()
        primary_hosts = self.nodes.primary_hosts
        standby_hosts = self.nodes.standby_hosts
        primary_host_info, standby_host_info = self._get_upgrade_host_info(
            primary_hosts, standby_hosts, pkg_version)
        if primary_host_info:
            primary_config_info = self._get_upgrade_config_info(
                primary_hosts, pkg_version)
            self._do_upgrade_task(
                primary_host_info, primary_config_info, pkg_version)
        if standby_host_info:
            standby_config_info = self._get_upgrade_config_info(
                standby_hosts, pkg_version)
            self._do_upgrade_task(
                standby_host_info, standby_config_info, pkg_version)
            self.config_one_a_three_s()
        IAM_API(self.project_id, self.pod_id).create_op_svc_iam_account()

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(30), reraise=True)
    def _update_os_soft(self):
        logger.info("Enter update_os_soft.")
        pkg_path = PKG_API.get_file_abs_path(Pkg.SERVER_PKG_PREFIX, Pkg.TOOL_PKG_SUFFIX)
        PKG_API.check_compressed_file(pkg_path, Capacity.MB_S, Pkg.TOOL_PKG_SIZE_LIMIT, Pkg.TOOL_PKG_FILE_LIMIT)
        thread_funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd, self.nodes.sudo_user, self.nodes.sudo_pwd)
            self.sudo_client = SSH_API.get_sudo_ssh_client(auth)
            thread_func = (
                self._update_os_soft_to_node, thread_name,
                (self.sudo_client, auth, pkg_path), {}
            )
            thread_funcs.append(thread_func)
        ExcThread.exec_func_in_thread(thread_funcs)
        logger.info("The update os soft is complete.")

    def _update_os_soft_to_node(self, sudo_client, auth, pkg_path):
        logger.info("Enter update_os_soft_to_node.")
        result = SSH_API.exec_command_return_list(sudo_client, r"echo CMD_RESULT=$(uname -r)")
        if not self._is_need_update_soft(result):
            logger.info("Node soft no need update.")
            return
        pkg_name = os.path.basename(pkg_path)
        if self._os_soft_is_already_upgrade(sudo_client):
            logger.info("Node soft already upgrade.")
            # 清理目录
            self._clear_upgrade_tool(pkg_name, sudo_client)
            return
        # 创建目录
        result = SSH_API.exec_command_return_list(sudo_client, f"[ -d {Path.TMP_PATCH_PATH} ];echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(result):
            SSH_API.send_command(sudo_client, f"mkdir -p {Path.TMP_PATCH_PATH};echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 传包
        cmd = f"[ -f '{Path.TMP_PATCH_PATH}/{pkg_name}' ];echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            self._upload_upgrade_tool_to_node(auth, sudo_client, pkg_path, pkg_name)
        # 解压
        cmd = f"[ -f '{Path.TMP_PATCH_PATH}/patch.sh' ];echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            cmd = f"cd {Path.TMP_PATCH_PATH} && tar -vxf '{pkg_name}' -C {Path.TMP_PATCH_PATH};echo CMD_RESULT=$?"
            SSH_API.send_command(sudo_client, cmd, "CMD_RESULT=0")
        # 执行patch
        result = SSH_API.exec_command_return_list(sudo_client, f"cd {Path.TMP_PATCH_PATH} && sh patch.sh", timeout=180)
        if "Patch successful." not in str(result):
            err_msg = f"Update soft to node {auth.host} failed."
            logger.error(err_msg)
            raise Exception(err_msg)
        self._clear_upgrade_tool(pkg_name, sudo_client)
        logger.info(f"Update soft to node {auth.host} success.")

    @staticmethod
    def _clear_upgrade_tool(pkg_name, sudo_client):
        logger.info("Enter clear upgrade tool.")
        # 清理目录
        SSH_API.send_command(sudo_client, f"rm -rf {Path.TMP_PATCH_PATH};echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.DR_USER_PATH}/'{pkg_name}';echo CMD_RESULT=$?",
                             "CMD_RESULT=0")
        logger.info("Clear upgrade tool success.")

    def _upload_upgrade_tool_to_node(self, auth, sudo_client, pkg_path, pkg_name):
        logger.info(f"Start upload upgrade tool to node {auth.host}")
        cmd = f"[ -f {Path.DR_USER_PATH}/'{pkg_name}' ];echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            upload_res = SSH_API.put_file(
                auth.host, auth.ssh_user, auth.ssh_pwd, pkg_path, Path.DR_USER_PATH)
            if not upload_res:
                self._clear_upgrade_tool(pkg_name, sudo_client)
                err_msg = f"Upload upgrade tool to {auth.host} failed."
                logger.error(err_msg)
                raise Exception(err_msg)
        # 拷贝包
        cmd = fr"\cp -rf {Path.DR_USER_PATH}/'{pkg_name}' {Path.TMP_PATCH_PATH} && echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            self._clear_upgrade_tool(pkg_name, sudo_client)
            err_msg = f"Copy upgrade tool failed."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Upload upgrade tool to node {auth.host} success.")

    @staticmethod
    def _os_soft_is_already_upgrade(sudo_client):
        cmd = f"strings /lib64/libc.so.6 | grep GLIBC_2.25;echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            return False
        cmd = f"strings /lib64/libstdc++.so.6 | grep GLIBCXX_3.4.22;echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            return False
        cmd = f"which expect 2>/dev/null;echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            return False
        return True

    @staticmethod
    def _is_need_update_soft(result):
        # 是欧拉2.7 X86规格及连欧拉版本都没有，则需要更新系统软件libc, libstdc++, 安装expect
        if "CMD_RESULT=3.10.0-" not in str(result):
            return False
        return "eulerosv2r7.x86_6" in str(result) or "h33.x86_64" in str(result)

    def modify_sys_config(self):
        """Modify sys config: lego.properties"""

        installed_service = self._get_all_installed_service()
        installed_service_iam = []
        for service in installed_service:
            installed_service_iam.append(service.lower() + "_service")
        if Condition(self.project_id).is_primary_hcs_global and \
                "csdr_service" not in installed_service_iam:
            installed_service_iam.append("csdr_service")
        global_domain = self.params.global_domain
        sdr_infos = DR_API.get_sdr_info(self.mo_api.cmdb_ins)
        for host in self.nodes.all_hosts:
            target_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                             self.nodes.sudo_user, self.nodes.sudo_pwd))
            DR_API.do_set_iam_service_name_info(
                target_client, installed_service_iam)
            DR_API.do_set_sdr_sys_config(
                target_client, sdr_infos, global_domain)
            DR_API.do_set_hcs_domain_sys_config(target_client, global_domain)
        logger.info("Modify sys config success.")

    def _get_upgrade_host_info(
            self, primary_hosts, standby_hosts, pkg_version):
        host_vars = ""
        standby_host_vars = ""
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)),
                         "conf", "server_hosts"))
        for host in primary_hosts:
            dr_api = DR_API(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                 self.nodes.sudo_user, self.nodes.sudo_pwd))
            current_version = dr_api.get_current_version()
            if pkg_version != current_version:
                host_vars = host if not host_vars else f"{host_vars},{host}"
        if standby_hosts:
            standby_local_ip = standby_hosts[0]
            standby_peer_ip = standby_hosts[1]
            for host in [standby_local_ip, standby_peer_ip]:
                dr_api = DR_API(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                     self.nodes.sudo_user, self.nodes.sudo_pwd))
                current_version = dr_api.get_current_version()
                if pkg_version != current_version:
                    standby_host_vars = host if not standby_host_vars else \
                        f"{standby_host_vars},{host}"
        primary_host_info = host_info.format(host_vars) if host_vars else ""
        standby_host_info = \
            host_info.format(standby_host_vars) if standby_host_vars else ""
        return primary_host_info, standby_host_info

    def _get_upgrade_config_info(self, hosts, pkg_version):
        global_domain_name = self.params.global_domain
        oc_domain = f"oc.{self.region_id}.{global_domain_name}"
        om_info = f"{oc_domain}:26335"
        er_om_info = f"{oc_domain}:26335"
        # 证书注册地址
        cert_url = f"{oc_domain}:26335"
        local_ip = hosts[0]
        peer_ip = hosts[1]
        region_id = DR_API(Auth(local_ip, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                self.nodes.sudo_user, self.nodes.sudo_pwd)).get_current_region_id()
        # 构造安装配置信息
        config_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)),
                         'conf', "server_upgrade.yml"))
        config_info = config_info.format(
            pkg_version, self.nodes.ssh_user, local_ip, peer_ip, cert_url,
            region_id, om_info, er_om_info)
        return config_info

    def _do_upgrade_task(self, host_info, config_info, version):
        DMK_API.login_dmk(
            self.params.dmk_float_ip, self.params.dmk_deploy_user,
            self.params.dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(
            self.params.dmk_deploy_user, self.nodes.ssh_user)
        if check_value_null(account_id):
            raise Exception("Get dmk account_id return false.")
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        result = DMK_API.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, version, Action.UPGRADE_SERVER,
                              host_info, config_info, account_id))
        if not result:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def upload_server_package(self):
        pkg_prefix = Pkg.SERVER_PKG_PREFIX
        pkg_suffix = Pkg.SERVER_PKG_SUFFIX
        pkg_path = PKG_API.get_file_abs_path(pkg_prefix, pkg_suffix)
        PKG_API.check_compressed_file(
            pkg_path, Capacity.GB_S, Pkg.SERVER_PKG_SIZE_LIMIT,
            Pkg.SERVER_PKG_FILE_LIMIT)
        pkg_version = PKG_API.get_version(pkg_path)
        DMK_API.login_dmk(
            self.params.dmk_float_ip, self.params.dmk_deploy_user,
            self.params.dmk_user_new_pwd)
        # 上传包之前先调用删除接口删除相同版本软件包，防止重新换包因为包仓库已存在相同版本包导致换包失败
        DMK_API.delete_pkg_from_dmk(Component.REPLICATION, pkg_version)
        # 获取上传包结果
        DMK_API.upload_oversize_pkg_to_dmk(pkg_path, self.params.dmk_float_ip, self.params.dmk_os_business_user,
                                           self.params.dmk_os_business_user_pwd)
        logger.info("Upload eReplication server package success.")

    def config_one_a_three_s(self):
        primary_ip, second_ip = self.nodes.primary_hosts
        local_ip, peer_ip = self.nodes.standby_hosts
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)),
                         "conf", "server_hosts"))
        pkg_version = PKG_API.get_pkg_version()
        host_info = host_info.format(
            f"{primary_ip},{second_ip},{local_ip},{peer_ip}")
        primary_region_id = DR_API(Auth(primary_ip, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                        self.nodes.sudo_user,
                                        self.nodes.sudo_pwd)).get_current_region_id()
        standby_region_id = DR_API(Auth(local_ip, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                        self.nodes.sudo_user,
                                        self.nodes.sudo_pwd)).get_current_region_id()
        region_ids = f"{primary_region_id},{standby_region_id}"
        config_info = \
            f"---\nversion: {pkg_version}\n\n" \
            f"remote_ssh_user: {self.nodes.ssh_user}"
        config_info += "\n".join((
            '\n# 配置一主三备,在安装备region的时候配置以下参数',
            '# 主端Server IP地址',
            f'primary_server_ip1: \'{primary_ip}\'\n',
            f'primary_server_ip2: \'{second_ip}\'\n',
            '# 备端Server IP地址',
            f'standby_server_ip1: \'{local_ip}\'\n',
            f'standby_server_ip2: \'{peer_ip}\'\n',
            '# 主备Region对应的Region ID，'
            '格式: \'主Region Region ID,备Region Region ID\'',
            f'region_ids: \'{region_ids}\'\n'))
        DMK_API.login_dmk(
            self.params.dmk_float_ip, self.params.dmk_deploy_user,
            self.params.dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(
            self.params.dmk_deploy_user, self.nodes.ssh_user)
        result = DMK_API.execute_dmk_deployment(DmkDeploymentInfo(Component.REPLICATION, True, pkg_version,
                                                                  Action.CONFIG_ONE_A_THREE_S, host_info, config_info,
                                                                  account_id))
        if not result:
            raise Exception(
                "Execute dmk deployment job configOneAThreeS "
                "return false.")

    def get_version_info(self):
        source_version = self.config_tool.get_value_by_key_and_sub_key(
            Component.REPLICATION, 'current_version')
        if not source_version:
            source_version = DR_API.get_bcm_version(
                self.project_id, self.pod_id)
        target_version = PKG_API.get_pkg_version()
        info = {
            "Originalversion": source_version, "Targetversion": target_version}
        return info

    def reset_iam(self):
        iam_pwd = self.params.iam_account_preset_pwd
        iam_users = [f"{service.lower()}_service"
                     for service in self._get_all_installed_service()]
        if Condition(self.project_id).is_primary_hcs_global and \
                "csdr_service" not in iam_users:
            iam_users.append("csdr_service")
        for iam_user in iam_users:
            if not self._confirm_to_reset(iam_user, iam_pwd):
                logger.debug(f"{iam_user} no need to reset.")
                continue
            logger.debug(f"Start reset {iam_user}.")
            preset_user_token = self._get_preset_user_token()
            user_id = self._get_iam_user_id(iam_user, preset_user_token)
            self._do_reset_iam_user_task(user_id, iam_pwd, preset_user_token)
            self._sync_password_to_unified_password(iam_user, iam_pwd)
        iam_user = "csdr_service" if Condition(self.project_id). \
            is_primary_hcs_global else iam_users[0]
        self._modify_openstack(iam_user, iam_pwd)

    def _get_all_installed_service(self):
        services = set()
        all_region_ids = self.mo_api.get_all_region_id_from_cmdb()
        for region_id in all_region_ids:
            endpoint_zh = self.mo_api.get_endpoints(region_id)
            if Component.CSDR_U in str(endpoint_zh):
                services.add(Component.CSDR_U)
            if Component.CSHA_U in str(endpoint_zh):
                services.add(Component.CSHA_U)
            if Component.VHA_U in str(endpoint_zh):
                services.add(Component.VHA_U)
        logger.info(f"Get all installed service return {services}.")
        return services

    def _confirm_to_reset(self, iam_user, iam_pwd):
        return False if self._get_iam_token(iam_user, iam_pwd) else True

    def _sync_password_to_unified_password(self, iam_user, iam_pwd):
        logger.info(f"Sync {iam_user}'s new password to unified password.")
        if Condition(self.project_id).is_cloud_platform6:
            logger.info(
                f"No need to Sync {iam_user}'s password.")
            return
        ssh_user = self.nodes.ssh_user
        ssh_pwd = self.nodes.ssh_pwd
        sudo_user = self.nodes.sudo_user
        sudo_pwd = self.nodes.sudo_pwd
        physical, float_ip = DR_API.get_server_node(
            self.nodes.all_hosts, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
        self.sudo_client = SSH_API.get_sudo_ssh_client(
            Auth(physical, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
        sync_cmd = \
            f"sh {Path.BCM_SCRIPTS_PATH}sync_unite_pwd.sh;echo CMD_RESULT=$?"
        SSH_API.send_command(self.sudo_client, sync_cmd, "Enter the user name")
        SSH_API.send_command(self.sudo_client, iam_user, "Enter the password")
        SSH_API.send_command(self.sudo_client, iam_pwd, "CMD_RESULT=0")
        logger.info(f"Sync {iam_user}'s password to unified password success.")

    def _do_reset_iam_user_task(self, user_id, pwd, token):
        logger.info(f"Start to reset iam user password: {user_id}.")
        iam_domain = f"iam-cache-proxy.{self.params.global_domain}"
        uri = f"https://{iam_domain}:26335/v3-ext/users/{user_id}"
        headers = {
            'Content-Type': 'application/json;charset=utf8',
            'X-Auth-Token': token
        }
        req_body = {"user": {"password": pwd}}
        response = requests.put(
            uri, headers=headers, data=json.dumps(req_body), verify=False)
        if response.status_code in HTTP_STATUS_OK:
            logger.info(f"Reset iam user password success: {user_id}.")
            return True
        if response.status_code == 400:
            response_body = self._get_resp_data(response.content)
            if "1108" not in response_body.get("errorcode"):
                logger.error(f"Reset iam user password failed: {user_id}.")
                raise Exception(f"Reset iam user password failed: {user_id}.")
            for index in range(5):
                tmp_pwd = pwd + f"BC&DR{index}"
                req_body = {"user": {"password": tmp_pwd}}
                response = requests.put(
                    uri, headers=headers, data=json.dumps(req_body),
                    verify=False)
                if response.status_code in HTTP_STATUS_OK:
                    logger.info(f"Reset iam user password success: {user_id}.")
                    sleep(10)
                    continue
                logger.error(f"Reset iam user password failed: {user_id}.")
                raise Exception(f"Reset iam user password failed: {user_id}.")
            req_body = {"user": {"password": pwd}}
            response = requests.put(
                uri, headers=headers, data=json.dumps(req_body), verify=False)
            if response.status_code in HTTP_STATUS_OK:
                logger.info(f"Reset iam user password success: {user_id}.")
                return True
        logger.error(f"Reset iam user password failed: {user_id}.")
        raise Exception(f"Reset iam user password failed: {user_id}.")

    def _get_iam_user_id(self, iam_user, token):
        iam_domain = f"iam-cache-proxy.{self.params.global_domain}"
        uri = f"https://{iam_domain}:26335/v3/users"
        headers = {
            'Content-Type': 'application/json;charset=utf8',
            'X-Auth-Token': token,
        }
        response = requests.get(uri, headers=headers, verify=False)
        if response.status_code not in HTTP_STATUS_OK:
            logger.error(
                f"Get iam users request failed: {response.status_code}.")
            raise Exception(
                f"Get iam users request failed: {response.status_code}.")
        response_body = self._get_resp_data(response.content)
        users = response_body.get("users")
        if not users:
            logger.error(
                f"Get iam users request failed: {response.status_code}.")
            raise Exception(
                f"Get iam users request failed: {response.status_code}.")
        for user in users:
            if user.get("name") == iam_user:
                logger.info(f"Get iam user {iam_user} id success.")
                return user.get("id")
        logger.error(f"{iam_user} not found.")
        raise Exception(f"{iam_user} not found.")

    def _get_iam_token(self, iam_user, iam_pwd):
        iam_domain = f"iam-cache-proxy.{self.params.global_domain}"
        uri = f"https://{iam_domain}:26335/v3/auth/tokens"
        headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json;charset=utf8'
        }
        req_body = self._get_iam_req_body(iam_user, iam_pwd)
        response = requests.post(
            uri, headers=headers, data=json.dumps(req_body), verify=False)
        if response.status_code in HTTP_STATUS_OK:
            response_headers = response.headers
            token = response_headers.get("X-Subject-Token")
            logger.info("Get the IAM token success.")
            return token
        logger.debug("Get the IAM token failed.")
        return ""

    def _get_preset_user_token(self):
        iam_user = "preset_user"
        iam_pwd = IAM_API(self.project_id, self.pod_id).get_preset_user_pwd()
        token = self._get_iam_token(iam_user, iam_pwd)
        if not token:
            raise Exception("Get iam preset_user token failed.")
        logger.info("Get preset user token success.")
        return token

    @staticmethod
    def _get_iam_req_body(iam_user, iam_pwd):
        req_body = {
            "auth": {
                "identity": {
                    "methods": ["password"],
                    "password": {
                        "user": {
                            "domain": {
                                "name": "op_service"
                            },
                            "name": iam_user,
                            "password": iam_pwd
                        }
                    }
                },
                "scope": {
                    "domain": {
                        "name": "op_service"
                    }
                }
            }
        }
        return req_body

    @staticmethod
    def _get_resp_data(resp):
        if resp is None:
            return defaultdict()
        if resp != 'None' and resp != 'null' and resp:
            return json.loads(resp)
        return defaultdict()

    def _modify_openstack(self, iam_user, iam_pwd):
        request_api = DR_API.get_dr_request_api(self.project_id, self.pod_id)
        uuid = DR_API.get_server_id(request_api)
        devsn = DR_API.get_dest_device_id(
            self.project_id, self.pod_id, request_api, uuid)
        iam_domain = f"iam-cache-proxy.{self.params.global_domain}"
        devsn_b64 = base64.b64encode(
            devsn.encode(encoding='utf-8')).decode(encoding='utf-8')
        rediscover_url = f"ws/resources/{devsn_b64}/action/modifyMgrProtocol"
        data = {
            "networkIpAddress": iam_domain, "networkUserName": iam_user,
            "isModifyPwd": True, "networkPassword": iam_pwd,
            "extendParams": {"port": 26335, "domain": "op_service"}
        }
        state, req_body = request_api.send_put_request(data, rediscover_url)
        if state != 200:
            logger.info("Modify OpenStack authentication information failed.")
            raise Exception(
                'Failed to modify OpenStack authentication information.')
        logger.info("Modify OpenStack authentication information success.")
        sleep(10)
        self._refresh_device(request_api, devsn)

    @staticmethod
    def _refresh_device(request_api, ops_uuid):
        encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        refresh_url = f"ws/fusionsphere/{encode_uuid}/action/refresh"
        state, body = request_api.send_put_request(
            None, refresh_url)
        if state != 200:
            if 'errorCode' in dict(body) and (
                    str(dict(body).get('errorCode', None)) ==
                    "1073947428" or
                    str(dict(body).get('errorCode', None)) == "606"):
                # 正在刷新中
                logger.info("The device is being refresh, "
                            "no need refress again.")
                sleep(20)
                return True
            raise Exception(f'Failed to refresh device[{str(body)}].')
        logger.info("Refresh_device cmd send successfully.")
        sleep(30)
        return True

    def upload_console_pkg(self):
        console_zip_path = PKG_API.get_file_abs_path_by_re(Pkg.CONSOLE_PKG_RE)
        unzip_path = os.path.join(
            os.path.dirname(console_zip_path), "console_pkg_temp")
        if os.path.exists(unzip_path):
            shutil.rmtree(unzip_path)
        # 创建目录
        os.mkdir(unzip_path)
        # 解压
        self._unzip_console_pkg(console_zip_path, unzip_path)
        # 上传
        self.mo_api.upload_console_pkg(unzip_path)

    @staticmethod
    def _unzip_console_pkg(file, target_path):
        zip_file = zipfile.ZipFile(file)
        logger.info(f"Start unzip {file} to {target_path}.")
        # 解压前校验文件大小
        PKG_API.check_compressed_file(
            file, Capacity.MB_S, Pkg.CONSOLE_PKG_SIZE_LIMIT,
            Pkg.CONSOLE_PKG_FILE_LIMIT)
        zip_file.extractall(path=target_path)
        logger.info("Unzip console file success.")

    def backup_console_static(self, backup_type="backup"):
        # 备份原有Console静态资源
        logger.info("Start backup console service.")
        ManageOneConsoleUtil().backup_console(
            self.pod_id, service_type="csdr,csha,vha",
            backup_type=backup_type, is_region=False)
        logger.info("Backup console service success.")

    def upgrade_console_service(self):
        self.upload_console_pkg()
        console_zip_path = PKG_API.get_file_abs_path_by_re(Pkg.CONSOLE_PKG_RE)
        unzip_path = os.path.join(
            os.path.dirname(console_zip_path), "console_pkg_temp")
        console_release_path = PKG_API.get_file_abs_path_by_re(
            Pkg.CONSOLE_RELEASE_RE, pkg_path=unzip_path)
        console_release_name = os.path.basename(console_release_path)
        console_util = ManageOneConsoleUtil()
        # 开始升级
        logger.info("Start upgrade console service.")
        console_util.update_console_by_region(
            self.pod_id, service_type="csdr,csha,vha",
            file_name=console_release_name, is_region=False)
        logger.info("upgrade console service success.")
        # 删除本地Console临时文件
        if os.path.exists(unzip_path):
            shutil.rmtree(unzip_path)

    def rollback_console_service(self):
        logger.info("Start rollback console service.")
        ManageOneConsoleUtil().rollback_console_by_region(
            self.pod_id, service_type="csdr,csha,vha", is_region=False)
        logger.info("Rollback console service success.")

    def update_cmdb(self, is_rollback=False):
        version_info = self.get_version_info()
        original_version = version_info.get("Originalversion", None)
        version = version_info.get("Targetversion", None)
        cipher_type = ''
        if original_version >= VERSION_MAP.get("8.1.0"):
            cipher_type = CIPHER_API(self.project_id, self.pod_id,
                                     self.region_id).get_current_cipher_type()
        region_info = self.mo_api.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            region_type = region.get("regionType")
            cloud_service_info = self.mo_api.cmdb_ins.get_cloud_service_info(
                region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            service_info = cloud_service_info[0]
            extend_infos = service_info.get("extendInfos")
            dr_service_scale = \
                [extend_info.get("value") for extend_info in extend_infos if
                 extend_info.get("key") == "global_service_scale"]
            if Component.CSDR_U not in str(extend_infos) or \
                    Component.CSHA_U not in str(extend_infos) or \
                    Component.VHA_U not in str(extend_infos):
                extend_infos = DR_API.get_cloud_dr_service_installed_list(
                    self.project_id, self.pod_id, region_id)
            # 查询MO CMDB信息，获取环境规模信息
            if region_type in ["master", "dr_standby"]:
                global_service_scale = region.get("globalServiceScale")
                global_service_scale = global_service_scale.split("_")[
                    -1].upper()
                global_service_scale = dr_service_scale[
                    0] if dr_service_scale else global_service_scale
            else:
                global_service_scale = ""
            region_service_info = RegionServiceInfo(region_id, region_type, original_version, global_service_scale,
                                                    cipher_type)
            extend_infos = self._confirm_extend_infos_value(
                extend_infos, region_service_info, is_rollback)
            service_info = self._update_cloud_service_info(
                service_info, extend_infos, version, original_version,
                is_rollback)
            self.mo_api.cmdb_ins.set_cloud_service_info(region_id, service_info)
            logger.info(f"Register CMDB for {region_id} success.")

    def _confirm_extend_infos_value(self, extend_infos, region_service_info, is_rollback):
        if is_rollback:
            extend_infos = [ele for ele in extend_infos if ele.get(
                "key") not in ["global_service_scale", "CipherType"]]
            if region_service_info.original_version < VERSION_MAP.get("8.2.0"):
                extend_infos = [ele
                                for ele in extend_infos
                                if ele.get("key") not in ["CSDR_supported_block", "CSDR_supported_nas"]]
            if region_service_info.original_version >= VERSION_MAP.get("8.0.3"):
                extend_infos.append({"key": "global_service_scale",
                                     "value": region_service_info.global_service_scale})
            if region_service_info.original_version >= VERSION_MAP.get("8.1.0"):
                extend_infos.append(
                    {"key": "CipherType", "value": region_service_info.cipher_type})
        else:
            block_supported, nas_supported = \
                DR_API.get_csdr_installed_info(
                    self.project_id, self.pod_id, region_service_info.region_id)
            if "CSDR_supported_block" not in str(extend_infos):
                extend_infos.append(
                    {"key": "CSDR_supported_block", "value": block_supported})
            if "CSDR_supported_nas" not in str(extend_infos):
                extend_infos.append(
                    {"key": "CSDR_supported_nas", "value": nas_supported})
            if "global_service_scale" not in str(extend_infos):
                extend_infos.append({"key": "global_service_scale",
                                     "value": region_service_info.global_service_scale})
            if "CipherType" not in str(extend_infos):
                extend_infos.append(
                    {"key": "CipherType", "value": region_service_info.cipher_type})
        # 升级/回退的时候都移除ark
        extend_infos = [ele for ele in extend_infos if ele.get("key") != "ARK"]
        if region_service_info.region_type not in ["master", "dr_standby"] or \
                not Condition(self.project_id).is_primary_hcs_global:
            return extend_infos
        for extend_info in extend_infos:
            if extend_info.get("key") == Component.CSDR_U:
                extend_info["value"] = "True"
        return extend_infos

    @staticmethod
    def _update_cloud_service_info(
            service_info, extend_infos, version, original_version,
            is_rollback):
        if not is_rollback:
            service_info.update(
                {"version": version, "originalVersion": original_version})
        else:
            service_info.update({"version": original_version})
        service_info.update({"extendInfos": extend_infos})
        return service_info

    def register_metrics(self):
        all_services = self._get_metrics_info()
        for region_id, metric_info in all_services.items():
            services, console_home = metric_info
            for service in services:
                logger.info(f"Register metrics for {service}.")
                MO_API.register_quota(
                    self.project_id, self.pod_id, region_id, console_home,
                    service.lower())
                MO_API.register_meter(
                    self.project_id, self.pod_id, region_id, console_home,
                    service.lower())

    def _get_metrics_info(self):
        services = dict()
        all_region_ids = self.mo_api.get_all_region_id_from_cmdb()
        for region_id in all_region_ids:
            console_home = ""
            region_services = set()
            endpoint_zh = self.mo_api.get_endpoints(region_id)
            for endpoint in endpoint_zh:
                if endpoint.get("id") not in \
                        [Component.CSDR_L, Component.CSHA_L, Component.VHA_L]:
                    continue
                region_services.add(endpoint.get("shortName"))
                console_home = endpoint.get("endpoint").split("/")[2]
            if not region_services:
                continue
            services[region_id] = [region_services, console_home]
        logger.info(f"Get metrics info return {services}.")
        return services

    def register_nginx(self, is_rollback=False):
        nginx_api = NGINX_API()
        if is_rollback:
            nginx_data = \
                json.loads(self.config_tool.get_value_by_key_and_sub_key(
                    "nginx_backup", "data"))
        else:
            nginx_data = nginx_api.get_dr_nginx_config(self.pod_id)
        if not nginx_data:
            return
        action = "Register_Nginx" if is_rollback else "Remove_Nginx"
        for key, value in nginx_data.items():
            ips, domains, services = value
            for service_type in services:
                have_service = services.get(service_type)
                if have_service.lower() == "false":
                    continue
                upstream_name = service_type + '_console'
                location = [
                    {'name': '/' + service_type, 'attach_url': '',
                     'random_code': service_type}]
                to_primary = True if key == "primary" else False
                to_standby = True if key == "standby" else False
                nginx_api.register_to_nginx(
                    action, self.pod_id, domains[0], upstream_name,
                    location, ips, 7443, to_primary=to_primary,
                    to_standby=to_standby)
                logger.info(
                    f"{action} for {service_type} successfully.")

    def store_origin_adapter_version(self):
        logger.info("Start to store origin adapter version to file.")
        old_version = self.config_tool.get_value_by_key_and_sub_key(Component.REPLICATION, 'adaptation_pkg_version')
        if old_version:
            logger.info("Origin adapter version already exist.")
            return
        origin_version = self.mo_api.query_old_adaptation_pkg_version(self.region_id)
        self.config_tool.set_value_by_key_and_sub_key(Component.REPLICATION, 'adaptation_pkg_version', origin_version)
        logger.info("Store origin adapter version to file successfully.")

    def register_adaptation_package(self):
        logger.info("Start to register adaptation package.")
        self.deal_adaptation_package()
        logger.info("Register adaptation package successfully.")

    def unregister_adaptation_package(self):
        logger.info("Start to unregister adaptation package.")
        old_version = self.config_tool.get_value_by_key_and_sub_key(Component.REPLICATION, 'adaptation_pkg_version')
        if not old_version:
            error_msg = "Old adapter package version not exist, unregister adaptation package failed."
            logger.error(error_msg)
            raise HCCIException(663621, error_msg)
        self.deal_adaptation_package(is_register=False, old_version=old_version)
        logger.info("Unregister adaptation package successfully.")

    def deal_adaptation_package(self, is_register=True, old_version=None):
        funcs = list()
        thread_name = threading.current_thread().name
        region_info = self.mo_api.get_region_info()
        mo_client = MO_API(self.project_id, self.pod_id)
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = self.mo_api.cmdb_ins.get_cloud_service_info(
                region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            if is_register:
                func = (DR_API.register_adaptation_package, thread_name,
                        (self.project_id, self.pod_id, region_id), {})
            else:
                func = (mo_client.unregister_adaptation_package, thread_name,
                        (region_id, old_version), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def config_service_monitor(self, is_rollback=False):
        logger.info("Start to config service monitor.")
        funcs = list()
        thread_name = threading.current_thread().name
        region_info = self.mo_api.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = self.mo_api.cmdb_ins.get_cloud_service_info(
                region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            origin_version = self.mo_api.query_old_adaptation_pkg_version(region_id)
            if origin_version >= VERSION_MAP.get("8.2.1"):
                logger.info(f"Current version is {origin_version}, not need delete service monitor")
                continue
            if is_rollback:
                hosts = self.nodes.primary_hosts if region.get("regionType") == "master" else self.nodes.standby_hosts
                func = (MONITOR_API.add_service_monitor_info, thread_name, (self.pod_id, region_id, hosts), {})
            else:
                func = (MONITOR_API.delete_server_monitor_info, thread_name, (self.pod_id, region_id), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Config service monitor success.")

    def __del__(self):
        if self.sudo_client:
            SSH_API.close_ssh(self.sudo_client)
        logger.info(f"SSH channel is closed")


@dataclass
class RegionServiceInfo:
    region_id: str
    region_type: str
    original_version: str
    global_service_scale: str
    cipher_type: str
