# -*- coding:utf-8 -*-
import os

CSBS_SERVICE_NAME = "CSBS-VBS"
KAEBOR_EXT_PORT = "8799"
KARBOR_INSTALL_WORKDIR = "/karbor"
CSBS_PLUGIN_PATH = "/opt/cloud/hcci/src/HCCI/plugins/CSBS"
DPA_PLUGIN_CONFIG_PATH = 'plugins/CSBS/conf/dpa_pkg_sha256.ini'

KARBOR_IMAGE_NAME = 'CSBS-VBS-image'
KARBOR_PROXY_IMAGE_NAME = 'KarborProxy-image'

PKG_VERSION = '8.5.1.SPC200'
OS_VERSION = '8.5.1'

PKG_SUFFIX_X86 = f'_CSBS-OceanProtect_Karbor_X86.zip'
PKG_SUFFIX_ARM = f'_CSBS-OceanProtect_Karbor_ARM.zip'

COMPONENT = 'CSBS-VBS'

KARBOR_USER_NAME = "djmanager"
KARBOR_ROOT_NAME = "root"
KARBOR_PROXY_USER_NAME = "dpamanager"
KARBOR_PROXY_ROOT_NAME = "root"

# 人机账号
MAN_MACHINE_ACCOUNT = 1

# 证书临时文件存放位置
TEMP_CERT_PATH = os.path.join(os.path.dirname(os.path.dirname(__file__)), "LocalCerts")


class RegisterSDRData(object):
    def __init__(self, region_id, ext_float_ip):
        self.__op_backup_id = "app_backup"
        self.__csbs_id = "csbs"
        self.__region_id = region_id
        self.__ext_float_ip = ext_float_ip
        self.__sdr_data()

    def __sdr_data(self):
        self.op_data = {
            "meterId": self.__op_backup_id,
            "resourceType": self.__op_backup_id,
            "service": "csbs",
            "csbiName": "CSBS",
            "enable": True,
            "regionId": self.__region_id,
            "dataSource":
                f"https://{self.__ext_float_ip}:{KAEBOR_EXT_PORT}/v1/meters?"
                f"service_type={self.__csbs_id}&resource_type={self.__op_backup_id}"
                f"&limit={{0}}&index={{1}}&start={{2}}&end={{3}}",
            "activeDataSource": "",
            "folder": self.__op_backup_id,
            "className": "CommonPluginImpl",
            "version": "v2",
            "cycleTime": 3600000,
            "generationDelayTime": 900000,
            "statistics": False,
            "limit": 1000,
            "collectThreadNum": 5
        }


class RegisterToMOInfo(object):
    def __init__(self, region_id, domain_name=None):
        self.__csbs_id = "csbs"
        self.__vbs_id = "vbs"
        self.__dbs_product_type = "dbs"
        self.__fsbs_product_type = "fsbs"
        self.__app_backup_id = "app_backup"
        self.__byte_to_gb = 1073741824
        self.__region_id = region_id
        self.__domain_name = domain_name
        self.__endpoint_info()
        self.__links_info()
        self.__product_action()
        self.__product_info()
        self.__quota_indicators()
        self.__metrics_info()

    def __endpoint_info(self):
        self.csbs_endpoint_info_zh = {
            "id": self.__csbs_id,
            "name": "云服务器备份 CSBS",
            "shortName": "CSBS",
            "description": "安全可靠的云服务器备份",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__csbs_id}/manager/tabs/backup",
            "serviceCss": self.__csbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "zh-cn",
            "catalogId": 2,
            "regionId": self.__region_id
        }
        self.csbs_endpoint_info_en = {
            "id": self.__csbs_id,
            "name": "Cloud Server Backup Service",
            "shortName": "CSBS",
            "description": "Secure and reliable protection for cloud servers",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__csbs_id}/manager/tabs/backup",
            "serviceCss": self.__csbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "en-us",
            "catalogId": 2,
            "regionId": self.__region_id
        }

    def __links_info(self):
        # Cloud server backup service links.
        self.csbs_detail_link = f"https://{self.__domain_name}/cbs/#" \
                                f"/{self.__csbs_id}/productDetail"

        # Volume backup service links.
        self.vbs_detail_link = f"https://{self.__domain_name}/cbs/#" \
                               f"/{self.__vbs_id}/productDetail"

    def __product_action(self):
        self.csbs_vbs_action_list = [
            {
                "service_type": self.__csbs_id,
                "action": "apply"
            },
            {
                "service_type": self.__csbs_id,
                "action": "modify"
            },
            {
                "service_type": self.__csbs_id,
                "action": "delete"
            },
            {
                "service_type": self.__vbs_id,
                "action": "apply"
            },
            {
                "service_type": self.__vbs_id,
                "action": "modify"
            },
            {
                "service_type": self.__vbs_id,
                "action": "delete"
            }
        ]

    def __product_info(self):
        # Cloud server backup service product info.
        self.csbs_product_info = {
            "proNameZh": "云服务器备份",
            "proNameEn": "Cloud Server Backup Service",
            "serviceId": self.__csbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "云服务器备份（Cloud Server Backup Service）为云服务器下所有云"
                "硬盘创建一致性在线备份，无需关机。针对病毒入侵、人为误删除、软"
                "硬件故障等场景，可将数据恢复到任意备份点。",
            "proDesEn":
                "Cloud Server Backup Service (CSBS) backs up Elastic Volume "
                "Service (EVS) disks consistently on Elastic Cloud Servers "
                "(ECSs) without interrupting services. "
                "Leveraging state-of-the-art protection from virus attacks "
                "and software and hardware faults, CSBS allows you to"
                " restore data from any backup point in time.",
            "catalogId": "2",
            "isDefault": "true"
        }
        # Volume backup service product info.
        self.vbs_product_info = {
            "proNameZh": "云硬盘备份",
            "proNameEn": "Volume Backup Service",
            "serviceId": self.__vbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "云硬盘备份（Volume Backup Service）为云硬盘创建在线备"
                "份，无需关机/重启。针对病毒入侵、人为误删除、软硬件故障"
                "等场景，可将数据恢复到任意备份点。",
            "proDesEn":
                "Volume Backup Service (VBS) provides online backup for "
                "Elastic Volume Service (EVS) disks and allows data "
                "restoration from backups, offering protection against "
                "virus attacks and software and hardware faults.",
            "catalogId": "2",
            "isDefault": "true"
        }

    def __quota_indicators(self):
        # CSBS and VBS quota.
        self.csbs_quota_indicators = {
            "quota_indicators": {
                "region_id": self.__region_id,
                "cloud_infra_type": "FUSION_CLOUD",
                "availability_zone_id": "",
                "indicators": [
                    {
                        "name": "op_backup_capacity",
                        "display_name":
                            '{"zh_cn": "备份空间",'
                            '"en_us": "Backup Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 140,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    },
                    {
                        "name": "op_copy_capacity",
                        "display_name":
                            '{"zh_cn": "复制空间",'
                            '"en_us": "Replication Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 141,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    }
                ]
            }
        }

    def __metrics_info(self):
        self.metric_info = self._get_csbs_metric_info()

    def _get_csbs_metric_info(self):
        return {
            "metrics": [
                {"service_id": self.__csbs_id,
                 "name": "op_backup_capacity",
                 "display_name": "{\"zh_CN\":\"备份容量\",\"en_US\":\"Backup Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__app_backup_id}.op_backup",
                 "resource_type_name": "{\"zh_CN\":\"云服务备份容量\",\"en_US\":\"CSBS Backup Capacity\"}",
                 "cdr_folder_name": self.__app_backup_id
                 },
                {"service_id": self.__csbs_id,
                 "name": "op_copy_capacity",
                 "display_name": "{\"zh_CN\":\"复制容量\",\"en_US\":\"Duplication Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__app_backup_id}.op_copy",
                 "resource_type_name": "{\"zh_CN\":\"云服务复制容量\",\"en_US\":\"CSBS Duplication Capacity\"}",
                 "cdr_folder_name": self.__app_backup_id
                 }
            ]
        }

    def get_query_expenses_notification_info(self, service_id):
        return {
            "notifications": [{
                "service_id": service_id,
                "region_id": self.__region_id,
                "event_type": "query_expenses",
                "global_level": False,
                "step": "before",
                "interface_type": "sync",
                "request_method": "POST",
                "request_url": "/v1/query_expenses",
                "request_template":
                    "{\"subscriptions\":[<#list subscriptions as sub>{\"operate_type\":\"${sub.operateType}\","
                    "\"project_id\":\"${sub.projectId}\",\"service_type\":\"${sub.serviceType}\","
                    "\"product_id\":\"${sub.productId}\",\"region_id\":\"${sub.regionId}\","
                    "\"params\":\"${sub.params?j_string}\",\"secret_params\":\"${sub.secretParams?j_string}\","
                    "\"tenancy\":\"${sub.tenancy}\",\"time_zone\":\"${sub.timeZone}\","
                    "\"old_subscription_id\":\"${sub.oldSubscriptionId}\","
                    "\"resource_info\":{\"ids\":[<#if sub.resourceInfo?? && sub.resourceInfo.ids??>"
                    "<#list sub.resourceInfo.ids as ids>{\"id\": \"${ids.id}\","
                    "\"service_type\": \"${ids.serviceType}\",\"name\": \"${ids.name}\"}<#if ids_has_next>,"
                    "</#if></#list></#if>]}}<#if sub_has_next>,</#if></#list>]}",
                "response_template":
                    "{<#if !Header.HttpCode?starts_with(\"2\")>\"retCode\":\"${Header.HttpCode}\","
                    "\"data\":{ }<#else>\"retCode\":\"200\",\"data\":{\"rate_params\":"
                    "[<#list Body.rate_params as rateParam>\"${rateParam?j_string}\"<#if rateParam_has_next>,"
                    "</#if></#list>]}</#if>}"
            }]
        }
