import time

import utils.common.log as logger
from utils.common.code2Msg import code2Msg

from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.dpa_login_check import DPALoginCheck
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.karborproxy.ssh_cmd import SShCmd
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation


class ProxyParamsCheck(object):
    def __init__(self, project_id, pod_id=None):
        if not pod_id:
            pod_id = project_id
        self.project_id = project_id
        self.ssh_client = SshClient()
        self.proxy_operation = KarborProxyOperation(self.project_id)
        self.proxy_node_list = self.proxy_operation.get_proxy_node_list()
        self.ssh = SShCmd(project_id, pod_id)

    def check_proxy_login_ability(self, check_result):
        check_result = self.check_proxy_dpamanager_login_ability(check_result)

        check_result = self.check_proxy_root_login_ability(check_result)
        return check_result

    def check_proxy_dpamanager_login_ability(self, check_result):
        dpamanager_pwd_will_expired = []
        dpamanager_pwd_no_ssh_ips = []

        check_list = ["csbs_karborproxy_dpamanager_password"]

        def _get_dpamanager_login_check_result():
            success_count = 0
            for node in self.proxy_node_list:
                ssh_client = None
                pwd_expired_flag = None
                try:
                    ssh_client = self.ssh_client.create_ssh_client(node.node_ip, node.user, node.user_pwd)
                except Exception as err:
                    logger.error(f"Failed to login to Proxy node: IP {node.node_ip}, User dpamanager, error:{str(err)}")
                    dpamanager_pwd_no_ssh_ips.append(node.node_ip)
                finally:
                    if ssh_client:
                        pwd_expired_flag = check_pwd_will_expired(self.ssh_client, ssh_client, node.user)
                        self.ssh_client.ssh_close(ssh_client)
                if pwd_expired_flag:
                    logger.error(f"Node {node.node_ip}, User {node.user} password will expire, "
                                 f"please change and retry.")
                    dpamanager_pwd_will_expired.append(node.node_ip)
                else:
                    success_count += 1
            return success_count

        check_success_count = _get_dpamanager_login_check_result()
        if check_success_count == len(self.proxy_node_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            if dpamanager_pwd_will_expired:
                logger.error(f"Node IP {','.join(dpamanager_pwd_will_expired)}, "
                             "User dpamanager password will expire.")
                error_msg = code2Msg(645056)
            else:
                logger.error(f"Failed to login to Proxy node: IP {','.join(dpamanager_pwd_no_ssh_ips)}, "
                             "User dpamanager.")
                error_msg = code2Msg(642001)
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=error_msg)
        return check_result

    def check_proxy_root_login_ability(self, check_result):
        check_list = ["csbs_karborproxy_root_password"]
        check_success_count = 0
        proxy_root_no_ssh_ips = []
        for proxy_node in self.proxy_node_list:
            ssh_client = None
            try:
                ssh_client = self.ssh_client.get_ssh_client_user_su_root(proxy_node)
                check_success_count += 1
            except Exception as err:
                logger.error(
                    "Failed to login to Proxy node: IP {}, User {}, "
                    "error: {}".format(proxy_node.node_ip, 'root', str(err))
                )
                proxy_root_no_ssh_ips.append(proxy_node.node_ip)
            finally:
                if ssh_client:
                    self.ssh_client.ssh_close(ssh_client)

        if check_success_count == len(self.proxy_node_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            logger.error(
                "Failed to login to Proxy node: IP {}, User {}".format(
                    ','.join(proxy_root_no_ssh_ips), "root")
            )
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(642001))
        return check_result

    def check_dpa_login_ability(self, check_result):
        check_list = ['DPA_Business_Address', 'DPA_Cluster_Address', 'DPA_Admin', 'DPA_Admin_Password', 'DPA_Sec_Admin',
                      'DPA_Sec_Admin_Password', 'DPA_General_User', 'DPA_General_User_Password']
        try:
            dpa_login_check = DPALoginCheck(self.project_id)
            dpa_login_check.check_account_info()
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        except Exception as exception:
            logger.error(f"Failed to login to dpa, exception: {str(exception)}")
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(645060))
        return check_result

    def check_proxy_expired(self):
        for node in self.proxy_operation.get_proxy_node_list():
            self.ssh.check_node_password_expired(node)

    def check_proxy_status(self):
        self.check_omm_ha()
        self.check_haproxy()
        self.check_cps_monitor()
        self.check_agentproxy()
        self.check_zookeeper()
        self.check_v_zookeeper()

    def check_omm_ha(self):
        self.config_ha_retry()

    def check_haproxy(self):
        logger.info('haproxy status check started.')
        haproxy_status_cmd = 'haproxyControl -CS CompontState;' \
                             'echo haproxy status check result $?'
        self.ssh.ssh_cmds_to_all_proxys(haproxy_status_cmd)

    def check_cps_monitor(self):
        logger.info('cps_monitor status check started.')

        cps_monitor_status_cmd = 'cps-monitorControl -CS CompontState;' \
                                 'echo cps-monitor status check result $?'
        self.ssh.ssh_cmds_to_all_proxys(cps_monitor_status_cmd)

    def check_agentproxy(self):
        logger.info('agentproxy status check started.')
        agent_proxy_status_cmd = 'agentproxyControl -CS CompontState;' \
                                 'echo agentproxy status check result $?'
        self.ssh.ssh_cmds_to_all_proxys(agent_proxy_status_cmd)

    def check_v_zookeeper(self):
        """
            # 0 单节点
            # 2 主
            # 3 备
            # 1 失败
        :return:
        """
        zook_success_status = ['0', '2', '3']
        logger.info('v_zookeeper status check started.')
        query_zookeeper_status_cmd = 'v_zookeeperControl  -CS CompontState;' \
                                     'echo V_ZooKeeper status check result $?'

        node = self.proxy_operation.get_proxy_node_list()[0]
        self.ssh.ssh_cmds_to_proxy(
            node,
            query_zookeeper_status_cmd,
            zook_success_status
        )

    def check_zookeeper(self):
        """
            # 0 单节点
            # 2 主
            # 3 备
            # 1 失败
        :return:
        """
        zook_success_status = ['0', '2', '3']
        logger.info('zookeeper status check started.')
        query_zookeeper_status_cmd = 'zookeeperControl  -CS CompontState;' \
                                     'echo ZooKeeper status check result $?'

        self.ssh.ssh_cmds_to_all_proxys(query_zookeeper_status_cmd, zook_success_status)

    def config_ha_retry(self):
        retry_count = 0
        while retry_count < 10:
            try:
                self.check_omm_ha_status()
                break
            except Exception as err:
                logger.error("ssh connect failed.%s" % err)
                retry_count = retry_count + 1
                if retry_count == 10:
                    raise Exception('Failed to configure the OMM HA.') from err
                time.sleep(60)

    def check_omm_ha_status(self):
        query_ha_status_cmd = 'omm-haControl -CS CompontState;' \
                              ' echo omm_ha status $?'
        for obj in self.proxy_operation.get_proxy_node_list():
            ha_success_status = ['0', '2', '3']
            self.ssh.ssh_cmds(
                obj, query_ha_status_cmd, ha_success_status)
