from collections import namedtuple

import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common import param_check

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.constant import COMPONENT
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import check_string_param


class ManageDataBackup(node_installer.Installer):
    def __init__(self, project_id, pod_id):
        super(ManageDataBackup, self).__init__()
        self.pod_id = pod_id
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.karbor_node_list = self.karbor_util.get_karbor_node_list()
        logger.info(f'Ssh to karbor node:{self.karbor_node_list[0].node_ip}.')
        self.ssh_client = SshClient()
        self.karbor_client = self.ssh_client.get_ssh_client(self.karbor_node_list[0])
        # default value
        self.backups_retain = 30
        # default value
        self.log_retain = 12
        self.param_util = ParamUtil()

    def install(self):
        return self._config_ftp()

    def check(self):
        return True

    def rollback(self):
        logger.info('Executing rollback')
        cmds = "source /opt/huawei/dj/inst/utils.sh;" \
               "set_backup_server_policy --enable false"
        logger.info('Cmd:{}'.format(cmds))
        result = self.ssh_client.ssh_exec_command_return(self.karbor_client, cmds)
        self.ssh_client.ssh_close(self.karbor_client)
        return self.ssh_client.is_ssh_cmd_executed(result)

    def get_manage_data_backup(self, key_name):
        manage_data_backup_info = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, key_name)
        return manage_data_backup_info

    def _get_ftp_info(self):
        ftp_ip = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "managedatabackup_ftp_ip")
        ftp_port = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "managedatabackup_ftp_port")
        ftp_username = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "managedatabackup_ftp_username")
        ftp_passwd = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "managedatabackup_ftp_password")
        ftp_path = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "managedatabackup_path")
        logger.info(f'Get sftp info:[sftp_ip:{ftp_ip}, sftp_port:{ftp_port}, sftp_username:{ftp_username}, '
                    f'sftp_path:{ftp_path}.]')

        if not all([ftp_ip, ftp_port, ftp_username, ftp_passwd, ftp_path]):
            logger.error('Failed to get all sftp server info.')
            raise Exception('Failed to get all sftp server info.')
        ftp_server = namedtuple('ftp_server', ['node_ip', 'port', 'username', 'passwd', 'backup_path'])
        return ftp_server(node_ip=ftp_ip, port=ftp_port, username=ftp_username, passwd=ftp_passwd, backup_path=ftp_path)

    def _params_check(self, ftp_info):
        ip_version = self.param_tool.get_ip_version()
        if not param_check.check_param_ip(ftp_info.node_ip, ip_version):
            raise Exception(f"Invalid ip address, the parma is {ftp_info.node_ip}.")
        if not check_string_param(ftp_info.port, max_len=5, expr=r"^[0-9]+$"):
            raise Exception(f"Invalid address port , the parma is {ftp_info.port}.")
        if not check_string_param(ftp_info.username):
            raise Exception(f"Invalid param, the parma is {ftp_info.username}.")
        if not check_string_param(ftp_info.backup_path, expr="^[a-zA-Z0-9/.\-_]+$"):
            raise Exception(f"Invalid param, the parma is {ftp_info.backup_path}.")

    def _config_ftp(self):
        ftp_info = self._get_ftp_info()
        self._params_check(ftp_info)

        expect_str = "Input the username of FTP:"
        cmds = "source /opt/huawei/dj/inst/utils.sh;set_backup_server_policy --enable true --schema ftp " \
               "--ftp_ip {} --ftp_port {} --ftp_schema {}  --ftp_path {} --log_retain {} --backups_retain {}".format(
                ftp_info.node_ip, ftp_info.port, "sftp", ftp_info.backup_path, self.log_retain, self.backups_retain)
        logger.info('Cmd:{}'.format(cmds))
        result = self.ssh_client.ssh_send_command_expect(self.karbor_client, cmds, expect_str)
        if self.ssh_client.failed_to_return(result, "Input the username of FTP: ", self.karbor_client):
            logger.error('Failed executing:{}, expect_str:{}, results:{}'.format(cmds, expect_str, result))
            return False

        expect_str = "Input the password of FTP:"
        result = self.ssh_client.ssh_send_command_expect(self.karbor_client, ftp_info.username, expect_str)
        if self.ssh_client.failed_to_return(result, "Input the password of FTP: ", self.karbor_client):
            logger.error('Failed, expect_str:{}, results:{}'.format(expect_str, result))
            return False
        expect_str = "Confirm password:"
        result = self.ssh_client.ssh_send_command_expect(self.karbor_client, ftp_info.passwd, expect_str)
        if self.ssh_client.failed_to_return(result, "Confirm password: ", self.karbor_client):
            logger.error('Failed, expect_str:{}, results:{}'.format(expect_str, result))
            return False
        self.ssh_client.ssh_send_command_expect(self.karbor_client, ftp_info.passwd)
        self.ssh_client.ssh_close(self.karbor_client)
        return True
