import utils.common.log as logger
from utils.business.arb_util import ArbApi
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karbor.cbs_installer import cbs_karborarbitercfg

logger.init("KarborAZ")


class KarborArbitration(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(KarborArbitration, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.djmanager_dir = "/home/djmanager/"
        self.openstack_dir = "/home/openstack/"
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._upload_arbitration_to_karbor(pod_id)
            self._connect_to_unified_quorum_server(project_id, pod_id)
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, HCCIException("640066"))
        return Message(200)

    def _upload_arbitration_to_karbor(self, pod_id):
        karbor_nodes = self.karbor_util.get_karbor_node_list()
        if not karbor_nodes:
            logger.error('Failed to get available karbor nodes, so can not sync the arb certs.')
            return
        logger.info('Start uploading arbitration certificates')
        arbapi = ArbApi()
        logger.info('Karbor ip list :{}'.format([node.node_ip for node in karbor_nodes]))
        karbor_node = karbor_nodes[0]
        logger.info(f'Upload arbitration ca.crt from Arbitration_DC1_01_API to karbor :{karbor_node.node_ip}, '
                    f'dir:{self.djmanager_dir}')
        arbapi.get_arb_cers(pod_id, karbor_node.node_ip, karbor_node.user, karbor_node.user_pwd, self.djmanager_dir)
        logger.info('Succeed uploading arbitration certificates')
        logger.info('Ssh to karbor node:{}'.format(karbor_node.node_ip))
        cmds = ["source /opt/huawei/dj/inst/utils.sh",
                f"\cp -f {self.djmanager_dir}ca.crt {self.openstack_dir}ca.crt",
                f"chown -h openstack:openstack {self.openstack_dir}*.crt",
                f"sync_karbor_certs --arb {self.openstack_dir}ca.crt"]
        karbor_client = None
        try:
            karbor_client = self.ssh_client.get_ssh_client(karbor_node)
            for cmd in cmds:
                logger.info(f'Cmd: {cmd}.')
                result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    logger.error(f"Failed to execute: {cmd}, returned message: {result[0]}.")
                    raise HCCIException("640066")
        except Exception as e:
            raise e
        finally:
            if karbor_client:
                self.ssh_client.ssh_close(karbor_client)

    @staticmethod
    def _connect_to_unified_quorum_server(project_id, pod_id):
        logger.info('Start connecting to Unified Quorum Server')
        configure = cbs_karborarbitercfg.KarborArbiterCfg(project_id, pod_id)
        if not configure.install():
            raise Exception("Connecting to Unified Quorum Server fail")
        logger.info('Succeed connecting to Unified Quorum Server')
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            if not cbs_karborarbitercfg.KarborArbiterCfg(project_id, pod_id).rollback():
                return Message(500, HCCIException("640049"))
        except Exception as err:
            logger.error("Execute error:%s" % err)
            return Message(500, HCCIException("640049"))
        return Message(200)
