# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
import utils.common.software_package_util as file_util
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common.code2Msg import code2Msg
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.business.manageone_util2 import ManageOneUtil2

from plugins.CSBS.common.error_code import UPLOAD_ADAPTIONPACKAGE_ERROR
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import auto_retry

logger.init("CSBS-VBS")
KARBOR_ADAPTATION_PKG_REG = f"^resource_10UnifiedAccess2MO_CSBS-OceanProtect_Karbor_(.*)\\.tar\\.gz$"
KARBOR_PROXY_ADAPTATION_PKG_REG = f"^resource_10UnifiedAccess2MO_CSBS-OceanProtect_KarborProxy_(.*)\\.tar\\.gz$"


class RegisterAdaptationPackage(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterAdaptationPackage, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_tool = ParamTool(project_id, pod_id)
        # 根据当前场景获取对应的十统一适配包
        # 1. 当前工程中不存在安装KarborProxy resource_10UnifiedAccess2MO_CSBS-VBS_Karbor*
        # 2. 当前工程中存在安装KarborProxy resource_10UnifiedAccess2MO_CSBS-VBS_KarborProxy*
        self.adaptation_pkg_reg = KARBOR_ADAPTATION_PKG_REG
        if get_project_condition_boolean(self.project_id, "InstallCSBS-VBS-Proxy"):
            self.adaptation_pkg_reg = KARBOR_PROXY_ADAPTATION_PKG_REG

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to register CSBS-VBS adaptation package.")
        try:
            path, name, version = file_util.find_software_package_by_name_re(self.adaptation_pkg_reg)
        except Exception as err:
            err_msg = f"Find adaptation_pkg_re:{self.adaptation_pkg_reg} failed. Reason is:{err}"
            logger.error(err_msg)
            return Message(500, err_msg)
        logger.info(f"Find adaptation_pkg[{path}] success, name[{name}, version is[{version}].")

        file_path = os.path.join(path, name)
        region_id = self.param_tool.get_region_id()
        result = self.register_union_adaptation_pkg(region_id, file_path)
        if not result:
            msg_cn, msg_en, _, _ = code2Msg(UPLOAD_ADAPTIONPACKAGE_ERROR)
            return Message(500, msg_cn, msg_en)
        logger.info("Succeeded to register CSBS-VBS adaptation package.")
        return Message(200)

    @auto_retry(max_retry_times=5, delay_time=60)
    def register_union_adaptation_pkg(self, region_id, file_path):
        package_infos = {
            "regionId": region_id,
            "file_path": file_path
        }
        logger.info(f"Start to register adaptation package, package_infos: {package_infos}.")
        result = ManageOneUtil2().upload_adaptation_package(self.project_id, self.pod_id, package_infos)
        if result.get("status") == "failed":
            message = result.get("msg")
            err_msg_en = f"Failed use manage service api to register adaptation package, " \
                         f"file path : {file_path}, err_msg:{message}."
            logger.error(err_msg_en)
            raise HCCIException("640133", (err_msg_en, err_msg_en))
        logger.info("Succeeded to register adaptation package.")
        return True
