# -- coding:utf-8 --
import json

from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.vm_util import PublicServiceOM
from utils.common.exception import HCCIException
from utils.common.message import Message
import utils.common.log as logger

from plugins.CSBS.common.karbor_cpshelper import CPSInfo, CpsHelper
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import UpgradeBaseSubJob

logger.init("Delete workflow info")
EBACKUP_COMPONENT_NAME = 'eBackup'
DELETE_ACCOUNT_OP = 2
SERVICE_ACCOUNT_TYPE = 1
GUI_ACCOUNT_TYPE = 6


class DeleteWorkflow(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.workflow_management_ip = None
        self.workflow_internal_ip = None
        self.cps_client = None
        self.workflow_ip = None
        self.manager_ip = None
        self.ssh_client = SshClient()
        self.network_name = "external_om"

    def execute(self, project_id, pod_id, regionid_list=None):
        cps_node = CPSInfo(self.pod_id).get_cps_node()
        try:
            self.cps_client = CpsHelper(cps_node).get_cps_client(self.ssh_client)
            self.get_workflow_info()
            self.delete_workflow_unipwd()
            self.delete_workflow_nodes()
            self.delete_workflow_vms()
            self.delete_workflow_image()
            self.delete_workflow_service()
        except Exception as e:
            logger.error(f"Execute failed! {e}")
            return Message(500, HCCIException(645068, str(e)))
        finally:
            if self.cps_client:
                self.ssh_client.ssh_close(self.cps_client)
        return Message(200)

    def get_workflow_info(self):
        logger.info("Start to get eBackup-Manager node info.")
        self.manager_ip = self._get_vm_ip('eBackup-Manager')
        logger.info("Start to get eBackup-Workflow node info.")
        self.workflow_ip = self._get_vm_ip('eBackup-Workflow')
        cmd = "neutron port-list | grep Workflow-Float --color=never | sed 's/ //g'"
        network_details = self.ssh_client.ssh_exec_command_return(self.cps_client, cmd)
        logger.info('Find data list:{}'.format(network_details))
        for detail in network_details:
            if "last CMD_RESULT" in detail or "|" not in detail:
                continue
            items = detail.split('|')
            if 'Internal-Workflow-Float' in detail:
                self.workflow_internal_ip = json.loads(items[4]).get('ip_address')
                logger.info(f"Get float ip address info is: {self.workflow_internal_ip}")
            if 'Management-Workflow-Float' in detail:
                self.workflow_management_ip = json.loads(items[4]).get('ip_address')
                logger.info(f"Get float ip address info is: {self.workflow_management_ip}")

    def _get_vm_ip(self, server_name):
        vm_infos = PublicServiceOM.get_vm_basic_detail_info(self.pod_id, server_name)
        if not vm_infos:
            return ''
        manager_vm = vm_infos[0]
        ip = manager_vm.get('addresses').get('external_om')[0].get('addr')
        logger.info(f"Get node info success. ip address: {ip}")
        return ip

    def delete_workflow_unipwd(self):
        """
        删除workflow统一密码注册信息
        """
        logger.info("Start to delete unipwd info.")
        account_list = []
        if self.manager_ip:
            account_list = account_list + self._build_data(self.manager_ip)
        if self.workflow_management_ip:
            account_list.append({
                "accountName": 'admin',
                "region": self.region_id,
                "ip": self.workflow_management_ip,
                "accountType": GUI_ACCOUNT_TYPE,
                "usedScene": self.workflow_management_ip,
                "operationType": DELETE_ACCOUNT_OP
            })
        if self.workflow_ip:
            account_list = account_list + self._build_data(self.workflow_ip)
        if not account_list:
            logger.warning("Doesn't exists unipwd info.")
            return

        data = {
            "componentName": EBACKUP_COMPONENT_NAME,
            "subComponents": [{"subComponentName": 'eBackupManager',
                               "createdAccountList": account_list}]
        }
        ManageOneUtil2().uniform_password_util(self.project_id, self.pod_id, data)
        logger.info("Delete unipwd info success.")

    def _build_data(self, ip):
        return [
            {
                "accountName": 'hcp',
                "region": self.region_id,
                "ip": ip,
                "accountType": SERVICE_ACCOUNT_TYPE,
                "usedScene": ip,
                "operationType": DELETE_ACCOUNT_OP
            },
            {
                "accountName": "root",
                "region": self.region_id,
                "ip": ip,
                "accountType": SERVICE_ACCOUNT_TYPE,
                "usedScene": ip,
                "operationType": DELETE_ACCOUNT_OP
            }
        ]

    def delete_workflow_service(self):
        """
        删除workflow云服务
        """
        cmdb_util = ManageOneCmdbUtil(self.project_id, self.pod_id)
        cmdb_info = cmdb_util.get_cloud_service_info(self.regionid_list[0], index_name=EBACKUP_COMPONENT_NAME)
        if not cmdb_info:
            logger.warning("Can't find workflow service info, delete skip.")
            return
        for service in cmdb_info:
            if 'workflow_service' in service.get('name'):
                logger.info(f"Find workflow service info {service}.")
                cmdb_util.remove_cloud_service_info(self.regionid_list[0], index_name=service.get('indexName'),
                                                    version=service.get('version'), name=service.get('name'))
                logger.info("Delete workflow service success.")

    def delete_workflow_vms(self):
        """
        删除workflow虚拟机
        """
        if self.manager_ip:
            logger.info("Start to delete eBackup-Manager vm and port.")
            PublicServiceOM.delete_vm_by_name('eBackup-Manager', self.pod_id)
            PublicServiceOM.delete_port_by_name(self.pod_id, 'eBackup-Manager-port0', self.manager_ip,
                                                self.network_name)
        if self.workflow_ip:
            logger.info("Start to delete eBackup-Workflow vm and port.")
            PublicServiceOM.delete_vm_by_name('eBackup-Workflow', self.pod_id)
            PublicServiceOM.delete_port_by_name(self.pod_id, 'eBackup-Workflow-port0', self.workflow_ip,
                                                self.network_name)
        if self.workflow_management_ip:
            logger.info("Start to delete Management-Workflow-Float port.")
            PublicServiceOM.delete_port_by_name(self.pod_id, 'Management-Workflow-Float', self.workflow_management_ip,
                                                self.network_name)
        if self.workflow_internal_ip:
            logger.info("Start to delete Internal-Workflow-Float port.")
            PublicServiceOM.delete_port_by_name(self.pod_id, 'Internal-Workflow-Float', self.workflow_internal_ip,
                                                self.network_name)

    def delete_workflow_nodes(self):
        """
        删除workflow部署节点
        """
        logger.info("Start to delete deploy nodes.")
        cmdb_util = ManageOneCmdbUtil(self.project_id, self.pod_id)
        cmdb_util.remove_deploy_node_info(self.region_id, 'eBackup-Workflow')
        cmdb_util.remove_deploy_node_info(self.region_id, 'eBackup-Manager')
        logger.info("Delete deploy nodes success.")

    def delete_workflow_image(self):
        """
        删除Workflow glance 镜像文件
        """
        logger.info("Start to delete glance image.")
        PublicServiceOM.delete_image(self.pod_id, "ebackup-image")
        logger.info("Delete glance image successfully.")
