import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.upgrade.dmk_client import DMKClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation

logger.init("CSBS-VBS")
CHECK_TIMES_LIMIT = 20
SLEEP_TIME = 60


class StatusCheck(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        try:
            super(StatusCheck, self).__init__(project_id, pod_id, regionid_list)
            self.karbor_operation = KarborOperation(self.project_id)
            self.dmk_client = DMKClient(self.params_store)
        except Exception as err:
            logger.error(f"Failed to initialize the StatusCheck object, err_msg:{str(err)}.")
            raise HCCIException(645076)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Check karbor service status after upgrade.

        """
        try:
            self.karbor_operation.change_operation_service(tar_status="start")
            self._check_karbor_service()
        except Exception as err:
            logger.error(f"Failed to check Karbor service, err_msg:{str(err)}.")
            return Message(500, HCCIException(645023))
        # reset max sync time interval for gaussdb to be 10 minutes
        self.karbor_operation.set_max_sync_time_interval_allowed(10)
        return Message(200)

    def _check_karbor_service(self):
        logger.info("Login dmk and call DMK to check Karbor.")
        check_times = 0
        while True:
            check_times += 1
            if check_times > CHECK_TIMES_LIMIT:
                err_msg = f"Karbor service status is abnormal, check {check_times} times."
                logger.error(err_msg)
                raise Exception(err_msg)
            action = "[Check] CSBS-VBS Karbor"
            result = None
            try:
                result = self.dmk_client.dmk_deploy_karbor(action, self.upgrade_path.karbor_pkg_version)
            except Exception as err:
                logger.warn(f"Failed to check the karbor service using DMK, err_msg:{str(err)}.")
            if result:
                logger.info("Check Karbor service successfully.")
                break
            logger.info(f"Sleep {SLEEP_TIME} seconds and retry to check karbor service, retry times:{check_times}.")
            time.sleep(SLEEP_TIME)
