import os

import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.business.project_util import ProjectApi
from utils.common.message import Message
from utils.constant.path_constant import SOURCES_ROOT_DIR

from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import ConfigIniFile
from plugins.CSBS.common.upgrade.params import ParamsStore
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation
from plugins.CSBS.common.upgrade.constant import ENV_CONFIG_PATH

logger.init("KarborProxy")


class CheckKarborProxyVersion(UpgradeBaseSubJob):

    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.karbor_proxy = KarborProxyOperation(self.project_id)
        self.params_store = ParamsStore(self.project_id, self.pod_id, self.regionid_list)
        self.project_api = ProjectApi()
        self.ssh_client = SshClient()
        self.karborproxy_full_version = self.karbor_proxy.get_karborproxy_version()
        self.karborproxy_plugin_version = self.karbor_proxy.get_karborproxy_upgrade_path_version()
        self.original_version = self.karborproxy_plugin_version.get("original_version", "")
        self.target_version = self.karborproxy_plugin_version.get("target_version", "")

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Init version information data.")
        self._init_karborproxy_version_param()
        logger.info("Succeeded to init version information data.")

        item_dict = self.params_store.get_check_config("ProxyCheckVersion")
        check_results = []
        if self.original_version == self.karborproxy_full_version:
            result = CheckResult(itemname_ch=item_dict["itemname_ch"],
                                 itemname_en=item_dict["itemname_en"],
                                 status=item_dict["success_status"])
            check_results.append(result)
        else:
            result = CheckResult(
                itemname_ch=item_dict["itemname_ch"],
                itemname_en=item_dict["itemname_en"],
                status=item_dict["failure_status"],
                error_msg_cn=item_dict["error_msg_cn"].format(self.original_version,
                                                              str(self.karborproxy_full_version)),
                error_msg_en=item_dict["error_msg_en"].format(self.original_version,
                                                              str(self.karborproxy_full_version)),
                suggestion_cn=item_dict["suggestion_cn"],
                suggestion_en=item_dict["suggestion_en"])
            check_results.append(result)
        logger.info("Check KarborProxy version succeed.")
        # save pre-upgrade and post-upgrade versions
        version_info = {
            "Originalversion": self.original_version,
            "Targetversion": self.target_version
        }
        self.project_api.save_version_info(self.project_id, "CSBS-VBS", version_info)
        return Message(200, check_results=check_results)

    def _init_karborproxy_version_param(self):
        karborproxy_pkg_version = self.karbor_proxy.get_karborproxy_pkg_version()

        env_config_path = os.path.join(SOURCES_ROOT_DIR, ENV_CONFIG_PATH)
        conf_env_file = ConfigIniFile(env_config_path)
        ret = conf_env_file.set_value_by_key_and_sub_key("KarborProxy", "cur_version", self.karborproxy_full_version)
        if not ret:
            raise Exception("Failed to save the karbor current version.")
        ret = conf_env_file.set_value_by_key_and_sub_key("KarborProxy", "tar_version", karborproxy_pkg_version)
        if not ret:
            raise Exception("Failed to save the karbor target version.")

# the code has been updated to python3.7
