import json
import os
import zipfile
from collections import namedtuple

import utils.common.log as logger
import utils.common.software_package_util as file_util
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.dmk_util import DmkApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.constant.path_constant import SOURCES_ROOT_DIR

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common import constant as common_constant
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.upgrade import models
from plugins.CSBS.common.upgrade.constant import ENV_CONFIG_PATH
from plugins.CSBS.common.util import ConfigIniFile

logger.init(constant.CSBS_SERVICE_NAME)


class ParamsStore(object):
    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.param_util = ParamUtil()
        self.params_tools = ParamsTools(project_id)
        self.ssh_client = SshClient()
        # get all params
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.db_params_dict = self.get_params()

    def get_params(self):
        db_params_dict = {}
        service_items = \
            BaseOps().get_service_cloud_param(self.pod_id,
                                              constant.CSBS_SERVICE_NAME,
                                              self.region_id)
        for key, value in service_items.items():
            db_params_dict[key] = value
        dmk_items = DmkApi().get_dmk_info(self.project_id)
        for key, value in dmk_items.items():
            db_params_dict[key] = value
        dmk_business_keys = ["DMK_dmk_os_business_username",
                             "DMK_dmk_os_business_user_password"]
        for key in dmk_business_keys:
            val = self.param_util.get_param_value(self.pod_id, "DMK", key)
            if val:
                db_params_dict[key] = val

        db_params_dict.update(self.params_tools.get_karbor_vm_ips())
        return db_params_dict

    def get_karbor_params(self):
        first_node = self.karbor_util.get_karbor_node_list()[0]
        karbor_params = models.KarborParams(
            djmanager_account_name=first_node.user,
            djmanager_account_pwd=first_node.user_pwd,
            root_account_name=first_node.root_name,
            root_account_pwd=first_node.root_pwd,
        )
        return karbor_params

    def get_dmk_params(self):
        dmk_params = models.DmkParams(
            account=self.db_params_dict.get("dmk_account", ""),
            tmp_account_pwd=self.db_params_dict.get("tmp_dmk_account_pwd", ""),
            account_pwd=self.db_params_dict.get("dmk_account_pwd", ""),
            account_group=self.db_params_dict.get("dmk_account_group", ""),
            float_ip=self.db_params_dict.get("DMK_dmk_floatIp", ""),
            # Super Administrator Account of DMK Portal, example: sysadmin
            ui_username=self.db_params_dict.get("DMK_dmk_ui_username", ""),
            ui_password=self.db_params_dict.get("DMK_dmk_ui_password", ""),
            # DMK Service Account (dmk/udmk)
            os_business_username=self.db_params_dict.get("DMK_dmk_os_business_username", ""),
            os_business_password=self.db_params_dict.get("DMK_dmk_os_business_user_password", "")
        )
        return dmk_params

    def get_project_scene_params(self):
        project_scene = \
            models.ProjectScene(
                is_csha_scene=self.params_tools.is_csha_scene,
                is_x86_scene=self.params_tools.is_x86_scene,
                is_primary_region=self.params_tools.is_primary_region,
                is_global_dr_scene=self.params_tools.is_global_dr_scene,
                region_id=self.region_id)
        return project_scene

    def get_upgrade_path_params(self):
        ser_version_dict = self.get_upgrade_path_version()
        current_upgrade_path = self.params_tools.get_current_upgrade_path()

        pkg_suffix = common_constant.PKG_SUFFIX_X86 \
            if self.params_tools.is_x86_scene else common_constant.PKG_SUFFIX_ARM
        karbor_pkg_path, karbor_pkg_name = \
            self.params_tools.get_pkg_path_and_pkg_name(pkg_suffix)
        karbor_pkg_version = \
            self.params_tools.get_pkg_version(karbor_pkg_path, karbor_pkg_name)

        console_pkg_path, console_pkg_name = \
            self.params_tools.get_pkg_path_and_pkg_name("_CSBS-OceanProtect_Console.zip")
        console_pkg_version = \
            self.params_tools.get_pkg_version(console_pkg_path,
                                              console_pkg_name)
        karbor_version = self.get_karbor_version()
        upgrade_path = \
            models.UpgradePath(
                current_upgrade_path=current_upgrade_path,
                current_version=karbor_version.get("short_version"),
                karbor_full_version=karbor_version.get("full_version"),
                original_version=ser_version_dict.get("original_version", ""),
                target_version=ser_version_dict.get("target_version", ""),
                pkg_path=karbor_pkg_path,
                karbor_pkg_name=karbor_pkg_name,
                karbor_pkg_version=karbor_pkg_version,
                console_pkg_name=console_pkg_name,
                console_pkg_version=console_pkg_version)
        return upgrade_path

    def get_current_upgrade_path(self):
        return self.params_tools.get_current_upgrade_path()

    def get_karbor_version(self):
        return self.karbor_util.get_karbor_version()

    def get_karbor_float_ip(self):
        return self.karbor_util.get_karbor_float_ip()

    def get_upgrade_path_version(self):
        solution, version = \
            ProjectApi.get_solution_and_version(self.project_id)
        sol_original_version, sol_target_version = version.strip().split("-")
        plugin_file_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../../plugin.json")
        ser_version_dict = dict()
        if os.path.exists(plugin_file_path):
            with open(plugin_file_path, 'r') as file:
                json_str = file.read()
                plugin_dic = json.loads(json_str)
                ser_version_dict["original_version"] = \
                    plugin_dic['plugins'][0]['workflow'][2]['VersionMatching'][sol_original_version]
                ser_version_dict["target_version"] = \
                    plugin_dic['plugins'][0]['workflow'][2]['VersionMatching'][sol_target_version]
                return ser_version_dict
        else:
            raise Exception("The 'plugin.json' file does not exist.")

    @staticmethod
    def get_check_config(item):
        file_path = os.path.join(os.path.dirname(os.path.realpath(__file__)),
                                 "../../conf/upgrade/upgrade_pre_check.json")
        if os.path.exists(file_path):
            with open(file_path, 'r', encoding='UTF-8') as file:
                json_str = file.read()
                check_dic = json.loads(json_str)
                return check_dic[item]
        else:
            raise Exception(
                "The 'upgrade_pre_check.json' file does not exist.")


class ParamsTools(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.service_name = constant.CSBS_SERVICE_NAME
        self.project_api = ProjectApi()
        self.param_util = ParamUtil()
        self.ssh_client = SshClient()

    def get_karbor_vm_names(self):
        vm_names = [
            constant.VMS_NAME.get('KARBOR1'),
            constant.VMS_NAME.get('KARBOR2'),
            constant.VMS_NAME.get('KARBOR3')
        ]
        vm_names_ha = [
            constant.VMS_NAME.get('KARBOR_HA1'),
            constant.VMS_NAME.get('KARBOR_HA2')
        ]
        return vm_names_ha if self.is_csha_scene else vm_names

    def get_karbor_vm_ips(self):
        vm_names = self.get_karbor_vm_names()
        vm_name_keys = ['karbor_host0', 'karbor_host1', 'karbor_host2']
        vm_ips = []
        for vm_name in vm_names:
            vm_info = self.param_util.get_vm_info_by_vm_name(self.project_id,
                                                             vm_name)
            if not vm_info:
                raise Exception('Failed to getting the VM information based '
                                'on the VM name, vm name:{}.'.format(vm_name))
            vm_ips.append(vm_info[0].get('ip'))
        vm_ip_dict = dict(zip(vm_name_keys, vm_ips))
        return vm_ip_dict

    @property
    def is_csha_scene(self):
        project_info = self.project_api.get_project_info(self.project_id)
        base_options = project_info.get('base_options')
        if base_options.get('is_csha_scene'):
            return True
        return False

    @property
    def is_x86_scene(self):
        project_dict = self.project_api.get_project_conditions(self.project_id)
        return True if project_dict.get('is_x86_scene') else False

    @property
    def is_arm_scene(self):
        project_dict = self.project_api.get_project_conditions(self.project_id)
        return True if project_dict.get('is_arm_scene') else False

    @property
    def is_global_dr_scene(self):
        project_dict = self.project_api.get_project_conditions(self.project_id)
        is_global_dr = project_dict.get("GlobalConDR")
        if is_global_dr:
            return True
        return False

    @property
    def is_primary_region(self):
        project_dict = self.project_api.get_project_conditions(self.project_id)
        is_primary = project_dict.get("PrimaryRegion")
        if is_primary:
            return True
        return False

    @staticmethod
    def get_env_config_path():
        return os.path.join(SOURCES_ROOT_DIR, ENV_CONFIG_PATH)

    @staticmethod
    def get_pkg_version(pkg_path, pkg_name):
        pkg_path = os.path.join(pkg_path, pkg_name)
        file_fs = None
        try:
            file_fs = zipfile.ZipFile(r'{}'.format(pkg_path), 'r')
        except Exception as err:
            raise Exception(f"Get package version failed, package name: {pkg_name}, err_msg: {str(err)}.") from err
        finally:
            if file_fs:
                file_fs.close()
        for file in file_fs.namelist():
            if file.endswith(".json"):
                return file.split("-").__getitem__(1)
        raise Exception(f"Get package version failed, package name:{pkg_name}.")

    @staticmethod
    def get_pkg_path_and_pkg_name(pkg_suffix, pkg_pre_name=None):
        if not pkg_pre_name:
            pkg_pre_name = "OceanStor BCManager"
        pkg_path, pkg_name = \
            file_util.find_software_package_by_name(pkg_pre_name=pkg_pre_name,
                                                    pkg_post_name=pkg_suffix)
        if not pkg_name:
            raise Exception(
                "Can not find {}***{} pkg.".format(pkg_pre_name, pkg_suffix))
        return pkg_path, pkg_name

    def get_primary_region_id(self):
        return BaseOps().get_all_regionid_by_projectid(self.project_id)[0]

    def get_current_upgrade_path(self):
        solution, version = \
            ProjectApi.get_solution_and_version(self.project_id)
        return '_'.join([solution, version])

    def get_region_id(self):
        region_list = BaseOps().get_regionId_by_projectId(self.project_id)
        if not region_list:
            raise Exception("Failed to obtain the region id.")
        return region_list[0]

    def get_karbor_version_from_conf(self):
        env_config_path = self.get_env_config_path()
        conf_env_file = ConfigIniFile(env_config_path)
        karbor_params = conf_env_file.get_params_dict_by_key_name("karbor")
        original_version = karbor_params.get("original_version", "")
        target_version = karbor_params.get("target_version", "")
        if not (original_version or target_version):
            raise Exception("Failed to karbor version from env_param.ini file.")
        version_model = namedtuple("karbor_version", ["original_version", "target_version"])
        return version_model(original_version=original_version,
                             target_version=target_version)

    def get_cipher_type(self):
        env_config_path = self.get_env_config_path()
        conf_env_file = ConfigIniFile(env_config_path)
        karbor_params = conf_env_file.get_params_dict_by_key_name("karbor")
        cur_cipher_type = karbor_params.get("cur_cipher_type", "")
        tar_cipher_type = karbor_params.get("tar_cipher_type", "")
        if not (cur_cipher_type or tar_cipher_type):
            raise Exception("Failed to obtain cipher type.")
        return models.CipherType(cur_cipher_type=cur_cipher_type,
                                 tar_cipher_type=tar_cipher_type)

    def get_tar_cipher_type(self):
        project_conditions = self.project_api.get_project_conditions(
            self.project_id)
        if project_conditions.get("generalCipher"):
            return "generalCipher"
        elif project_conditions.get("SMCompatible"):
            return "SMCompatible"
        elif project_conditions.get("SMOnly"):
            return "SMOnly"
        else:
            raise Exception("Failed to obtain the target cipher type.")

    def get_oc_float_ip(self):
        """Get mo_om_float_ip for oc."""
        project_dict = self.project_api.get_project_conditions(self.project_id)
        if_ipv6 = project_dict.get("ManageIPV6", 0)
        try:
            om_info = ManageOneUtil2().get_mo_om_info(self.project_id)
        except Exception as err:
            logger.error(f"Failed to get oc float ip, err_msg:{err}.")
            raise Exception(f"Failed to get oc float ip, err_msg:{err}.") from err
        if not om_info:
            logger.error("Get oc float ip failed.")
            raise Exception("Failed to get oc float ip.")
        oc_float_ip = om_info[0]
        return f"[{oc_float_ip}]" if if_ipv6 else oc_float_ip

    def get_service_info_from_cmdb(self, region_id, service_name=None):
        service_name = service_name if service_name else self.service_name
        cmdb_service_name = f"{service_name}_{region_id}"
        logger.info(f"Start to obtain cloud service info from CMDB, "
                    f"cmdb service name:{cmdb_service_name}.")
        service_info_list = ManageOneCmdbUtil(self.project_id) \
            .get_cloud_service_info(region_code=region_id,
                                    index_name=service_name,
                                    name=cmdb_service_name)
        if not service_info_list:
            raise Exception("Failed to obtain cloud service info from CMDB.")
        logger.info(f"The cloud service information obtained "
                    f"from the CMDB is: {service_info_list}.")
        return service_info_list

    def get_cipher_type_from_cmdb(self, region_id):
        service_info_list = self.get_service_info_from_cmdb(region_id)
        for service_info in service_info_list:
            if not service_info.get("name") == f"{self.service_name}_{region_id}":
                continue
            for extend_info in service_info.get("extendInfos"):
                if extend_info.get("key") == "CipherType":
                    return extend_info.get("value")
        raise Exception("Failed to obtain the cipher_type from CMDB.")

    def get_global_domain(self, region_id):
        mo_public_params = self.param_util.get_service_cloud_param(
            self.project_id, "ManageOne_public_params", region_id)
        global_domain_name = mo_public_params.get(
            "ManageOne_global_domain_name")
        if not global_domain_name:
            global_domain_name = self.param_util.get_value_from_cloud_param(
                self.project_id, "ManageOne",
                "ManageOne_global_domain_name", region_id)
        if not global_domain_name:
            raise ValueError('Get global_domain_name in upgrade scene failed.')
        return global_domain_name

    def get_oc_url(self, region_id, is_oc_float_ip=False):
        global_domain_name = self.get_global_domain(region_id)
        if is_oc_float_ip:
            oc_float_ip = self.get_oc_float_ip()
            return f"https://{oc_float_ip}:26335"
        return f"https://oc.{region_id}.{global_domain_name}:26335"

    def get_iam_url(self, region_id):
        global_domain_name = self.get_global_domain(region_id)
        return f"https://iam-cache-proxy.{region_id}.{global_domain_name}:26335"

    def get_region_name_by_region_id(self, region_id):
        region_info_list = ManageOneCmdbUtil(self.project_id).get_region_info(region_id)
        region_name = None
        for region_info in region_info_list:
            if region_info.get("regionCode") == region_id:
                region_name = region_info.get("name")
                break
        if not region_name:
            raise Exception("Failed to obtain region name.")
        return region_name

    def get_all_region_code(self):
        region_info_list = ManageOneCmdbUtil(self.project_id).get_region_info()
        return [region_info.get('regionCode') for region_info in region_info_list if region_info.get('regionCode')]
