import time
from typing import List

import utils.common.log as logger
from utils.business.param_util import ParamUtil, HandleParamCheck
from utils.common.exception import HCCIException

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import to_str
from plugins.CSBS.common.model import Node
from plugins.CSBS.common.constant import KARBOR_USER_NAME, KARBOR_ROOT_NAME


class ParamsChecker(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.data_to_check = ParamUtil().get_need_check_cloud_params(
            self.project_id, constant.CSBS_SERVICE_NAME)
        self.ssh_client = SshClient()
        self.params_tools = ParamsTools(project_id)
        self.karbor_util = KarborUtil(project_id, project_id)

    def check_karbor_login_ability(self, check_result: HandleParamCheck):
        karbor_node_list = self.karbor_util.get_karbor_node_list()

        check_list = ["csbs_karbor_djmanager_password"]
        try:
            self.check_karbor_djmanager_login_ability(karbor_node_list)
        except HCCIException as e:
            check_result.set_check_result(check_list, status=500, error_msg=e)
        else:
            check_result.set_check_result(check_list, status=200)

        check_list = ["csbs_karbor_root_password"]
        try:
            self.check_karbor_root_login_ability(karbor_node_list)
        except HCCIException as e:
            check_result.set_check_result(check_list, status=500, error_msg=e)
        else:
            check_result.set_check_result(check_list, status=200)
        return check_result

    def check_karbor_djmanager_login_ability(self, node_list: List[Node]):
        djmanager_pwd_err = []
        djmanager_pwd_will_expired = []
        djmanager_pwd_no_ssh_ips = []

        self._check_passwd_unify(node_list, KARBOR_USER_NAME)

        def _get_djmanager_login_check_result():
            check_success_count = 0
            for node in node_list:
                ssh_client = None
                pwd_will_expired_flag = None
                error_flag = False
                try:
                    ssh_client = self.ssh_client.create_ssh_client(node.node_ip, node.user, node.user_pwd)
                    pwd_will_expired_flag = check_pwd_will_expired(self.ssh_client, ssh_client, node.user)
                except Exception as err:
                    logger.error(
                        f"Failed to login to Karbor node: IP {node.node_ip}, User {node.user}, error:{str(err)}")
                    error_flag = True
                    if str(err).lower().find("authentication") != -1:
                        djmanager_pwd_err.append(node.node_ip)
                    else:
                        djmanager_pwd_no_ssh_ips.append(node.node_ip)
                finally:
                    if ssh_client:
                        self.ssh_client.ssh_close(ssh_client)
                if error_flag:
                    logger.error(f"The {node.user} account password has expired or password error, please check.")
                elif pwd_will_expired_flag:
                    logger.error(
                        f"Node {node.node_ip}, User {node.user} password will expire， please change and retry.")
                    djmanager_pwd_will_expired.append(node.node_ip)
                else:
                    check_success_count += 1
            return check_success_count

        check_success_count = _get_djmanager_login_check_result()
        if check_success_count == len(node_list):
            return
        if djmanager_pwd_err:
            logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_err)}, "
                         f"User {constant.KARBOR_USER} passwd is expired or passwd error.")
            raise HCCIException(642002)
        if djmanager_pwd_will_expired:
            logger.error(f"Node IP {', '.join(djmanager_pwd_will_expired)}, "
                         f"User {constant.KARBOR_USER} password will expire.")
            raise HCCIException(642003, ', '.join(djmanager_pwd_will_expired))
        logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_no_ssh_ips)}, "
                     f"User {constant.KARBOR_USER}, please check ip.")
        raise HCCIException(642001)

    def check_karbor_root_login_ability(self, node_list: List[Node]):
        check_success_count = 0
        karbor_root_no_ssh_ips = []
        self._check_passwd_unify(node_list, KARBOR_ROOT_NAME)
        for node in node_list:
            ssh_client = None
            try:
                ssh_client = self.ssh_client.get_ssh_client_user_su_root(node)
            except Exception as err:
                logger.error(
                    "Failed to login to Karbor node: IP {}, User {}, "
                    "error: {}".format(node.node_ip, 'root', str(err))
                )
                karbor_root_no_ssh_ips.append(node.node_ip)
            finally:
                if ssh_client:
                    self.ssh_client.ssh_close(ssh_client)
            check_success_count += 1

        if check_success_count == len(node_list):
            return
        logger.error(
            "Failed to login to Karbor node: IP {}, User {}".format(
                ','.join(karbor_root_no_ssh_ips), "root")
        )
        raise HCCIException("642001")

    @staticmethod
    def _check_expires_str(ssh_client):
        expire_str = "has expired"
        ssh_client['channel'].send('\n')
        recv_str = ""
        num = 0
        while not (recv_str.endswith("$ ") or recv_str.endswith("# ")):
            num += 1
            time.sleep(1)
            if expire_str in recv_str or num > 10:
                logger.error("Get expired str, received shell prompt: "
                             "{}, in times:{}.".format(recv_str, num))
                return True
            if not ssh_client['channel'].recv_ready():
                continue
            recv_str = recv_str + to_str(ssh_client['channel'].recv(65535))
        logger.info("Test login success, no passwd expired.")
        return False

    @staticmethod
    def _check_passwd_unify(node_list: List[Node], account_name=KARBOR_USER_NAME):
        """
        由于当前无法满足完全去除SSH通道，所有节点密码必须要保持一致
        """
        list_size = len(node_list)
        for inx, node in enumerate(node_list):
            if inx > list_size - 2:
                return
            if account_name == KARBOR_USER_NAME and node.user_pwd != node_list[inx + 1].user_pwd:
                raise HCCIException(640136, account_name)
            if account_name == KARBOR_ROOT_NAME and node.root_pwd != node_list[inx + 1].root_pwd:
                raise HCCIException(640136, account_name)
