import os
import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.constant.path_constant import SOURCES_ROOT_DIR

from plugins.CSBS.common.util import compare_version
from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation
from plugins.CSBS.common.upgrade.constant import ENV_CONFIG_PATH
from plugins.CSBS.common.util import ConfigIniFile

logger.init("KarborProxy")


class CheckKarborProxyVersion(UpgradeBaseSubJob):

    def __init__(self, project_id, pod_id, regionid_list):
        super(CheckKarborProxyVersion, self).__init__(project_id, pod_id, regionid_list)
        self.karborproxy = KarborProxyOperation(project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        self._init_cipher_type_param()
        karborproxy_version = self.karborproxy.get_karborproxy_version()
        if not karborproxy_version:
            raise Exception("Failed to obtain the karborproxy version.")

        # 8.2.0之前版本不支持密码算法切换
        if compare_version("8.2.0", karborproxy_version):
            logger.error("karborproxy_version 8.2.0 or later supports cipher type switchover. "
                         "The current version is earlier and does not support cipher type switchover. "
                         f"The current version is {karborproxy_version}.")
            return Message(500, HCCIException(645034, karborproxy_version))

        logger.info(f"Succeed to check the karborproxy version, current version is {karborproxy_version}.")
        return Message(200, check_results=[CheckResult(status="success")])

    def _init_cipher_type_param(self):
        env_config_path = os.path.join(SOURCES_ROOT_DIR, ENV_CONFIG_PATH)
        conf_env_file = ConfigIniFile(env_config_path)
        # 切换算法前，从karborproxy获取的cipher_type为cur_cipher_type
        karborproxy_cipher_type = self.karborproxy.get_karborproxy_cur_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("KarborProxy", "cur_cipher_type",
                                                         karborproxy_cipher_type)
        if not ret:
            raise Exception("Failed to save the current cipher type.")

        tar_cipher_type = self.karborproxy.get_karborproxy_tar_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("KarborProxy", "tar_cipher_type",
                                                         tar_cipher_type)
        if not ret:
            raise Exception("Failed to save the target cipher type.")
